-- -*- mode: sql; sql-product: postgres; -*-

DROP TABLE IF EXISTS henry_notes   CASCADE;
DROP TABLE IF EXISTS henry         CASCADE;
DROP TABLE IF EXISTS species       CASCADE;
DROP TABLE IF EXISTS subcategories CASCADE;
DROP TABLE IF EXISTS categories    CASCADE;
DROP TABLE IF EXISTS notes         CASCADE;
DROP TABLE IF EXISTS literature    CASCADE;
CREATE TABLE notes (id SERIAL PRIMARY KEY,
    notelabel TEXT,
    noterefs  TEXT,
    notetext  TEXT);
CREATE TABLE literature (id SERIAL PRIMARY KEY,
    bibtexkey    TEXT,
    quotation    TEXT,
    bibliography TEXT,
    doi          TEXT);
CREATE TABLE categories (id SERIAL PRIMARY KEY,
    catname TEXT);
CREATE TABLE subcategories (id SERIAL PRIMARY KEY,
    subcatname TEXT,
    cat_id INT REFERENCES categories(id));
CREATE TABLE species (id SERIAL PRIMARY KEY,
    iupac    TEXT,
    formula  TEXT,
    trivial  TEXT,
    casrn    TEXT,
    inchikey TEXT,
    subcat_id INT REFERENCES subcategories(id));
CREATE TABLE henry (id SERIAL PRIMARY KEY,
    Hominus TEXT,
    mindHR  TEXT,
    htype   TEXT,
    species_id    INT REFERENCES species(id),
    literature_id INT REFERENCES literature(id));
CREATE TABLE henry_notes (id SERIAL,
    henry_id INT REFERENCES henry(id) ON UPDATE CASCADE,
    notes_id INT REFERENCES notes(id) ON UPDATE CASCADE,
    CONSTRAINT henry_notes_pkey PRIMARY KEY (henry_id, notes_id));

INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2291',
    'Abd-El-Bary et&nbsp;al. (1986)',
    'Abd-El-Bary, M. F., Hamoda, M. F., Tanisho, S., &#x0026; Wakao, N.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">constants for phenol over its diluted aqueous solution</span>, J. Chem. Eng. Data, 31, 229–230, <a href="https://doi.org/10.1021/JE00044A027" >doi:10.1021/JE00044A027</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3521',
    'Abney (2021)',
    'Abney, C. A.: Predicting Henry&#x2019;s Law constants of volatile organic compounds present in bourbon using molecular simulations, Master&#x2019;s thesis, University of Louisville, Kentucky, USA, <a href="https://doi.org/10.18297/etd/3440" >doi:10.18297/etd/3440</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3006',
    'Abou-Naccoul et&nbsp;al. (2014)',
    'Abou-Naccoul, R., Mokbel, I., Bassil, G., Saab, J., Stephan, K., &#x0026; Jose, J.: <span class="cmti-10">Aqueous solubility (in the range between 298.15 and 338.15 </span>K<span class="cmti-10">),</span> <span class="cmti-10">vapor pressures (in the range between </span>10<sup><span class="cmsy-7">−</span><span class="cmr-7">5</span></sup> <span class="cmti-10">and 80 Pa) and Henry&#x2019;s</span> <span class="cmti-10">law constant of 1,2,3,4-dibenzanthracene and 1,2,5,6-dibenzanthracene</span>, Chemosphere, 95, 41–49, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2013.08.010" >doi:10.1016/J.CHEMOSPHERE.2013.08.010</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2526',
    'Abraham et&nbsp;al. (2007)',
    'Abraham, M. A., Enomoto, K., Clarke, E. D., Rosés, M., Ràfols, C., &#x0026; Fuguet, E.: <span class="cmti-10">Henry&#x2019;s law constants or air to water partition coefficients</span> <span class="cmti-10">for 1,3,5-triazines by an LFER method</span>, J. Environ. Monit., 9, 234–239, <a href="https://doi.org/10.1039/B617181H" >doi:10.1039/B617181H</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2226',
    'Abraham (1979)',
    'Abraham, M. H.: <span class="cmti-10">Free energies of solution of rare</span> <span class="cmti-10">gases and alkanes in water and nonaqueous solvents. A quantitative</span> <span class="cmti-10">assessment of the hydrophobic effect</span>, J. Am. Chem. Soc., 101, 5477–5484, <a href="https://doi.org/10.1021/JA00513A004" >doi:10.1021/JA00513A004</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1497',
    'Abraham (1984)',
    'Abraham, M. H.: <span class="cmti-10">Thermodynamics of solution of</span> <span class="cmti-10">homologous series of solutes in water</span>, J. Chem. Soc. Faraday Trans. 1, 80, 153–181, <a href="https://doi.org/10.1039/F19848000153" >doi:10.1039/F19848000153</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3613',
    'Abraham (2003)',
    'Abraham, M. H.: <span class="cmti-10">The determination of air/water</span> <span class="cmti-10">partition coefficients for alkyl carboxylic acids by a new indirect method</span>, J. Environ. Monit., 5, 747–752, <a href="https://doi.org/10.1039/B308175C" >doi:10.1039/B308175C</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2580',
    'Abraham and Acree (2007)',
    'Abraham, M. H. &#x0026; Acree, Jr., W. E.: <span class="cmti-10">Prediction</span> <span class="cmti-10">of gas to water partition coefficients from 273 to 373 </span>K <span class="cmti-10">using predicted</span> <span class="cmti-10">enthalpies and heat capacities of hydration</span>, Fluid Phase Equilib., 262, 97–110, <a href="https://doi.org/10.1016/J.FLUID.2007.08.011" >doi:10.1016/J.FLUID.2007.08.011</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1498',
    'Abraham and Matteoli (1988)',
    'Abraham, M. H. &#x0026; Matteoli, E.: <span class="cmti-10">The</span> <span class="cmti-10">temperature variation of the hydrophobic effect</span>, J. Chem. Soc. Faraday Trans. 1, 84, 1985–2000, <a href="https://doi.org/10.1039/F19888401985" >doi:10.1039/F19888401985</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2225',
    'Abraham and Nasehzadeh (1981)',
    'Abraham, M. H. &#x0026; Nasehzadeh, A.: <span class="cmti-10">Thermodynamics of solution of gaseous tetramethyltin in 36 solvents.</span> <span class="cmti-10">Comparison of experimental results with cavity-theory calculations</span>, J. Chem. Soc. Faraday Trans. 1, 77, 321–339, <a href="https://doi.org/10.1039/F19817700321" >doi:10.1039/F19817700321</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3614',
    'Abraham and Weathersby (1994)',
    'Abraham, M. H. &#x0026; Weathersby, P. K.: <span class="cmti-10">Hydrogen bonding. 30. Solubility of gases and vapors in biological liquids</span> <span class="cmti-10">and tissues</span>, J. Pharm. Sci., 83, 1450–1456, <a href="https://doi.org/10.1002/JPS.2600831017" >doi:10.1002/JPS.2600831017</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1500',
    'Abraham et&nbsp;al. (1990)',
    'Abraham, M. H., Whiting, G. S., Fuchs, R., &#x0026; Chambers, E. J.: <span class="cmti-10">Thermodynamics</span> <span class="cmti-10">of solute transfer from water to hexadecane</span>, J. Chem. Soc. Perkin Trans. 2, pp. 291–300, <a href="https://doi.org/10.1039/P29900000291" >doi:10.1039/P29900000291</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1501',
    'Abraham et&nbsp;al. (1994a)',
    'Abraham, M. H., Andonian-Haftvan, J., Whiting, G. S., Leo, A., &#x0026; Taft, R. S.: <span class="cmti-10">Hydrogen bonding. Part 34. The factors that influence the</span> <span class="cmti-10">solubility of gases and vapours in water at 298 </span>K<span class="cmti-10">, and a new method</span> <span class="cmti-10">for its determination</span>, J. Chem. Soc. Perkin Trans. 2, pp. 1777–1791, <a href="https://doi.org/10.1039/P29940001777" >doi:10.1039/P29940001777</a> (1994a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3072',
    'Abraham et&nbsp;al. (1994b)',
    'Abraham, M. H., Chadha, H. S., Whiting, G. S., &#x0026; Mitchell, R.: <span class="cmti-10">Hydrogen bonding. 32. An analysis of water-octanol and water-alkane</span> <span class="cmti-10">partitioning and the </span>Δlog <span class="cmmi-10">P </span><span class="cmti-10">parameter of Seiler</span>, J. Pharm. Sci., 83, 1085–1100, <a href="https://doi.org/10.1002/JPS.2600830806" >doi:10.1002/JPS.2600830806</a> (1994b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2577',
    'Abraham et&nbsp;al. (2008)',
    'Abraham, M. H., Gil-Lostes, J., Acree, Jr., W. E., Cometto-Muñiz, J. E., &#x0026; Cain, W. S.: <span class="cmti-10">Solvation parameters for mercury and mercury(II)</span> <span class="cmti-10">compounds: calculation of properties of environmental interest</span>, J. Environ. Monit., 10, 435–442, <a href="https://doi.org/10.1039/B719685G" >doi:10.1039/B719685G</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3627',
    'Abraham et&nbsp;al. (2019)',
    'Abraham, M. H., Acree Jr., W. E., Hoekman, D., Leo, A. J., &#x0026; Medlin, M. L.: <span class="cmti-10">A new method for the determination of Henry&#x2019;s law constants</span> <span class="cmti-10">(air–water-partition coefficients)</span>, Fluid Phase Equilib., 502, 112 300, <a href="https://doi.org/10.1016/J.FLUID.2019.112300" >doi:10.1016/J.FLUID.2019.112300</a> (2019). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3939',
    'Abusallout et&nbsp;al. (2022)',
    'Abusallout, I., Holton, C., Wang, J., &#x0026; Hanigan, D.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">Law constants of 15 per- and polyfluoroalkyl substances determined</span> <span class="cmti-10">by static headspace analysis</span>, J. Hazard. Mater. Lett., 3, 100 070, <a href="https://doi.org/10.1016/J.HAZL.2022.100070" >doi:10.1016/J.HAZL.2022.100070</a> (2022). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3925',
    'Adams and Edmonds (1937)',
    'Adams, F. W. &#x0026; Edmonds, R. G.: <span class="cmti-10">Absorption</span> <span class="cmti-10">of chlorine by water in a packed tower</span>, Ind. Eng. Chem., 29, 447–451, <a href="https://doi.org/10.1021/IE50328A021" >doi:10.1021/IE50328A021</a> (1937). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3587',
    'Addison et&nbsp;al. (1983)',
    'Addison, R. F., Paterson, S., &#x0026; Mackay, D.: <span class="cmti-10">The predicted environmental</span> <span class="cmti-10">distribution of some PCB replacements</span>, Chemosphere, 12, 827–834, <a href="https://doi.org/10.1016/0045-6535(83)90148-0" >doi:10.1016/0045-6535(83)90148-0</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3535',
    'Aieta and Roberts (1986)',
    'Aieta, E. M. &#x0026; Roberts, P. V.: <span class="cmti-10">Henry constant of</span> <span class="cmti-10">molecular chlorine in aqueous solution</span>, J. Chem. Eng. Data, 31, 51–53, <a href="https://doi.org/10.1021/JE00043A017" >doi:10.1021/JE00043A017</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1597',
    'Alaee et&nbsp;al. (1996)',
    'Alaee, M., Whittal, R. M., &#x0026; Strachan, W. M. J.: <span class="cmti-10">The effect of water temperature and composition</span> <span class="cmti-10">on Henry&#x2019;s law constant for various PAH&#x2019;s</span>, Chemosphere, 32, 1153–1164, <a href="https://doi.org/10.1016/0045-6535(96)00031-8" >doi:10.1016/0045-6535(96)00031-8</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3946',
    'Albanese et&nbsp;al. (1987)',
    'Albanese, V., Milano, J. C., &#x0026; Vernet, J. L.: <span class="cmti-10">Etude de l&#x2019;evaporation de</span> <span class="cmti-10">quelques hydrocarbures halogenenes de faible masse moleculaire dissous</span> <span class="cmti-10">a l&#x2019;etat de traces dans l&#x2019;eau</span>, Environ. Technol. Lett., 8, 657–668, <a href="https://doi.org/10.1080/09593338709384529" >doi:10.1080/09593338709384529</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '985',
    'Allen et&nbsp;al. (1998)',
    'Allen, J. M., Balcavage, W. X., Ramachandran, B. R., &#x0026; Shrout, A. L.: <span class="cmti-10">Determination of Henry&#x2019;s Law constants by equilibrium partitioning in</span> <span class="cmti-10">a closed system using a new in situ optical absorbance method</span>, Environ. Toxicol. Chem., 17, 1216–1221, <a href="https://doi.org/10.1002/ETC.5620170704" >doi:10.1002/ETC.5620170704</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3696',
    'Allott et&nbsp;al. (1973)',
    'Allott, P. R., Steward, A., Flook, V., &#x0026; Mapleson, W. W.: <span class="cmti-10">Variation with temperature</span> <span class="cmti-10">of the solubilities of inhaled anaesthestics in water, oil and biological</span> <span class="cmti-10">media</span>, Br. J. Anaesth., 45, 294–300, <a href="https://doi.org/10.1093/BJA/45.3.294" >doi:10.1093/BJA/45.3.294</a> (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2817',
    'Allou et&nbsp;al. (2011)',
    'Allou, L., El Maimouni, L., &#x0026; Le Calvé, S.: <span class="cmti-10">Henry&#x2019;s law constant</span> <span class="cmti-10">measurements for formaldehyde and benzaldehyde as a function of</span> <span class="cmti-10">temperature and water composition</span>, Atmos. Environ., 45, 2991–2998, <a href="https://doi.org/10.1016/J.ATMOSENV.2010.05.044" >doi:10.1016/J.ATMOSENV.2010.05.044</a> (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3676',
    'Alton and Browne (2022)',
    'Alton, M. W. &#x0026; Browne, E. C.: <span class="cmti-10">Atmospheric degradation of cyclic volatile methyl</span> <span class="cmti-10">siloxanes: Radical chemistry and oxidation products</span>, ACS Environ. Au, 2, 263–274, <a href="https://doi.org/10.1021/ACSENVIRONAU.1C00043" >doi:10.1021/ACSENVIRONAU.1C00043</a> (2022). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1195',
    'Altschuh et&nbsp;al. (1999)',
    'Altschuh, J., Brüggemann, R., Santl, H., Eichinger, G., &#x0026; Piringer, O. G.: <span class="cmti-10">Henry&#x2019;s law constants for a diverse set of organic chemicals:</span> <span class="cmti-10">Experimental determination and comparison of estimation methods</span>, Chemosphere, 39, 1871–1887, <a href="https://doi.org/10.1016/S0045-6535(99)00082-X" >doi:10.1016/S0045-6535(99)00082-X</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '670',
    'Amels et&nbsp;al. (1996)',
    'Amels, P., Elias, H., Götz, U., Steingens, U., &#x0026; Wannowius, K. J.: <span class="cmti-10">Chapter 3.1: Kinetic investigation of the stability of peroxonitric acid</span> <span class="cmti-10">and of its reaction with sulfur(IV) in aqueous solution</span>, in: Heterogeneous and Liquid-Phase Processes, edited by Warneck, P., pp. 77–88, Springer Verlag, Berlin, <a href="https://doi.org/10.1007/978-3-642-61445-3_3" >doi:10.1007/978-3-642-61445-3_3</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3788',
    'Ammari and Schroen (2019)',
    'Ammari, A. &#x0026; Schroen, K.: <span class="cmti-10">Effect of ethanol</span> <span class="cmti-10">and temperature on partition coefficients of ethyl acetate, isoamyl acetate,</span> <span class="cmti-10">and isoamyl alcohol: Instrumental and predictive investigation</span>, J. Chem. Eng. Data, 64, 3224–3230, <a href="https://doi.org/10.1021/ACS.JCED.8B01125" >doi:10.1021/ACS.JCED.8B01125</a> (2019). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2388',
    'Amoore and Buttery (1978)',
    'Amoore, J. E. &#x0026; Buttery, R. G.: <span class="cmti-10">Partition</span> <span class="cmti-10">coefficient and comparative olfactometry</span>, Chem. Senses Flavour, 3, 57–71, <a href="https://doi.org/10.1093/CHEMSE/3.1.57" >doi:10.1093/CHEMSE/3.1.57</a> (1978). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3262',
    'Anderson (2011)',
    'Anderson, G. K.: <span class="cmti-10">A thermodynamic study of the</span> <span class="cmti-10">(difluoromethane + water) system</span>, J. Chem. Thermodyn., 43, 1331–1335, <a href="https://doi.org/10.1016/J.JCT.2011.03.020" >doi:10.1016/J.JCT.2011.03.020</a> (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2983',
    'Anderson (1992)',
    'Anderson, M. A.: <span class="cmti-10">Influence of surfactants on vapor-liquid partitioning</span>, Environ. Sci. Technol., 26, 2186–2191, <a href="https://doi.org/10.1021/ES00035A017" >doi:10.1021/ES00035A017</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2820',
    'Andersson et&nbsp;al. (2008)',
    'Andersson, M. E., Gårdfeldt, K., Wängberg, I., &#x0026; Strömberg, D.: <span class="cmti-10">Determination of Henry&#x2019;s law constant for elemental mercury</span>, Chemosphere, 73, 587–592, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2008.05.067" >doi:10.1016/J.CHEMOSPHERE.2008.05.067</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '719',
    'Andon et&nbsp;al. (1954)',
    'Andon, R. J. L., Cox, J. D., &#x0026; Herington, E. F. G.: <span class="cmti-10">Phase relationships</span> <span class="cmti-10">in the pyridine series. Part V. The thermodynamic properties of dilute</span> <span class="cmti-10">solutions of pyridine bases in water at 25</span><span class="cmsy-7">∘</span> <span class="cmti-10">and 40</span><span class="cmsy-7">∘</span>, J. Chem. Soc., pp. 3188–3196, <a href="https://doi.org/10.1039/JR9540003188" >doi:10.1039/JR9540003188</a> (1954). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3590',
    'Andreozzi et&nbsp;al. (1996)',
    'Andreozzi, R., Caprio, V., Ermellino, I., Insola, A., &#x0026; Tufano, V.: <span class="cmti-10">Ozone</span> <span class="cmti-10">solubility in phosphate-buffered aqueous solutions: effect of temperature,</span> <span class="cmti-10">tert-butyl alcohol, and pH</span>, Ind. Eng. Chem. Res., 35, 1467–1471, <a href="https://doi.org/10.1021/IE940778R" >doi:10.1021/IE940778R</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1151',
    'Andrew and Hanson (1961)',
    'Andrew, S. P. S. &#x0026; Hanson, D.: <span class="cmti-10">The dynamics of nitrous gas absorption</span>, Chem. Eng. Sci., 14, 105–113, <a href="https://doi.org/10.1016/0009-2509(61)85060-4" >doi:10.1016/0009-2509(61)85060-4</a> (1961). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2371',
    'Aprea et&nbsp;al. (2007)',
    'Aprea, E., Biasioli, F., Märk, T. D., &#x0026; Gasperi, F.: <span class="cmti-10">PTR-MS study of esters in water and</span> <span class="cmti-10">water/ethanol</span> <span class="cmti-10">solutions: Fragmentation patterns and partition coefficients</span>, Int. J. Mass Spectrom., 262, 114–121, <a href="https://doi.org/10.1016/J.IJMS.2006.10.016" >doi:10.1016/J.IJMS.2006.10.016</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2481',
    'Arbuckle (1983)',
    'Arbuckle, W. B.: <span class="cmti-10">Estimating activity coefficients for use</span> <span class="cmti-10">in calculating environmental parameters</span>, Environ. Sci. Technol., 17, 537–542, <a href="https://doi.org/10.1021/ES00115A008" >doi:10.1021/ES00115A008</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '629',
    'Arijs and Brasseur (1986)',
    'Arijs, E. &#x0026; Brasseur, G.: <span class="cmti-10">Acetonitrile in the</span> <span class="cmti-10">stratosphere and implications for positive ion composition</span>, J. Geophys. Res., 91, 4003–4016, <a href="https://doi.org/10.1029/JD091ID03P04003" >doi:10.1029/JD091ID03P04003</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3942',
    'Arkadiev (1918)',
    'Arkadiev, V.: <span class="cmti-10">Solubility of chlorine in water</span>, J. Russ. Phys. Chem. Soc., 50, 205–209, (in Russian) (1918). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3073',
    'Armbrust (2000)',
    'Armbrust, K. L.: <span class="cmti-10">Pesticide hydroxyl radical rate constants: Measurements and estimates of</span> <span class="cmti-10">their importance in aquatic environments</span>, Environ. Toxicol. Chem., 19, 2175–2180, <a href="https://doi.org/10.1002/ETC.5620190905" >doi:10.1002/ETC.5620190905</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3562',
    'Armstrong et&nbsp;al. (2015)',
    'Armstrong, D. A., Huie, R. E., Koppenol, W. H., Lymar, S. V., Merényi, G., Neta, P., Ruscic, B., Stanbury, D. M., Steenken, S., &#x0026; Wardman, P.: <span class="cmti-10">Standard electrode potentials involving radicals in aqueous</span> <span class="cmti-10">solution: inorganic radicals (IUPAC Technical Report)</span>, Pure Appl. Chem., 87, 1139–1150, <a href="https://doi.org/10.1515/PAC-2014-0502" >doi:10.1515/PAC-2014-0502</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2378',
    'Arnett and Chawla (1979)',
    'Arnett, E. M. &#x0026; Chawla, B.: <span class="cmti-10">Complete thermodynamic analysis of the hydration of thirteen pyridines</span> <span class="cmti-10">and pyridinium ions. The special case of 2,6-di-tert-butylpyridine</span>, J. Am. Chem. Soc., 101, 7141–7146, <a href="https://doi.org/10.1021/JA00518A001" >doi:10.1021/JA00518A001</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2386',
    'Arnett et&nbsp;al. (1977)',
    'Arnett, E. M., Chawla, B., Bell, L., Taagepera, M., Hehre, W. J., &#x0026; Taft, R. W.: <span class="cmti-10">Solvation and hydrogen bonding of pyridinium ions</span>, J. Am. Chem. Soc., 99, 5729–5738, <a href="https://doi.org/10.1021/JA00459A034" >doi:10.1021/JA00459A034</a> (1977). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2646',
    'Arp and Schmidt (2004)',
    'Arp, H. P. H. &#x0026; Schmidt, T. C.: <span class="cmti-10">Air–water transfer</span> <span class="cmti-10">of MTBE, its degradation products, and alternative fuel oxygenates:</span> <span class="cmti-10">the role of temperature</span>, Environ. Sci. Technol., 38, 5405–5412, <a href="https://doi.org/10.1021/ES049286O" >doi:10.1021/ES049286O</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2992',
    'Arp et&nbsp;al. (2006)',
    'Arp, H. P. H., Niederer, C., &#x0026; Goss, K. U.: <span class="cmti-10">Predicting the partitioning behavior of various</span> <span class="cmti-10">highly fluorinated compounds</span>, Environ. Sci. Technol., 40, 7298–7304, <a href="https://doi.org/10.1021/ES060744Y" >doi:10.1021/ES060744Y</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2831',
    'Ashton et&nbsp;al. (1968)',
    'Ashton, J. T., Dawe, R. A., Miller, K. W., Smith, E. B., &#x0026; Stickings, B. J.: <span class="cmti-10">The solubility of certain gaseous fluorine compounds in water</span>, J. Chem. Soc. A, pp. 1793–1796, <a href="https://doi.org/10.1039/J19680001793" >doi:10.1039/J19680001793</a> (1968). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1156',
    'Ashworth et&nbsp;al. (1988)',
    'Ashworth, R. A., Howe, G. B., Mullins, M. E., &#x0026; Rogers, T. N.: <span class="cmti-10">Air–water</span> <span class="cmti-10">partitioning coefficients of organics in dilute aqueous solutions</span>, J. Hazard. Mater., 18, 25–36, <a href="https://doi.org/10.1016/0304-3894(88)85057-X" >doi:10.1016/0304-3894(88)85057-X</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1926',
    'Atlas et&nbsp;al. (1982)',
    'Atlas, E., Foster, R., &#x0026; Giam, C. S.: <span class="cmti-10">Air-sea exchange of high-molecular</span> <span class="cmti-10">weight organic pollutants: laboratory studies</span>, Environ. Sci. Technol., 16, 283–286, <a href="https://doi.org/10.1021/ES00099A010" >doi:10.1021/ES00099A010</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2533',
    'Atlas et&nbsp;al. (1983)',
    'Atlas, E., Velasco, A., Sullivan, K., &#x0026; Giam, C. S.: <span class="cmti-10">A radiotracer study of air–water</span> <span class="cmti-10">exchange of synthetic organic compounds</span>, Chemosphere, 12, 1251–1258, <a href="https://doi.org/10.1016/0045-6535(83)90130-3" >doi:10.1016/0045-6535(83)90130-3</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1076',
    'Ayers (1983)',
    'Ayers, G. P.: <span class="cmti-10">Equilibrium</span> <span class="cmti-10">partial pressures over</span> (NH<sub><span class="cmr-7">4</span></sub>)<sub><span class="cmr-7">2</span></sub>SO<sub><span class="cmr-7">4</span></sub><span class="cmti-10">/</span>H<sub><span class="cmr-7">2</span></sub>SO<sub><span class="cmr-7">4</span></sub> <span class="cmti-10">mixtures</span>, Aust. J. Chem., 36, 179–182, <a href="https://doi.org/10.1071/CH9830179" >doi:10.1071/CH9830179</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '201',
    'Ayers et&nbsp;al. (1980)',
    'Ayers, G. P., Gillett, R. W., &#x0026; Gras, J. L.: <span class="cmti-10">On the vapor pressure of sulfuric acid</span>, Geophys. Res. Lett., 7, 433–436, <a href="https://doi.org/10.1029/GL007I006P00433" >doi:10.1029/GL007I006P00433</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2969',
    'Ayuttaya et&nbsp;al. (2001)',
    'Ayuttaya, P. C. N., Rogers, T. N., Mullins, M. E., &#x0026; Kline, A. A.: <span class="cmti-10">Henry&#x2019;s law constants derived from equilibrium static cell measurements</span> <span class="cmti-10">for dilute organic-water mixtures</span>, Fluid Phase Equilib., 185, 359–377, <a href="https://doi.org/10.1016/S0378-3812(01)00484-8" >doi:10.1016/S0378-3812(01)00484-8</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3697',
    'Bachofen and Farhi (1971)',
    'Bachofen, H. &#x0026; Farhi, L. E.: <span class="cmti-10">Simple manometric</span> <span class="cmti-10">apparatus for measuring partition coefficients of highly soluble gases</span>, J. Appl. Physiol., 30, 136–139, <a href="https://doi.org/10.1152/JAPPL.1971.30.1.136" >doi:10.1152/JAPPL.1971.30.1.136</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3771',
    'Badia et&nbsp;al. (2019)',
    'Badia, A., Reeves, C. E., Baker, A. R., Saiz-Lopez, A., Volkamer, R., Koenig, T. K., Apel, E. C., Hornbrook, R. S., Carpenter, L. J., Andrews, S. J., Sherwen, T., &#x0026; von Glasow, R.: <span class="cmti-10">Importance</span> <span class="cmti-10">of reactive halogens in the tropical marine atmosphere: a regional</span> <span class="cmti-10">modelling study using WRF-Chem</span>, Atmos. Chem. Phys., 19, 3161–3189, <a href="https://doi.org/10.5194/ACP-19-3161-2019" >doi:10.5194/ACP-19-3161-2019</a> (2019). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2016',
    'Bagno et&nbsp;al. (1991)',
    'Bagno, A., Lucchini, V., &#x0026; Scorrano, G.: <span class="cmti-10">Thermodynamics of protonation of ketones and esters</span> <span class="cmti-10">and energies of hydration of their conjugate acids</span>, J. Phys. Chem., 95, 345–352, <a href="https://doi.org/10.1021/J100154A063" >doi:10.1021/J100154A063</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3938',
    'Bakhuis Roozeboom (1884)',
    'Bakhuis Roozeboom, H. W.: <span class="cmti-10">Sur l&#x2019;hydrate de chlore</span>, Recl. Trav. Chim. Pays-Bas, 3, 59–72, <a href="https://doi.org/10.1002/RECL.18840030203" >doi:10.1002/RECL.18840030203</a> (1884). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2923',
    'Bakierowska and Trzeszczyński (2003)',
    'Bakierowska, A.-M. &#x0026; Trzeszczyński, J.: <span class="cmti-10">Graphical method for the determination of water/gas partition</span> <span class="cmti-10">coefficients of volatile organic compounds by a headspace gas</span> <span class="cmti-10">chromatography technique</span>, Fluid Phase Equilib., 213, 139–146, <a href="https://doi.org/10.1016/S0378-3812(03)00286-3" >doi:10.1016/S0378-3812(03)00286-3</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2952',
    'Balls (1980)',
    'Balls, P. W.: Gas transfer across air–water interfaces, Ph.D. thesis, University of East Anglia, Great Britain (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2974',
    'Ballschmiter and Wittlinger (1991)',
    'Ballschmiter, K. &#x0026; Wittlinger, R.: <span class="cmti-10">Interhemisphere exchange of hexachlorocyclohexanes, hexachlorobenzene,</span> <span class="cmti-10">polychlorobiphenyls, and 1,1,1-trichloro-2,2-bis(p-chlorophenyl)ethane</span> <span class="cmti-10">in the lower troposphere</span>, Environ. Sci. Technol., 25, 1103–1111, <a href="https://doi.org/10.1021/ES00018A014" >doi:10.1021/ES00018A014</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2477',
    'Bamford et&nbsp;al. (1999a)',
    'Bamford, H. A., Poster, D. L., &#x0026; Baker, J. E.: <span class="cmti-10">Temperature dependence of Henry&#x2019;s law constants</span> <span class="cmti-10">of thirteen polycyclic aromatic hydrocarbons between 4</span><span class="cmsy-7">∘</span>C <span class="cmti-10">and 31</span><span class="cmsy-7">∘</span>C, Environ. Toxicol. Chem., 18, 1905–1912, <a href="https://doi.org/10.1002/ETC.5620180906" >doi:10.1002/ETC.5620180906</a> (1999a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2914',
    'Bamford et&nbsp;al. (1999b)',
    'Bamford, H. A., Poster, D. L., &#x0026; Baker, J. E.: <span class="cmti-10">Method for measuring the temperature dependence</span> <span class="cmti-10">of the Henry&#x2019;s law constant of selected polycyclic aromatic hydrocarbons</span>, Polycyclic Aromat. Compd., 14, 11–22, <a href="https://doi.org/10.1080/10406639908019107" >doi:10.1080/10406639908019107</a> (1999b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2340',
    'Bamford et&nbsp;al. (2000)',
    'Bamford, H. A., Poster, D. L., &#x0026; Baker, J. E.: <span class="cmti-10">Henry&#x2019;s law constants of polychlorinated biphenyl</span> <span class="cmti-10">congeners and their variation with temperature</span>, J. Chem. Eng. Data, 45, 1069–1074, <a href="https://doi.org/10.1021/JE0000266" >doi:10.1021/JE0000266</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2472',
    'Bamford et&nbsp;al. (2002)',
    'Bamford, H. A., Poster, D. L., Huie, R. E., &#x0026; Baker, J. E.: <span class="cmti-10">Using</span> <span class="cmti-10">extrathermodynamic relationships to model the temperature dependence</span> <span class="cmti-10">of Henry&#x2019;s law constants of 209 PCB congeners</span>, Environ. Sci. Technol., 36, 4395–4402, <a href="https://doi.org/10.1021/ES020599Y" >doi:10.1021/ES020599Y</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1773',
    'Barcellos da Rosa et&nbsp;al. (2003)',
    'Barcellos da Rosa, M., Behnke, W., &#x0026; Zetzsch, C.: <span class="cmti-10">Study</span> <span class="cmti-10">of the heterogeneous reaction of</span> O<sub><span class="cmr-7">3</span></sub> <span class="cmti-10">with</span> CH<sub><span class="cmr-7">3</span></sub>SCH<sub><span class="cmr-7">3</span></sub> <span class="cmti-10">using the</span> <span class="cmti-10">wetted-wall flowtube technique</span>, Atmos. Chem. Phys., 3, 1665–1673, <a href="https://doi.org/10.5194/ACP-3-1665-2003" >doi:10.5194/ACP-3-1665-2003</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'barcelo97',
    'Barcelo and Hennion (1997)',
    'Barcelo, D. &#x0026; Hennion, M. C.: Trace Determination of Pesticides and Their Degradation Products in Water, Elsevier Science, Amsterdam, ISBN 9780444818423 (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1147',
    'Barr and Newsham (1987)',
    'Barr, R. S. &#x0026; Newsham, D. M. T.: <span class="cmti-10">Phase</span> <span class="cmti-10">equilibria in very dilute mixtures of water and chlorinated hydrocarbons.</span> <span class="cmti-10">Part I. Experimental results</span>, Fluid Phase Equilib., 35, 189–205, <a href="https://doi.org/10.1016/0378-3812(87)80012-2" >doi:10.1016/0378-3812(87)80012-2</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2928',
    'Barrett et&nbsp;al. (1988)',
    'Barrett, T. J., Anderson, G. M., &#x0026; Lugowski, J.: <span class="cmti-10">The solubility of hydrogen sulphide</span> <span class="cmti-10">in 0–5m NaCl solutions at 25</span><span class="cmsy-7">∘</span><span class="cmti-10">-95</span><span class="cmsy-7">∘</span>C <span class="cmti-10">and one atmosphere</span>, Geochim. Cosmochim. Acta, 52, 807–811, <a href="https://doi.org/10.1016/0016-7037(88)90352-3" >doi:10.1016/0016-7037(88)90352-3</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3393',
    'Bartelt-Hunt et&nbsp;al. (2008)',
    'Bartelt-Hunt, S. L., Knappe, D. R. U., &#x0026; Barlaz, M. A.: <span class="cmti-10">A review of chemical warfare</span> <span class="cmti-10">agent simulants for the study of environmental behavior</span>, Crit. Rev. Environ. Sci. Technol., 38, 112–136, <a href="https://doi.org/10.1080/10643380701643650" >doi:10.1080/10643380701643650</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1038',
    'Bartlett and Margerum (1999)',
    'Bartlett, W. P. &#x0026; Margerum, D. W.: <span class="cmti-10">Temperature dependencies of the Henry&#x2019;s law constant and the aqueous</span> <span class="cmti-10">phase dissociation constant of bromine chloride</span>, Environ. Sci. Technol., 33, 3410–3414, <a href="https://doi.org/10.1021/ES990300K" >doi:10.1021/ES990300K</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3853',
    'Battino (1981)',
    'Battino, R.: IUPAC Solubility Data Series, Volume 7, Oxygen and Ozone, Pergamon Press, Oxford, <a href="https://doi.org/10.1016/C2013-0-03145-3" >doi:10.1016/C2013-0-03145-3</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3856',
    'Battino (1982)',
    'Battino, R.: IUPAC Solubility Data Series, Volume 10, Nitrogen and Air, Pergamon Press, Oxford, ISBN 0080239226 (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2413',
    'Battino and Clever (1966)',
    'Battino, R. &#x0026; Clever, H. L.: <span class="cmti-10">The solubility of</span> <span class="cmti-10">gases in liquids</span>, Chem. Rev., 66, 395–463, <a href="https://doi.org/10.1021/CR60242A003" >doi:10.1021/CR60242A003</a> (1966). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2301',
    'Battino et&nbsp;al. (1983)',
    'Battino, R., Rettich, T. R., &#x0026; Tominaga, T.: <span class="cmti-10">The solubility of oxygen and ozone in liquids</span>, J. Phys. Chem. Ref. Data, 12, 163–178, <a href="https://doi.org/10.1063/1.555680" >doi:10.1063/1.555680</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2433',
    'Battino et&nbsp;al. (1984)',
    'Battino, R., Rettich, T. R., &#x0026; Tominaga, T.: <span class="cmti-10">The solubility of nitrogen and air in liquids</span>, J. Phys. Chem. Ref. Data, 13, 563–600, <a href="https://doi.org/10.1063/1.555713" >doi:10.1063/1.555713</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3407',
    'Battino et&nbsp;al. (2018)',
    'Battino, R., Rettich, T. R., &#x0026; Wilhelm, E.: <span class="cmti-10">Gas solubilities in liquid water near the</span> <span class="cmti-10">temperature of the density maximum, </span><span class="cmmi-10">T</span><sub><span class="cmr-7">max</span></sub><span class="cmti-10">(</span>H<sub><span class="cmr-7">2</span></sub>O<span class="cmti-10">) = 277.13K</span>, Monatsh. Chem. – Chem. Mon., 149, 219–230, <a href="https://doi.org/10.1007/S00706-017-2097-3" >doi:10.1007/S00706-017-2097-3</a> (2018). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2963',
    'Bebahani et&nbsp;al. (2002)',
    'Bebahani, G. R. R., Hogan, P., &#x0026; Waghorne, W. E.: <span class="cmti-10">Ostwald concentration</span> <span class="cmti-10">coefficients of acetonitrile in aqueous mixed solvents: a new, rapid method</span> <span class="cmti-10">for measuring the solubilities of volatile solutes</span>, J. Chem. Eng. Data, 47, 1290–1292, <a href="https://doi.org/10.1021/JE0200665" >doi:10.1021/JE0200665</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '625',
    'Becker et&nbsp;al. (1996)',
    'Becker, K. H., Kleffmann, J., Kurtenbach, R., &#x0026; Wiesen, P.: <span class="cmti-10">Solubility</span> <span class="cmti-10">of nitrous acid (</span>HONO<span class="cmti-10">) in sulfuric acid solutions</span>, J. Phys. Chem., 100, 14 984–14 990, <a href="https://doi.org/10.1021/JP961140R" >doi:10.1021/JP961140R</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1056',
    'Becker et&nbsp;al. (1998)',
    'Becker, K. H., Kleffmann, J., Negri, R. M., &#x0026; Wiesen, P.: <span class="cmti-10">Solubility of nitrous</span> <span class="cmti-10">acid (</span>HONO<span class="cmti-10">) in ammonium sulfate solutions</span>, J. Chem. Soc. Faraday Trans., 94, 1583–1586, <a href="https://doi.org/10.1039/A800458G" >doi:10.1039/A800458G</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '536',
    'Behnke et&nbsp;al. (1997)',
    'Behnke, W., George, C., Scheer, V., &#x0026; Zetzsch, C.: <span class="cmti-10">Production and decay of</span> ClNO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">from the reaction of gaseous</span> N<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">5</span></sub> <span class="cmti-10">with</span> NaCl <span class="cmti-10">solution: Bulk and aerosol</span> <span class="cmti-10">experiments</span>, J. Geophys. Res., 102, 3795–3804, <a href="https://doi.org/10.1029/96JD03057" >doi:10.1029/96JD03057</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '821',
    'Beilke and Gravenhorst (1978)',
    'Beilke, S. &#x0026; Gravenhorst, G.: <span class="cmti-10">Heterogeneous</span> SO<sub><span class="cmr-7">2</span></sub><span class="cmti-10">-oxidation in the droplet phase</span>, Atmos. Environ., 12, 231–239, <a href="https://doi.org/10.1016/0004-6981(78)90203-2" >doi:10.1016/0004-6981(78)90203-2</a> (1978). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'bell66',
    'Bell (1966)',
    'Bell, R. P.: <span class="cmti-10">The reversible hydration of carbonyl compounds</span>, Adv. Phys. Org. Chem., 4, 1–29, <a href="https://doi.org/10.1016/S0065-3160(08)60351-2" >doi:10.1016/S0065-3160(08)60351-2</a> (1966). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3734',
    'Ben-Naim and Battino (1985)',
    'Ben-Naim, A. &#x0026; Battino, R.: <span class="cmti-10">Solubilization of</span> <span class="cmti-10">methane, ethane, propane and </span><span class="cmmi-10">n</span><span class="cmti-10">-butane in aqueous solutions of sodium</span> <span class="cmti-10">dodecylsulfate</span>, J. Solution Chem., 14, 245–253, <a href="https://doi.org/10.1007/BF00644456" >doi:10.1007/BF00644456</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2219',
    'Ben-Naim and Wilf (1980)',
    'Ben-Naim, A. &#x0026; Wilf, J.: <span class="cmti-10">Solubilities and</span> <span class="cmti-10">hydrophobic interactions in aqueous solutions of monoalkylbenzene</span> <span class="cmti-10">molecules</span>, J. Phys. Chem., 84, 583–586, <a href="https://doi.org/10.1021/J100443A004" >doi:10.1021/J100443A004</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1996',
    'Beneš and Dohnal (1999)',
    'Beneš, M. &#x0026; Dohnal, V.: <span class="cmti-10">Limiting activity</span> <span class="cmti-10">coefficients of some aromatic and aliphatic nitro compounds in water</span>, J. Chem. Eng. Data, 44, 1097–1102, <a href="https://doi.org/10.1021/JE9900326" >doi:10.1021/JE9900326</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '617',
    'Benkelberg et&nbsp;al. (1995)',
    'Benkelberg, H.-J., Hamm, S., &#x0026; Warneck, P.: <span class="cmti-10">Henry&#x2019;s law coefficients for aqueous</span> <span class="cmti-10">solutions of acetone, acetaldehyde and acetonitrile, and equilibrium</span> <span class="cmti-10">constants for the addition compounds of acetone and acetaldehyde with</span> <span class="cmti-10">bisulfite</span>, J. Atmos. Chem., 20, 17–34, <a href="https://doi.org/10.1007/BF01099916" >doi:10.1007/BF01099916</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2437',
    'Benson et&nbsp;al. (1979)',
    'Benson, B. B., Krause, Jr., D., &#x0026; Peterson, M. A.: <span class="cmti-10">The solubility and isotopic fractionation of</span> <span class="cmti-10">gases in dilute aqueous solution. I. oxygen</span>, J. Solution Chem., 8, 655–690, <a href="https://doi.org/10.1007/BF01033696" >doi:10.1007/BF01033696</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '763',
    'Berdnikov and Bazhin (1970)',
    'Berdnikov, V. M. &#x0026; Bazhin, N. M.: <span class="cmti-10">Oxidation-reduction potentials of certain inorganic radicals in aqueous</span> <span class="cmti-10">solutions</span>, Russ. J. Phys. Chem., 44, 395–398 (1970). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2601',
    'Bernauer and Dohnal (2008)',
    'Bernauer, M. &#x0026; Dohnal, V.: <span class="cmti-10">Temperature</span> <span class="cmti-10">dependence of air–water partitioning of </span>N<span class="cmti-10">-methylated (</span>C1 <span class="cmti-10">and</span> C2<span class="cmti-10">) fatty</span> <span class="cmti-10">acid amides</span>, J. Chem. Eng. Data, 53, 2622–2631, <a href="https://doi.org/10.1021/JE800517R" >doi:10.1021/JE800517R</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2826',
    'Bernauer and Dohnal (2009)',
    'Bernauer, M. &#x0026; Dohnal, V.: <span class="cmti-10">Temperature</span> <span class="cmti-10">dependences of limiting activity coefficients and Henry&#x2019;s law constants</span> <span class="cmti-10">for N-methylpyrrolidone, pyridine, and piperidine in water</span>, Fluid Phase Equilib., 282, 100–107, <a href="https://doi.org/10.1016/J.FLUID.2009.05.005" >doi:10.1016/J.FLUID.2009.05.005</a> (2009). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2123',
    'Bernauer et&nbsp;al. (2006)',
    'Bernauer, M., Dohnal, V., Roux, A. H., Roux-Desgranges, G., &#x0026; Majer, V.: <span class="cmti-10">Temperature dependences of limiting activity coefficients and Henry&#x2019;s</span> <span class="cmti-10">law constants for nitrobenzene, aniline, and cyclohexylamine in water</span>, J. Chem. Eng. Data, 51, 1678–1685, <a href="https://doi.org/10.1021/JE060136Y" >doi:10.1021/JE060136Y</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '495',
    'Betterton (1991)',
    'Betterton, E. A.: <span class="cmti-10">The partitioning of ketones between</span> <span class="cmti-10">the gas and aqueous phases</span>, Atmos. Environ., 25A, 1473–1477, <a href="https://doi.org/10.1016/0960-1686(91)90006-S" >doi:10.1016/0960-1686(91)90006-S</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '510',
    'Betterton (1992)',
    'Betterton, E. A.: <span class="cmti-10">Henry&#x2019;s law constants of soluble and</span> <span class="cmti-10">moderately soluble organic gases: Effects on aqueous phase chemistry</span>, Adv. Environ. Sci. Technol., 24, 1–50 (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '484',
    'Betterton and Hoffmann (1988)',
    'Betterton, E. A. &#x0026; Hoffmann, M. R.: <span class="cmti-10">Henry&#x2019;s law constants of some environmentally important aldehydes</span>, Environ. Sci. Technol., 22, 1415–1418, <a href="https://doi.org/10.1021/ES00177A004" >doi:10.1021/ES00177A004</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2806',
    'Betterton and Robinson (1997)',
    'Betterton, E. A. &#x0026; Robinson, J. L.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">law coefficient of hydrazoic acid</span>, J. Air Waste Manage. Assoc., 47, 1216–1219, <a href="https://doi.org/10.1080/10473289.1997.10464060" >doi:10.1080/10473289.1997.10464060</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3802',
    'Bettin and Spieweck (1990)',
    'Bettin, H. &#x0026; Spieweck, F.: <span class="cmti-10">Die</span> <span class="cmti-10">Dichte des Wassers als Funktion der Temperatur</span> <span class="cmti-10">nach Einf</span><span class="cmti-10">ührung der Internationalen Temperaturskala</span> <span class="cmti-10">von 1990</span>, PTB-Mitteilungen, 100, 195–196, URL https://www.ptb.de/cms/fileadmin/internet/publikationen/ptb_mitteilungen/mitt_pdf_vor_2007/1990/PTB-Mitteilungen_3_1990.pdf (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3424',
    'Bhangare et&nbsp;al. (2019)',
    'Bhangare, R. C., Ajmal, P. Y., Rathod, T. D., Tiwari, M., &#x0026; Sahu, S. K.: <span class="cmti-10">Experimental and theoretical determination of Henry&#x2019;s law</span> <span class="cmti-10">constant for polychlorinated biphenyls: its dependence on solubility and</span> <span class="cmti-10">degree of chlorination</span>, Arch. Environ. Contam. Toxicol., 76, 142–152, <a href="https://doi.org/10.1007/S00244-018-0577-Z" >doi:10.1007/S00244-018-0577-Z</a> (2019). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2522',
    'Bierwagen and Keller (2001)',
    'Bierwagen, B. G. &#x0026; Keller, A. A.: <span class="cmti-10">Measurement of Henry&#x2019;s law constant</span> <span class="cmti-10">for methyl tert-butyl ether using solid-phase microextraction</span>, Environ. Toxicol. Chem., 20, 1625–1629, <a href="https://doi.org/10.1002/ETC.5620200802" >doi:10.1002/ETC.5620200802</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3967',
    'Bigorgne (1947)',
    'Bigorgne, M.: <span class="cmti-10">Formation de chlorites </span><span class="cmti-10">à partir de peroxyde</span> <span class="cmti-10">de chlore et de m</span><span class="cmti-10">étaux</span>, C. R. Hebd. Séances Acad. Sci., 225, 527–529 (1947). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3747',
    'Billitzer (1902)',
    'Billitzer, J.: <span class="cmti-10">Über die saure Natur des Acetylens</span>, Z. Phys. Chem., 40, 535–544, <a href="https://doi.org/10.1515/ZPCH-1902-4026" >doi:10.1515/ZPCH-1902-4026</a> (1902). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3589',
    'Biń (2005)',
    'Biń, A. K.: <span class="cmti-10">Ozone solubility in liquids</span>, Ozone: Sci. Eng., 28, 67–75, <a href="https://doi.org/10.1080/01919510600558635" >doi:10.1080/01919510600558635</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2270',
    'Bissonette et&nbsp;al. (1990)',
    'Bissonette, E. M., Westrick, J. J., &#x0026; Morand, J. M.: <span class="cmti-10">Determination of Henry&#x2019;s</span> <span class="cmti-10">coefficient for volatile organic compounds in dilute aqueous systems</span>, in: Proceedings of the Annual Conference of the American Water Works Association, Cincinnati, OH, June 17–21, pp. 1913–1922 (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '486',
    'Blair and Ledbury (1925)',
    'Blair, E. W. &#x0026; Ledbury, W.: <span class="cmti-10">The partial</span> <span class="cmti-10">formaldehyde vapour pressures of aqueous solutions of formaldehyde.</span> <span class="cmti-10">Part I</span>, J. Chem. Soc., 127, 26–40, <a href="https://doi.org/10.1039/CT9252700026" >doi:10.1039/CT9252700026</a> (1925). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '288',
    'Blatchley et&nbsp;al. (1992)',
    'Blatchley, III, E. R., Johnson, R. W., Alleman, J. E., &#x0026; McCoy, W. F.: <span class="cmti-10">Effective</span> <span class="cmti-10">Henry&#x2019;s law constants for free chlorine and free bromine</span>, Wat. Res., 26, 99–106, <a href="https://doi.org/10.1016/0043-1354(92)90117-M" >doi:10.1016/0043-1354(92)90117-M</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2899',
    'Bobadilla et&nbsp;al. (2003)',
    'Bobadilla, R., Huybrechts, T., Dewulf, J., &#x0026; van Langenhove, H.: <span class="cmti-10">Determination of the Henry&#x2019;s constant of volatile and semi-volatile</span> <span class="cmti-10">organic componuds of environmental concern by the bas (batch air</span> <span class="cmti-10">stripping) technique: a new mathematical approach</span>, J. Chilean Chem. Soc., 48, <a href="https://doi.org/10.4067/S0717-97072003000300001" >doi:10.4067/S0717-97072003000300001</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3110',
    'Bobra et&nbsp;al. (1985)',
    'Bobra, A., Shiu, W. Y., &#x0026; Mackay, D.: <span class="cmti-10">Quantitative structure-activity relationships for the acute</span> <span class="cmti-10">toxicity of chlorobenzenes to daphnia magna</span>, Environ. Toxicol. Chem., 4, 297–305, <a href="https://doi.org/10.1002/ETC.5620040305" >doi:10.1002/ETC.5620040305</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2552',
    'Boggs and Buck (1958)',
    'Boggs, J. E. &#x0026; Buck, Jr., A. E.: <span class="cmti-10">The solubility</span> <span class="cmti-10">of some chloromethanes in water</span>, J. Phys. Chem., 62, 1459–1461, <a href="https://doi.org/10.1021/J150569A031" >doi:10.1021/J150569A031</a> (1958). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3302',
    'Böhme et&nbsp;al. (2008)',
    'Böhme, A., Paschke, A., Vrbka, P., Dohnal, V., &#x0026; Schüürmann, G.: <span class="cmti-10">Determination of temperature-dependent Henry&#x2019;s law constant of four</span> <span class="cmti-10">oxygenated solutes in water using headspace solid-phase microextraction</span> <span class="cmti-10">technique</span>, J. Chem. Eng. Data, 53, 2873–2877, <a href="https://doi.org/10.1021/JE800623X" >doi:10.1021/JE800623X</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '790',
    'Bohon and Claussen (1951)',
    'Bohon, R. J. &#x0026; Claussen, W. F.: <span class="cmti-10">The solubility</span> <span class="cmti-10">of aromatic hydrocarbons in water</span>, J. Am. Chem. Soc., 73, 1571–1578, <a href="https://doi.org/10.1021/JA01148A047" >doi:10.1021/JA01148A047</a> (1951). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2841',
    'Bohr (1899)',
    'Bohr, C.: <span class="cmti-10">Definition und Methode zur Bestimmung der Invasions- und</span> <span class="cmti-10">Evasionscoefficienten bei der Aufl</span><span class="cmti-10">ösung von Gasen in Fl</span><span class="cmti-10">üssigkeiten.</span> <span class="cmti-10">Werthe der genannten Constanten sowie der Absorptionscoefficienten</span> <span class="cmti-10">der Kohlens</span><span class="cmti-10">äure bei Aufl</span><span class="cmti-10">ösung in Wasser und in Chlornatriuml</span><span class="cmti-10">ösungen</span>, Wied. Ann., 68, 500–525, <a href="https://doi.org/10.1002/ANDP.18993040707" >doi:10.1002/ANDP.18993040707</a> (1899). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3713',
    'Bohr and Bock (1891)',
    'Bohr, C. &#x0026; Bock, J.: <span class="cmti-10">Bestimmung der Absorption</span> <span class="cmti-10">einiger Gase in Wasser bei den Temperaturen zwischen 0 und 100</span><span class="cmsy-7">∘</span>, Ann. Phys., 280, 318–343, <a href="https://doi.org/10.1002/ANDP.18912801010" >doi:10.1002/ANDP.18912801010</a> (1891). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '641',
    'Bone et&nbsp;al. (1983)',
    'Bone, R., Cullis, P., &#x0026; Wolfenden, R.: <span class="cmti-10">Solvent effects on equilibria of addition of nucleophiles to</span> <span class="cmti-10">acetaldehyde and the hydrophilic character of diols</span>, J. Am. Chem. Soc., 105, 1339–1343, <a href="https://doi.org/10.1021/JA00343A044" >doi:10.1021/JA00343A044</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2610',
    'Bonifácio et&nbsp;al. (2001)',
    'Bonifácio, R. P., Pádua, A. A. H., &#x0026; Costa Gomes, M. F.: <span class="cmti-10">Perfluoroalkanes in</span> <span class="cmti-10">water: experimental Henry&#x2019;s law coefficients for hexafluoroethane and</span> <span class="cmti-10">computer simulations for tetrafluoromethane and hexafluoroethane</span>, J. Phys. Chem. B, 105, 8403–8409, <a href="https://doi.org/10.1021/JP010597K" >doi:10.1021/JP010597K</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2835',
    'Booth and Jolley (1943)',
    'Booth, N. &#x0026; Jolley, L. J.: <span class="cmti-10">The removal of organic</span> <span class="cmti-10">sulphur compounds from gases</span>, J. Soc. Chem. Ind., 62, 87–88, <a href="https://doi.org/10.1002/JCTB.5000620603" >doi:10.1002/JCTB.5000620603</a> (1943). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3794',
    'Bopp (1983)',
    'Bopp, R. F.: <span class="cmti-10">Revised parameters for modeling the transport</span> <span class="cmti-10">of PCB components across an air water interface</span>, J. Geophys. Res., 88, 2521–2529, <a href="https://doi.org/10.1029/JC088IC04P02521" >doi:10.1029/JC088IC04P02521</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3411',
    'Borduas et&nbsp;al. (2016)',
    'Borduas, N., Place, B., Wentworth, G. R., Abbatt, J. P. D., &#x0026; Murphy, J. G.: <span class="cmti-10">Solubility and reactivity of HNCO in water: insights</span> <span class="cmti-10">into HNCO&#x2019;s fate in the atmosphere</span>, Atmos. Chem. Phys., 16, 703–714, <a href="https://doi.org/10.5194/ACP-16-703-2016" >doi:10.5194/ACP-16-703-2016</a> (2016). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '619',
    'Bowden et&nbsp;al. (1996)',
    'Bowden, D. J., Clegg, S. L., &#x0026; Brimblecombe, P.: <span class="cmti-10">The Henry&#x2019;s law constant of</span> <span class="cmti-10">trifluoroacetic acid</span> <span class="cmti-10">and its partitioning into liquid water in the atmosphere</span>, Chemosphere, 32, 405–420, <a href="https://doi.org/10.1016/0045-6535(95)00330-4" >doi:10.1016/0045-6535(95)00330-4</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '896',
    'Bowden et&nbsp;al. (1998a)',
    'Bowden, D. J., Clegg, S. L., &#x0026; Brimblecombe, P.: <span class="cmti-10">The Henry&#x2019;s law</span> <span class="cmti-10">constants of the haloacetic acids</span>, J. Atmos. Chem., 29, 85–107, <a href="https://doi.org/10.1023/A:1005899813756" >doi:10.1023/A:1005899813756</a> (1998a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '992',
    'Bowden et&nbsp;al. (1998b)',
    'Bowden, D. J., Clegg, S. L., &#x0026; Brimblecombe, P.: <span class="cmti-10">The Henry&#x2019;s law</span> <span class="cmti-10">constant of trichloroacetic acid</span>, Water Air Soil Pollut., 101, 197–215, <a href="https://doi.org/10.1023/A:1004966126770" >doi:10.1023/A:1004966126770</a> (1998b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3790',
    'Brandsch et&nbsp;al. (1993)',
    'Brandsch, R., Gruber, L., &#x0026; Santl, H.: <span class="cmti-10">Experimental determination of Henry&#x2019;s law constant for</span> <span class="cmti-10">some dioxins</span>, Organohalogen Compd., 12, 369–372 (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2570',
    'Braun and Dransfeld (1989)',
    'Braun, H. &#x0026; Dransfeld, P.: <span class="cmti-10">Abscheidung von</span> <span class="cmti-10">Quecksilber</span>, GVC/VDI-Tagung “Entsorgung von Sonderabfällen durch Verbrennung”, Baden-Baden, 4–6 Dec 1989 (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3795',
    'Braun (1900)',
    'Braun, L.: <span class="cmti-10">Über die Absorption von Stickstoff und von</span> <span class="cmti-10">Wasserstoff in w</span><span class="cmti-10">ässerigen L</span><span class="cmti-10">ösungen verschieden dissociierter Stoffe</span>, Z. Phys. Chem., 33U, 721–739, <a href="https://doi.org/10.1515/ZPCH-1900-3349" >doi:10.1515/ZPCH-1900-3349</a> (1900). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2241',
    'Breiter et&nbsp;al. (1998)',
    'Breiter, W. A., Baker, J. M., &#x0026; Koskinen, W. C.: <span class="cmti-10">Direct measurement of Henry&#x2019;s constant</span> <span class="cmti-10">for S-ethyl N,N-di-n-propylthiocarbamate</span>, J. Agric. Food Chem., 46, 1624–1629, <a href="https://doi.org/10.1021/JF980042V" >doi:10.1021/JF980042V</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1064',
    'Brennan et&nbsp;al. (1998)',
    'Brennan, R. A., Nirmalakhandan, N., &#x0026; Speece, R. E.: <span class="cmti-10">Comparison of predictive</span> <span class="cmti-10">methods for Henrys law coefficients of organic chemicals</span>, Wat. Res., 32, 1901–1911, <a href="https://doi.org/10.1016/S0043-1354(97)00402-8" >doi:10.1016/S0043-1354(97)00402-8</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '750',
    'Brian et&nbsp;al. (1962)',
    'Brian, P. L. T., Vivian, J. E., &#x0026; Habib, A. G.: <span class="cmti-10">The effect of the hydrolysis reaction upon the</span> <span class="cmti-10">rate of absorption of chlorine into water</span>, AIChE J., 8, 205–209, <a href="https://doi.org/10.1002/AIC.690080215" >doi:10.1002/AIC.690080215</a> (1962). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '695',
    'Brimblecombe (1986)',
    'Brimblecombe, P.: Air Composition &#x0026; Chemistry, Cambridge University Press, Cambridge, ISBN 0521459729 (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '260',
    'Brimblecombe and Clegg (1988)',
    'Brimblecombe, P. &#x0026; Clegg, S. L.: <span class="cmti-10">The</span> <span class="cmti-10">solubility and behaviour of acid gases in the marine aerosol</span>, J. Atmos. Chem., 7, 1–18, <a href="https://doi.org/10.1007/BF00048251" >doi:10.1007/BF00048251</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '530',
    'Brimblecombe and Clegg (1989)',
    'Brimblecombe, P. &#x0026; Clegg, S. L.: <span class="cmti-10">Erratum</span>, J. Atmos. Chem., 8, 95, <a href="https://doi.org/10.1007/BF00053818" >doi:10.1007/BF00053818</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2231',
    'Brimblecombe et&nbsp;al. (1992)',
    'Brimblecombe, P., Clegg, S. L., &#x0026; Khan, I.: <span class="cmti-10">Thermodynamic properties</span> <span class="cmti-10">of carboxylic acids relevant to their solubility in aqueous solutions</span>, J. Aerosol Sci., 23, S901–S904, <a href="https://doi.org/10.1016/0021-8502(92)90557-C" >doi:10.1016/0021-8502(92)90557-C</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '528',
    'Briner and Perrottet (1939)',
    'Briner, E. &#x0026; Perrottet, E.: <span class="cmti-10">D</span><span class="cmti-10">étermination des</span> <span class="cmti-10">solubilit</span><span class="cmti-10">és de l&#x2019;ozone dans l&#x2019;eau et dans une solution aqueuse de chlorure</span> <span class="cmti-10">de sodium; calcul des solubilit</span><span class="cmti-10">és de l&#x2019;ozone atmosph</span><span class="cmti-10">érique dans les eaux</span>, Helv. Chim. Acta, 22, 397–404, <a href="https://doi.org/10.1002/HLCA.19390220151" >doi:10.1002/HLCA.19390220151</a> (1939). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3518',
    'Brockbank (2013)',
    'Brockbank, S. A.: Aqueous Henry&#x2019;s law constants, infinite dilution activity coefficients, and water solubility: critically evaluated database, experimental analysis, and prediction methods, Ph.D. thesis, Brigham Young University, USA, URL https://scholarsarchive.byu.edu/etd/3691/ (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2926',
    'Brockbank et&nbsp;al. (2013)',
    'Brockbank, S. A., Russon, J. L., Giles, N. F., Rowley, R. L., &#x0026; Wilding, W. V.: <span class="cmti-10">Infinite dilution activity coefficients and Henry&#x2019;s law constants</span> <span class="cmti-10">of compounds in water using the inert gas stripping method</span>, Fluid Phase Equilib., 348, 45–51, <a href="https://doi.org/10.1016/J.FLUID.2013.03.023" >doi:10.1016/J.FLUID.2013.03.023</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3698',
    'Brody et&nbsp;al. (1971)',
    'Brody, A. W., Lyons, K. P., Kurowski, J. L., McGill, J. J., &#x0026; Weaver, M. J.: <span class="cmti-10">Analysis and solubility of dimethyl and diethyl ether</span> <span class="cmti-10">in water, saline, oils, and blood</span>, J. Appl. Physiol., 31, 125–131, <a href="https://doi.org/10.1152/JAPPL.1971.31.1.125" >doi:10.1152/JAPPL.1971.31.1.125</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2981',
    'Brown and Wasik (1974)',
    'Brown, R. L. &#x0026; Wasik, S. P.: <span class="cmti-10">A method of</span> <span class="cmti-10">measuring the solubilities of hydrocarbons in aqueous solutions</span>, J. Res. Natl. Bureau Standards A: Phys. Chem., 78A, 453–460, <a href="https://doi.org/10.6028/JRES.078A.028" >doi:10.6028/JRES.078A.028</a> (1974). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3793',
    'Brownawell (1986)',
    'Brownawell, B. J.: The role of colloidal organic matter in the marine geochemistry of PCBs, Ph.D. thesis, Massachusetts Institute of Technology and the Woods Hole Oceanographic Institution, <a href="https://doi.org/10.1575/1912/3932" >doi:10.1575/1912/3932</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3508',
    'Bruneel et&nbsp;al. (2016)',
    'Bruneel, J., Walgraeve, C., Van Huffel, K., &#x0026; Van Langenhove, H.: <span class="cmti-10">Determination of the gas-to-liquid partitioning coefficients using a new</span> <span class="cmti-10">dynamic absorption method (DynAb method)</span>, Chem. Eng. J., 283, 544–552, <a href="https://doi.org/10.1016/J.CEJ.2015.07.053" >doi:10.1016/J.CEJ.2015.07.053</a> (2016). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1194',
    'Brunner et&nbsp;al. (1990)',
    'Brunner, S., Hornung, E., Santl, H., Wolff, E., Piringer, O. G., Altschuh, J., &#x0026; Brüggemann, R.: <span class="cmti-10">Henry&#x2019;s law constants</span> <span class="cmti-10">for polychlorinated biphenyls: Experimental determination and</span> <span class="cmti-10">structure-property relationships</span>, Environ. Sci. Technol., 24, 1751–1754, <a href="https://doi.org/10.1021/ES00081A021" >doi:10.1021/ES00081A021</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1775',
    'Bu and Warner (1995)',
    'Bu, X. &#x0026; Warner, M. J.: <span class="cmti-10">Solubility</span> <span class="cmti-10">of chlorofluorocarbon 113 in water and seawater</span>, Deep-Sea Res. I, 42, 1151–1161, <a href="https://doi.org/10.1016/0967-0637(95)00052-8" >doi:10.1016/0967-0637(95)00052-8</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3803',
    'Buck (1981)',
    'Buck, A. L.: <span class="cmti-10">New equations for computing vapor pressure</span> <span class="cmti-10">and enhancement factor</span>, J. Appl. Meteorol. Clim., 20, 1527–1532, <a href="https://doi.org/10.1175/1520-0450(1981)020<1527:NEFCVP>2.0.CO;2" >doi:10.1175/1520-0450(1981)020<1527:NEFCVP>2.0.CO;2</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3571',
    'Bullister and Wisegarver (1998)',
    'Bullister, J. L. &#x0026; Wisegarver, D. P.: <span class="cmti-10">The</span> <span class="cmti-10">solubility of carbon tetrachloride in water and seawater</span>, Deep-Sea Res. I, 45, 1285–1302, <a href="https://doi.org/10.1016/S0967-0637(98)00017-X" >doi:10.1016/S0967-0637(98)00017-X</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1776',
    'Bullister et&nbsp;al. (2002)',
    'Bullister, J. L., Wisegarver, D. P., &#x0026; Menzia, F. A.: <span class="cmti-10">The solubility of</span> <span class="cmti-10">sulfur hexafluoride in water and seawater</span>, Deep-Sea Res. I, 49, 175–187, <a href="https://doi.org/10.1016/S0967-0637(01)00051-6" >doi:10.1016/S0967-0637(01)00051-6</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3717',
    'Bunsen (1855a)',
    'Bunsen, R.: <span class="cmti-10">Ueber das Gesetz der Gasabsorption</span>, Liebigs Ann. Chem., 93, 1–50, <a href="https://doi.org/10.1002/JLAC.18550930102" >doi:10.1002/JLAC.18550930102</a> (1855a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3927',
    'Bunsen (1855b)',
    'Bunsen, R.: <span class="cmti-10">XV. On the law of absorption of gases</span>, Philos. Mag., 9, 116–130, <a href="https://doi.org/10.1080/14786445508641836" >doi:10.1080/14786445508641836</a> (1855b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3930',
    'Bunsen (1855c)',
    'Bunsen, R.: <span class="cmti-10">XXVII. On the law of absorption of gases</span>, Philos. Mag., 9, 181–201, <a href="https://doi.org/10.1080/14786445508641851" >doi:10.1080/14786445508641851</a> (1855c). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '850',
    'Burkhard et&nbsp;al. (1985)',
    'Burkhard, L. P., Armstrong, D. E., &#x0026; Andren, A. W.: <span class="cmti-10">Henry&#x2019;s law constants for</span> <span class="cmti-10">the polychlorinated biphenyls</span>, Environ. Sci. Technol., 19, 590–596, <a href="https://doi.org/10.1021/ES00137A002" >doi:10.1021/ES00137A002</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2968',
    'Burkhard and Guth (1981)',
    'Burkhard, N. &#x0026; Guth, J. A.: <span class="cmti-10">Rate of volatilisation of pesticides from soil surfaces; comparison</span> <span class="cmti-10">of calculated results with those determined in a laboratory model system</span>, Pestic. Sci., 12, 37–44, <a href="https://doi.org/10.1002/PS.2780120106" >doi:10.1002/PS.2780120106</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3245',
    'Burkholder et&nbsp;al. (2015)',
    'Burkholder, J. B., Sander, S. P., Abbatt, J., Barker, J. R., Huie, R. E., Kolb, C. E., Kurylo, M. J., Orkin, V. L., Wilmouth, D. M., &#x0026; Wine, P. H.: <span class="cmti-10">Chemical Kinetics and Photochemical Data for Use in</span> <span class="cmti-10">Atmospheric Studies, Evaluation No. 18</span>, JPL Publication 15-10, Jet Propulsion Laboratory, Pasadena, URL https://jpldataeval.jpl.nasa.gov (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3500',
    'Burkholder et&nbsp;al. (2019)',
    'Burkholder, J. B., Sander, S. P., Abbatt, J., Barker, J. R., Cappa, C., Crounse, J. D., Dibble, T. S., Huie, R. E., Kolb, C. E., Kurylo, M. J., Orkin, V. L., Percival, C. J., Wilmouth, D. M., &#x0026; Wine, P. H.: <span class="cmti-10">Chemical Kinetics and Photochemical Data for Use in Atmospheric</span> <span class="cmti-10">Studies, Evaluation No. 19</span>, JPL Publication 19-5, Jet Propulsion Laboratory, Pasadena, URL https://jpldataeval.jpl.nasa.gov (2019). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '496',
    'Burnett (1963)',
    'Burnett, M. G.: <span class="cmti-10">Determination of partition coefficients at</span> <span class="cmti-10">infinite dilution by the gas chromatographic analysis of the vapor above</span> <span class="cmti-10">dilute solutions</span>, Anal. Chem., 35, 1567–1570, <a href="https://doi.org/10.1021/AC60204A007" >doi:10.1021/AC60204A007</a> (1963). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '713',
    'Butler and Ramchandani (1935)',
    'Butler, J. A. V. &#x0026; Ramchandani, C. N.: <span class="cmti-10">The solubility of non-electrolytes. Part II. The influence of the polar</span> <span class="cmti-10">group on the free energy of hydration of aliphatic compounds</span>, J. Chem. Soc., pp. 952–955, <a href="https://doi.org/10.1039/JR9350000952" >doi:10.1039/JR9350000952</a> (1935). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '711',
    'Butler et&nbsp;al. (1933)',
    'Butler, J. A. V., Thomson, D. W., &#x0026; Maclennan, W. H.: <span class="cmti-10">The free</span> <span class="cmti-10">energy of the normal aliphatic alcohols in aqueous solution. Part I.</span> <span class="cmti-10">The partial vapour pressures of aqueous solutions of methyl, </span><span class="cmmi-10">n</span><span class="cmti-10">-propyl,</span> <span class="cmti-10">and </span><span class="cmmi-10">n</span><span class="cmti-10">-butyl alcohols. Part II. The solubilities of some normal aliphatic</span> <span class="cmti-10">alcohols in water. Part III. The theory of binary solutions, and its</span> <span class="cmti-10">application to aqueous-alcoholic solutions</span>, J. Chem. Soc., pp. 674–686, <a href="https://doi.org/10.1039/JR9330000674" >doi:10.1039/JR9330000674</a> (1933). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '712',
    'Butler et&nbsp;al. (1935)',
    'Butler, J. A. V., Ramchandani, C. N., &#x0026; Thomson, D. W.: <span class="cmti-10">The solubility of</span> <span class="cmti-10">non-electrolytes. Part I. The free energy of hydration of some aliphatic</span> <span class="cmti-10">alcohols</span>, J. Chem. Soc., pp. 280–285, <a href="https://doi.org/10.1039/JR9350000280" >doi:10.1039/JR9350000280</a> (1935). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3075',
    'Buttery et&nbsp;al. (1965)',
    'Buttery, R. G., Guadagni, D. G., &#x0026; Okano, S.: <span class="cmti-10">Air–water partition</span> <span class="cmti-10">coefficients of some aldehydes</span>, J. Sci. Food Agric., 16, 691–692, <a href="https://doi.org/10.1002/JSFA.2740161110" >doi:10.1002/JSFA.2740161110</a> (1965). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '754',
    'Buttery et&nbsp;al. (1969)',
    'Buttery, R. G., Ling, L. C., &#x0026; Guadagni, D. G.: <span class="cmti-10">Volatilities of aldehydes, ketones, and</span> <span class="cmti-10">esters in dilute water solutions</span>, J. Agric. Food Chem., 17, 385–389, <a href="https://doi.org/10.1021/JF60162A025" >doi:10.1021/JF60162A025</a> (1969). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '755',
    'Buttery et&nbsp;al. (1971)',
    'Buttery, R. G., Bomben, J. L., Guadagni, D. G., &#x0026; Ling, L. C.: <span class="cmti-10">Some considerations of</span> <span class="cmti-10">volatilities of organic flavor compounds in foods</span>, J. Agric. Food Chem., 19, 1045–1048, <a href="https://doi.org/10.1021/JF60178A004" >doi:10.1021/JF60178A004</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '722',
    'Cabani et&nbsp;al. (1971a)',
    'Cabani, S., Conti, G., &#x0026; Lepori, L.: <span class="cmti-10">Thermodynamic study on aqueous dilute solutions of organic</span> <span class="cmti-10">compounds. Part 1. – Cyclic amines</span>, Trans. Faraday Soc., 67, 1933–1942, <a href="https://doi.org/10.1039/TF9716701933" >doi:10.1039/TF9716701933</a> (1971a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '723',
    'Cabani et&nbsp;al. (1971b)',
    'Cabani, S., Conti, G., &#x0026; Lepori, L.: <span class="cmti-10">Thermodynamic study on aqueous dilute solutions of organic</span> <span class="cmti-10">compounds. Part 2. – Cyclic ethers</span>, Trans. Faraday Soc., 67, 1943–1950, <a href="https://doi.org/10.1039/TF9716701943" >doi:10.1039/TF9716701943</a> (1971b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2376',
    'Cabani et&nbsp;al. (1975a)',
    'Cabani, S., Conti, G., Giannessi, D., &#x0026; Lepori, L.: <span class="cmti-10">Thermodynamic</span> <span class="cmti-10">study of aqueous dilute solutions of organic compounds. Part 3. –</span> <span class="cmti-10">Morpholines and piperazines</span>, J. Chem. Soc. Faraday Trans. 1, 71, 1154–1160, <a href="https://doi.org/10.1039/F19757101154" >doi:10.1039/F19757101154</a> (1975a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2377',
    'Cabani et&nbsp;al. (1975b)',
    'Cabani, S., Conti, G., Mollica, V., &#x0026; Lepori, L.: <span class="cmti-10">Thermodynamic study of dilute aqueous</span> <span class="cmti-10">solutions of organic compounds. Part 4. – Cyclic and straight chain</span> <span class="cmti-10">secondary alcohols</span>, J. Chem. Soc. Faraday Trans. 1, 71, 1943–1952, <a href="https://doi.org/10.1039/F19757101943" >doi:10.1039/F19757101943</a> (1975b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2381',
    'Cabani et&nbsp;al. (1978)',
    'Cabani, S., Mollica, V., &#x0026; Lepori, L.: <span class="cmti-10">Thermodynamic study of dilute aqueous solutions of organic</span> <span class="cmti-10">compounds. Part 5. – Open-chain saturated bifunctional compounds</span>, J. Chem. Soc. Faraday Trans. 1, 74, 2667–2671, <a href="https://doi.org/10.1039/F19787402667" >doi:10.1039/F19787402667</a> (1978). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1910',
    'Cabani et&nbsp;al. (1981)',
    'Cabani, S., Gianni, P., Mollica, V., &#x0026; Lepori, L.: <span class="cmti-10">Group contributions to the thermodynamic</span> <span class="cmti-10">properties of non-ionic organic solutes in dilute aqueous solution</span>, J. Solution Chem., 10, 563–595, <a href="https://doi.org/10.1007/BF00646936" >doi:10.1007/BF00646936</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2203',
    'Cady and Misra (1974)',
    'Cady, G. H. &#x0026; Misra, S.: <span class="cmti-10">Hydrolysis of sulfuryl</span> <span class="cmti-10">fluoride</span>, Inorg. Chem., 13, 837–841, <a href="https://doi.org/10.1021/IC50134A016" >doi:10.1021/IC50134A016</a> (1974). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3548',
    'Cady et&nbsp;al. (1922)',
    'Cady, H. P., Elsey, H. M., &#x0026; Berger, E. V.: <span class="cmti-10">The solubility of helium in water</span>, J. Am. Chem. Soc., 44, 1456–1461, <a href="https://doi.org/10.1021/JA01428A009" >doi:10.1021/JA01428A009</a> (1922). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2977',
    'Calamari et&nbsp;al. (1991)',
    'Calamari, D., Bacci, E., Focardi, S., Gaggi, C., Morosini, M., &#x0026; Vighi, M.: <span class="cmti-10">Role of plant biomass in the global environmental partitioning</span> <span class="cmti-10">of chlorinated hydrocarbons</span>, Environ. Sci. Technol., 25, 1489–1495, <a href="https://doi.org/10.1021/ES00020A020" >doi:10.1021/ES00020A020</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3866',
    'Cargill (1990)',
    'Cargill, R. W.: IUPAC Solubility Data Series, Volume 43, Carbon Monoxide, Pergamon Press, Oxford, <a href="https://doi.org/10.1016/C2009-0-01228-8" >doi:10.1016/C2009-0-01228-8</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3708',
    'Carius (1855)',
    'Carius, L.: <span class="cmti-10">Absorptiometrische Untersuchungen</span>, Liebigs Ann. Chem., 94, 129–166, <a href="https://doi.org/10.1002/JLAC.18550940202" >doi:10.1002/JLAC.18550940202</a> (1855). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2976',
    'Caron et&nbsp;al. (1985)',
    'Caron, G., Suffet, I. H., &#x0026; Belton, T.: <span class="cmti-10">Effect of dissolved organic carbon on the environmental</span> <span class="cmti-10">distribution of nonpolar organic compounds</span>, Chemosphere, 14, 993–1000, <a href="https://doi.org/10.1016/0045-6535(85)90020-7" >doi:10.1016/0045-6535(85)90020-7</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '525',
    'Carpenter (1966)',
    'Carpenter, J. H.: <span class="cmti-10">New measurements of oxygen solubility</span> <span class="cmti-10">in pure and natural water</span>, Limnol. Oceanogr., 11, 264–277, <a href="https://doi.org/10.4319/LO.1966.11.2.0264" >doi:10.4319/LO.1966.11.2.0264</a> (1966). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '615',
    'Carroll and Mather (1989)',
    'Carroll, J. J. &#x0026; Mather, A. E.: <span class="cmti-10">The solubility of</span> <span class="cmti-10">hydrogen sulphide</span> <span class="cmti-10">in water from 0 to 90</span><span class="cmsy-7">∘</span>C <span class="cmti-10">and pressures to 1</span> MPa, Geochim. Cosmochim. Acta, 53, 1163–1170, <a href="https://doi.org/10.1016/0016-7037(89)90053-7" >doi:10.1016/0016-7037(89)90053-7</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '631',
    'Carroll et&nbsp;al. (1991)',
    'Carroll, J. J., Slupsky, J. D., &#x0026; Mather, A. E.: <span class="cmti-10">The solubility of carbon dioxide in water at low</span> <span class="cmti-10">pressure</span>, J. Phys. Chem. Ref. Data, 20, 1201–1209, <a href="https://doi.org/10.1063/1.555900" >doi:10.1063/1.555900</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3124',
    'Carroll et&nbsp;al. (1997)',
    'Carroll, J. J., Jou, F.-Y., &#x0026; Mather, A. E.: <span class="cmti-10">Fluid phase equilibria in the system </span><span class="cmmi-10">n</span><span class="cmti-10">-butane + water</span>, Fluid Phase Equilib., 140, 157–169, <a href="https://doi.org/10.1016/S0378-3812(97)00199-4" >doi:10.1016/S0378-3812(97)00199-4</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2840',
    'Carslaw et&nbsp;al. (1995)',
    'Carslaw, K. S., Clegg, S. L., &#x0026; Brimblecombe, P.: <span class="cmti-10">A thermodynamic model of the</span> <span class="cmti-10">system</span> HCl<span class="cmti-10">-</span>HNO<sub><span class="cmr-7">3</span></sub><span class="cmti-10">-</span>H<sub><span class="cmr-7">2</span></sub>SO<sub><span class="cmr-7">4</span></sub><span class="cmti-10">-</span>H<sub><span class="cmr-7">2</span></sub>O<span class="cmti-10">, including solubilities of</span> HBr<span class="cmti-10">, from </span><span class="cmmi-10">&#x003C;</span><span class="cmti-10">200</span> <span class="cmti-10">to 328 </span>K, J. Phys. Chem., 99, 11 557–11 574, <a href="https://doi.org/10.1021/J100029A039" >doi:10.1021/J100029A039</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '793',
    'Carter et&nbsp;al. (1968)',
    'Carter, G. B., McIver, M. C., &#x0026; Miller, G. J.: <span class="cmti-10">Evidence for the formation of a</span> <span class="cmti-10">hexahydrotriazine in the condensation of acetaldehyde with methylamine</span>, J. Chem. Soc. C, pp. 2591–2592, <a href="https://doi.org/10.1039/J39680002591" >doi:10.1039/J39680002591</a> (1968). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2468',
    'Cetin and Odabasi (2005)',
    'Cetin, B. &#x0026; Odabasi, M.: <span class="cmti-10">Measurement of Henry&#x2019;s</span> <span class="cmti-10">law constants of seven polybrominated diphenyl ether (PBDE) congeners</span> <span class="cmti-10">as a function of temperature</span>, Atmos. Environ., 39, 5273–5280, <a href="https://doi.org/10.1016/J.ATMOSENV.2005.05.029" >doi:10.1016/J.ATMOSENV.2005.05.029</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2814',
    'Cetin et&nbsp;al. (2006)',
    'Cetin, B., Ozer, S., Sofuoglu, A., &#x0026; Odabasi, M.: <span class="cmti-10">Determination of Henry&#x2019;s law</span> <span class="cmti-10">constants of organochlorine pesticides in deionized and saline water</span> <span class="cmti-10">as a function of temperature</span>, Atmos. Environ., 40, 4538–4546, <a href="https://doi.org/10.1016/J.ATMOSENV.2006.04.009" >doi:10.1016/J.ATMOSENV.2006.04.009</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2825',
    'Chai et&nbsp;al. (2005)',
    'Chai, X.-S., Falabella, J. B., &#x0026; Teja, A. S.: <span class="cmti-10">A relative headspace method for Henry&#x2019;s constants</span> <span class="cmti-10">of volatile organic compounds</span>, Fluid Phase Equilib., 231, 239–245, <a href="https://doi.org/10.1016/J.FLUID.2005.02.006" >doi:10.1016/J.FLUID.2005.02.006</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1073',
    'Chaintreau et&nbsp;al. (1995)',
    'Chaintreau, A., Grade, A., &#x0026; Muñoz-Box, R.: <span class="cmti-10">Determination of partition coefficients</span> <span class="cmti-10">and quantitation of headspace volatile compounds</span>, Anal. Chem., 67, 3300–3304, <a href="https://doi.org/10.1021/AC00114A029" >doi:10.1021/AC00114A029</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '87',
    'Chameides (1984)',
    'Chameides, W. L.: <span class="cmti-10">The</span> <span class="cmti-10">photochemistry of a remote marine stratiform cloud</span>, J. Geophys. Res., 89, 4739–4755, <a href="https://doi.org/10.1029/JD089ID03P04739" >doi:10.1029/JD089ID03P04739</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '88',
    'Chameides (1986)',
    'Chameides, W. L.: <span class="cmti-10">Reply</span>, J. Geophys. Res., 91, 14 571–14 572, <a href="https://doi.org/10.1029/JD091ID13P14571" >doi:10.1029/JD091ID13P14571</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '171',
    'Chameides and Stelson (1992)',
    'Chameides, W. L. &#x0026; Stelson, A. W.: <span class="cmti-10">Aqueous</span> <span class="cmti-10">phase chemical processes in deliquescent sea-salt aerosols: A mechanism</span> <span class="cmti-10">that couples the atmospheric cycles of </span>S <span class="cmti-10">and sea salt</span>, J. Geophys. Res., 97, 20 565–20 580, <a href="https://doi.org/10.1029/92JD01923" >doi:10.1029/92JD01923</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3097',
    'Chan et&nbsp;al. (2010)',
    'Chan, M. N., Surratt, J. D., Claeys, M., Edgerton, E. S., Tanner, R. L., Shaw, S. L., Zheng, M., Knipping, E. M., Eddingsaas, N. C., Wennberg, P. O., &#x0026; Seinfeld, J. H.: <span class="cmti-10">Characterization and quantification of</span> <span class="cmti-10">isoprene-derived epoxydiols in ambient aerosol in the southeastern United</span> <span class="cmti-10">States</span>, Environ. Sci. Technol., 44, 4590–4596, <a href="https://doi.org/10.1021/ES100596B" >doi:10.1021/ES100596B</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2628',
    'Chancel and Parmentier (1885)',
    'Chancel, G. &#x0026; Parmentier, F.: <span class="cmti-10">Sur la</span> <span class="cmti-10">solubilit</span><span class="cmti-10">é du sulfure de carbone et sur celle du chloroforme</span>, C. R. Hebd. Séances Acad. Sci., 100, 773–776 (1885). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2121',
    'Chang and Criddle (1995)',
    'Chang, W.-K. &#x0026; Criddle, C. S.: <span class="cmti-10">Biotransformation of HCFC-22, HCFC-142b, HCFC-123,</span> <span class="cmti-10">and HFC-134a by methanotrophic mixed culture MM1</span>, Biodegrad., 6, 1–9, <a href="https://doi.org/10.1007/BF00702293" >doi:10.1007/BF00702293</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3404',
    'Chao et&nbsp;al. (2017)',
    'Chao, H.-P., Lee, J.-F., &#x0026; Chiou, C. T.: <span class="cmti-10">Determination of the Henry&#x2019;s law constants of low-volatility</span> <span class="cmti-10">compounds via the measured air-phase transfer coefficients</span>, Wat. Res., 120, 238–244, <a href="https://doi.org/10.1016/J.WATRES.2017.04.074" >doi:10.1016/J.WATRES.2017.04.074</a> (2017). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3129',
    'Chapoy et&nbsp;al. (2004)',
    'Chapoy, A., Mokraoui, S., Valtz, A., Richon, D., Mohammadi, A. H., &#x0026; Tohidi, B.: <span class="cmti-10">Solubility measurement and modeling for the system</span> <span class="cmti-10">propane-water from 277.62 to 368.16K</span>, Fluid Phase Equilib., 226, 213–220, <a href="https://doi.org/10.1016/J.FLUID.2004.08.040" >doi:10.1016/J.FLUID.2004.08.040</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2629',
    'Chapoy et&nbsp;al. (2005)',
    'Chapoy, A., Mohammadi, A. H., Tohidi, B., Valtz, A., &#x0026; Richon, D.: <span class="cmti-10">Experimental measurement and phase behavior modeling of hydrogen</span> <span class="cmti-10">sulfide-water binary system</span>, Ind. Eng. Chem. Res., 44, 7567–7574, <a href="https://doi.org/10.1021/IE050201H" >doi:10.1021/IE050201H</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2496',
    'Charles and Destaillats (2005)',
    'Charles, M. J. &#x0026; Destaillats, H.: <span class="cmti-10">Experimental determinations of Henry&#x2019;s law constants of polybrominated</span> <span class="cmti-10">diphenyl ethers (PBDEs) to evaluate exposure to aquatic biota</span>, technical completion report, University of California Water Resources Center, UC Berkeley, URL https://escholarship.org/uc/item/9zv0s4np (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1137',
    'Chen et&nbsp;al. (1979)',
    'Chen, C.-C., Britt, H. I., Boston, J. F., &#x0026; Evans, L. B.: <span class="cmti-10">Extension and application of the Pitzer</span> <span class="cmti-10">equation for vapor-liquid equlibrium of aqueous electrolyte systems with</span> <span class="cmti-10">molecular solutes</span>, AIChE J., 25, 820–831, <a href="https://doi.org/10.1002/AIC.690250510" >doi:10.1002/AIC.690250510</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2824',
    'Chen et&nbsp;al. (2012)',
    'Chen, F., Freedman, D. L., Falta, R. W., &#x0026; Murdoch, L. C.: <span class="cmti-10">Henry&#x2019;s law</span> <span class="cmti-10">constants of chlorinated solvents at elevated temperatures</span>, Chemosphere, 86, 156–165, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2011.10.004" >doi:10.1016/J.CHEMOSPHERE.2011.10.004</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2808',
    'Chen et&nbsp;al. (2003)',
    'Chen, L., Takenaka, N., Bandow, H., &#x0026; Maeda, Y.: <span class="cmti-10">Henry&#x2019;s law constants for</span> C<sub><span class="cmr-7">2</span></sub><span class="cmti-10">-</span>C<sub><span class="cmr-7">3</span></sub> <span class="cmti-10">fluorinated alcohols and their wet deposition in the atmosphere</span>, Atmos. Environ., 37, 4817–4822, <a href="https://doi.org/10.1016/J.ATMOSENV.2003.08.002" >doi:10.1016/J.ATMOSENV.2003.08.002</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1856',
    'Cheng et&nbsp;al. (2003)',
    'Cheng, W.-H., Chu, F.-S., &#x0026; Liou, J.-J.: <span class="cmti-10">Air–water interface equilibrium</span> <span class="cmti-10">partitioning coefficients of aromatic hydrocarbons</span>, Atmos. Environ., 37, 4807–4815, <a href="https://doi.org/10.1016/J.ATMOSENV.2003.08.012" >doi:10.1016/J.ATMOSENV.2003.08.012</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2483',
    'Cheng et&nbsp;al. (2004)',
    'Cheng, W.-H., Chou, M.-S., Perng, C.-H., &#x0026; Chu, F.-S.: <span class="cmti-10">Determining the</span> <span class="cmti-10">equilibrium partitioning coefficients of</span> <span class="cmti-10">volatile organic compounds at an air–water interface</span>, Chemosphere, 54, 935–942, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2003.08.038" >doi:10.1016/J.CHEMOSPHERE.2003.08.038</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3076',
    'Chesters et&nbsp;al. (1989)',
    'Chesters, G., Simsiman, G. V., Levy, J., Alhajjar, B. J., Fathulla, R. N., &#x0026; Harkin, J. M.: <span class="cmti-10">Environmental fate of alachlor and metolachlor</span>, Rev. Environ. Contam. Toxicol., 110, 1–74, <a href="https://doi.org/10.1007/978-1-4684-7092-5_1" >doi:10.1007/978-1-4684-7092-5_1</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1502',
    'Cheung et&nbsp;al. (2000)',
    'Cheung, J. L., Li, Y. Q., Boniface, J., Shi, Q., Davidovits, P., Worsnop, D. R., Jayne, J. T., &#x0026; Kolb, C. E.: <span class="cmti-10">Heterogeneous interactions of</span> NO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">with aqueous surfaces</span>, J. Phys. Chem. A, 104, 2655–2662, <a href="https://doi.org/10.1021/JP992929F" >doi:10.1021/JP992929F</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1113',
    'Chiang et&nbsp;al. (1998)',
    'Chiang, P.-C., Hung, C.-H., Mar, J. C., &#x0026; Chang, E. E.: <span class="cmti-10">Henry&#x2019;s constants and mass transfer</span> <span class="cmti-10">coefficients of halogenated organic pollutants in an air stripping packed</span> <span class="cmti-10">column</span>, Wat. Sci. Tech., 38, 287–294 (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2979',
    'Chiou et&nbsp;al. (1980)',
    'Chiou, C. T., Freed, V. H., Peters, L. J., &#x0026; Kohnert, R. L.: <span class="cmti-10">Evaporation of solutes from</span> <span class="cmti-10">water</span>, Environ. Int., 3, 231–236, <a href="https://doi.org/10.1016/0160-4120(80)90123-3" >doi:10.1016/0160-4120(80)90123-3</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '721',
    'Christie and Crisp (1967)',
    'Christie, A. O. &#x0026; Crisp, D. J.: <span class="cmti-10">Activity coefficients on the </span><span class="cmmi-10">n</span><span class="cmti-10">-primary, secondary</span> <span class="cmti-10">and tertiary aliphatic amines in aqueous solution</span>, J. Appl. Chem., 17, 11–14, <a href="https://doi.org/10.1002/JCTB.5010170103" >doi:10.1002/JCTB.5010170103</a> (1967). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2821',
    'Cimetiere and de Laat (2009)',
    'Cimetiere, N. &#x0026; de Laat, J.: <span class="cmti-10">Henry&#x2019;s law</span> <span class="cmti-10">constant of N,N-dichloromethylamine: Application to the contamination</span> <span class="cmti-10">of the atmosphere of indoor swimming pools</span>, Chemosphere, 77, 465–470, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2009.07.056" >doi:10.1016/J.CHEMOSPHERE.2009.07.056</a> (2009). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3917',
    'Clarke and Glew (1971)',
    'Clarke, E. C. W. &#x0026; Glew, D. N.: <span class="cmti-10">Aqueous</span> <span class="cmti-10">nonelectrolyte solutions. Part VIII. Deuterium and hydrogen sulfides</span> <span class="cmti-10">solubilities in deuterium oxide and water</span>, Can. J. Chem., 49, 691–698, <a href="https://doi.org/10.1139/V71-116" >doi:10.1139/V71-116</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3658',
    'Clarke et&nbsp;al. (1962)',
    'Clarke, G. A., Williams, T. R., &#x0026; Taft, R. W.: <span class="cmti-10">A manometric determination of</span> <span class="cmti-10">the solvolysis rate of gaseous </span><span class="cmmi-10">t</span><span class="cmti-10">-butyl chloride in aqueous solution</span>, J. Am. Chem. Soc., 84, 2292–2295, <a href="https://doi.org/10.1021/JA00871A004" >doi:10.1021/JA00871A004</a> (1962). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1071',
    'Clegg and Brimblecombe (1986)',
    'Clegg, S. L. &#x0026; Brimblecombe, P.: <span class="cmti-10">The dissociation constant and Henry&#x2019;s law</span> <span class="cmti-10">constant of</span> HCl <span class="cmti-10">in aqueous solution</span>, Atmos. Environ., 20, 2483–2485, <a href="https://doi.org/10.1016/0004-6981(86)90079-X" >doi:10.1016/0004-6981(86)90079-X</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '620',
    'Clegg and Brimblecombe (1989)',
    'Clegg, S. L. &#x0026; Brimblecombe, P.: <span class="cmti-10">Solubility</span> <span class="cmti-10">of ammonia in pure aqueous and multicomponent solutions</span>, J. Phys. Chem., 93, 7237–7248, <a href="https://doi.org/10.1021/J100357A041" >doi:10.1021/J100357A041</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2839',
    'Clegg and Brimblecombe (1990)',
    'Clegg, S. L. &#x0026; Brimblecombe, P.: <span class="cmti-10">Equilibrium partial pressures and mean activity and osmotic coefficients</span> <span class="cmti-10">of 0–100% nitric acid as a function of temperature</span>, J. Phys. Chem., 94, 5369–5380, <a href="https://doi.org/10.1021/J100376A038" >doi:10.1021/J100376A038</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1925',
    'Clegg et&nbsp;al. (1996)',
    'Clegg, S. L., Brimblecombe, P., &#x0026; Khan, I.: <span class="cmti-10">The Henry&#x2019;s law constant of oxalic</span> <span class="cmti-10">acid and its partitioning into in the atmospheric aerosol</span>, Idöjárás, 100, 51–68 (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2838',
    'Clegg et&nbsp;al. (1998)',
    'Clegg, S. L., Brimblecombe, P., &#x0026; Wexler, A. S.: <span class="cmti-10">Thermodynamic model of</span> <span class="cmti-10">the system</span> H<sup><span class="cmr-7">+</span></sup><span class="cmti-10">-</span>NH<sub><span class="cmr-7">4</span></sub><sup><span class="cmr-7">+</span></sup><span class="cmti-10">-</span>SO<sub><span class="cmr-7">4</span></sub><sup><span class="cmr-7">2</span><span class="cmsy-7">−</span></sup><span class="cmti-10">-</span>NO<sub><span class="cmr-7">3</span></sub><sup><span class="cmsy-7">−</span></sup><span class="cmti-10">-</span>H<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">at tropospheric temperatures</span>, J. Phys. Chem. A, 102, 2137–2154, <a href="https://doi.org/10.1021/JP973042R" >doi:10.1021/JP973042R</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3849',
    'Clever (1979a)',
    'Clever, H. L.: IUPAC Solubility Data Series, Volume 1, Helium and Neon, Pergamon Press, Oxford, ISBN 0080223516 (1979a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3850',
    'Clever (1979b)',
    'Clever, H. L.: IUPAC Solubility Data Series, Volume 2, Krypton, Xenon and Radon, Pergamon Press, Oxford, ISBN 0080223524 (1979b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3851',
    'Clever (1980)',
    'Clever, H. L.: IUPAC Solubility Data Series, Volume 4, Argon, Pergamon Press, Oxford, ISBN 0080223532 (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3862',
    'Clever (1987)',
    'Clever, H. L.: IUPAC Solubility Data Series, Volume 29, Mercury in Liquids, Compressed Gases, Molten Salts and Other Elements, Pergamon Press, Oxford, <a href="https://doi.org/10.1016/C2009-0-01263-X" >doi:10.1016/C2009-0-01263-X</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3861',
    'Clever and Young (1987)',
    'Clever, H. L. &#x0026; Young, C. L.: IUPAC Solubility Data Series, Volume 27/28, Methane, Pergamon Press, Oxford, <a href="https://doi.org/10.1016/C2009-0-00752-1" >doi:10.1016/C2009-0-00752-1</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2279',
    'Clever et&nbsp;al. (1985)',
    'Clever, H. L., Johnson, S. A., &#x0026; Derrick, M. E.: <span class="cmti-10">The solubility of mercury and some sparingly</span> <span class="cmti-10">soluble mercury salts in water and aqueous-electrolyte solutions</span>, J. Phys. Chem. Ref. Data, 14, 631–681, <a href="https://doi.org/10.1063/1.555732" >doi:10.1063/1.555732</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2447',
    'Clever et&nbsp;al. (2005)',
    'Clever, H. L., Battino, R., Jaselskis, B., Yampol&#x2019;skii, Y. P., Jaselskis, B., Scharlin, P., &#x0026; Young, C. L.: <span class="cmti-10">IUPAC-NIST solubility data series.</span> <span class="cmti-10">80. Gaseous fluorides of boron, nitrogen, sulfur, carbon, and silicon and</span> <span class="cmti-10">solid xenon fluorides in all solvents</span>, J. Phys. Chem. Ref. Data, 34, 201–438, <a href="https://doi.org/10.1063/1.1794762" >doi:10.1063/1.1794762</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3410',
    'Clever et&nbsp;al. (2014)',
    'Clever, H. L., Battino, R., Miyamoto, H., Yampolski, Y., &#x0026; Young, C. L.: <span class="cmti-10">IUPAC-NIST solubility data series. 103. Oxygen and ozone in</span> <span class="cmti-10">water, aqueous solutions, and organic liquids (supplement to solubility</span> <span class="cmti-10">data series volume 7)</span>, J. Phys. Chem. Ref. Data, 43, 033 102, <a href="https://doi.org/10.1063/1.4883876" >doi:10.1063/1.4883876</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '730',
    'Cline and Bates (1983)',
    'Cline, J. D. &#x0026; Bates, T. S.: <span class="cmti-10">Dimethyl sulfide in the</span> <span class="cmti-10">equatorial Pacific Ocean: A natural source of sulfur to the atmosphere</span>, Geophys. Res. Lett., 10, 949–952, <a href="https://doi.org/10.1029/GL010I010P00949" >doi:10.1029/GL010I010P00949</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3022',
    'Compernolle and Müller (2014a)',
    'Compernolle, S. &#x0026; Müller, J.-F.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">law constants of diacids and hydroxy polyacids: recommended values</span>, Atmos. Chem. Phys., 14, 2699–2712, <a href="https://doi.org/10.5194/ACP-14-2699-2014" >doi:10.5194/ACP-14-2699-2014</a> (2014a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3067',
    'Compernolle and Müller (2014b)',
    'Compernolle, S. &#x0026; Müller, J.-F.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">law constants of polyols</span>, Atmos. Chem. Phys., 14, 12 815–12 837, <a href="https://doi.org/10.5194/ACP-14-12815-2014" >doi:10.5194/ACP-14-12815-2014</a> (2014b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2534',
    'Conway et&nbsp;al. (1983)',
    'Conway, R. A., Waggy, G. T., Spiegel, M. H., &#x0026; Berglund, R. L.: <span class="cmti-10">Environmental</span> <span class="cmti-10">fate and effects of ethylene oxide</span>, Environ. Sci. Technol., 17, 107–112, <a href="https://doi.org/10.1021/ES00108A009" >doi:10.1021/ES00108A009</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1148',
    'Cooling et&nbsp;al. (1992)',
    'Cooling, M. R., Khalfaoui, B., &#x0026; Newsham, D. M. T.: <span class="cmti-10">Phase equilibria</span> <span class="cmti-10">in very dilute mixtures of water and unsaturated chlorinated hydrocarbons</span> <span class="cmti-10">and of water and benzene</span>, Fluid Phase Equilib., 81, 217–229, <a href="https://doi.org/10.1016/0378-3812(92)85153-Y" >doi:10.1016/0378-3812(92)85153-Y</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3554',
    'Copolovici and Niinemets (2007)',
    'Copolovici, L. &#x0026; Niinemets, U.: <span class="cmti-10">Salting-in</span> <span class="cmti-10">and salting-out effects of ionic and neutral osmotica on limonene and</span> <span class="cmti-10">linalool Henry&#x2019;s law constants and octanol/water partition coefficients</span>, Chemosphere, 69, 621–629, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2007.02.066" >doi:10.1016/J.CHEMOSPHERE.2007.02.066</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3216',
    'Copolovici and Niinemets (2015)',
    'Copolovici, L. &#x0026; Niinemets, U.: <span class="cmti-10">Temperature dependencies of Henry&#x2019;s law</span> <span class="cmti-10">constants for different plant sesquiterpenes</span>, Chemosphere, 138, 751–757, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2015.07.075" >doi:10.1016/J.CHEMOSPHERE.2015.07.075</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2124',
    'Copolovici and Niinemets (2005)',
    'Copolovici, L. O. &#x0026; Niinemets, U.: <span class="cmti-10">Temperature dependencies of</span> <span class="cmti-10">Henry&#x2019;s law constants and octanol/water partition coefficients for</span> <span class="cmti-10">key plant volatile monoterpenoids</span>, Chemosphere, 61, 1390–1400, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2005.05.003" >doi:10.1016/J.CHEMOSPHERE.2005.05.003</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1929',
    'Coquelet and Richon (2005)',
    'Coquelet, C. &#x0026; Richon, D.: <span class="cmti-10">Measurement</span> <span class="cmti-10">of Henry&#x2019;s law constants and infinite dilution activity coefficients</span> <span class="cmti-10">of propyl mercaptan, butyl mercaptan, and dimethyl sulfide in</span> <span class="cmti-10">methyldiethanolamine (1) + water (2) with </span><span class="cmmi-10">w</span><sub><span class="cmr-7">1</span></sub> <span class="cmti-10">= 0.50 using</span> <span class="cmti-10">a gas stripping technique</span>, J. Chem. Eng. Data, 50, 2053–2057, <a href="https://doi.org/10.1021/JE050268B" >doi:10.1021/JE050268B</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3601',
    'Cosgrove and Walkley (1981)',
    'Cosgrove, B. A. &#x0026; Walkley, J.: <span class="cmti-10">Solubilities</span> <span class="cmti-10">of gases in</span> H<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">and</span> <sup><span class="cmr-7">2</span></sup>H<sub><span class="cmr-7">2</span></sub>O, J. Chromatogr., 216, 161–167, <a href="https://doi.org/10.1016/S0021-9673(00)82344-4" >doi:10.1016/S0021-9673(00)82344-4</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1995',
    'Cotham and Bidleman (1989)',
    'Cotham, W. E. &#x0026; Bidleman, T. F.: <span class="cmti-10">Degradation of malathion, endosulfan, and fenvalerate</span> <span class="cmti-10">in seawater and seawater/sediment microcosms</span>, J. Agric. Food Chem., 37, 824–828, <a href="https://doi.org/10.1021/JF00087A055" >doi:10.1021/JF00087A055</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2957',
    'Cousins and Mackay (2000)',
    'Cousins, I. &#x0026; Mackay, D.: <span class="cmti-10">Correlating the physical-chemical properties of</span> <span class="cmti-10">phthalate esters using the ‘three solubility&#x2019; approach</span>, Chemosphere, 41, 1389–1399, <a href="https://doi.org/10.1016/S0045-6535(00)00005-9" >doi:10.1016/S0045-6535(00)00005-9</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3628',
    'Cramer (1980)',
    'Cramer, S. D.: <span class="cmti-10">The solubility of oxygen in brines from</span> <span class="cmti-10">0 to 300</span><span class="cmsy-7">∘</span>C, Ind. Eng. Chem. Process Des. Dev., 19, 300–305, <a href="https://doi.org/10.1021/I260074A018" >doi:10.1021/I260074A018</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1153',
    'Crovetto (1991)',
    'Crovetto, R.: <span class="cmti-10">Evaluation of solubility data for the system</span> CO<sub><span class="cmr-7">2</span></sub><span class="cmti-10">-</span>H<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">from 273 </span>K <span class="cmti-10">to the critical point of water</span>, J. Phys. Chem. Ref. Data, 20, 575–589, <a href="https://doi.org/10.1063/1.555905" >doi:10.1063/1.555905</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2557',
    'Crovetto et&nbsp;al. (1982)',
    'Crovetto, R., Fernández-Prini, R., &#x0026; Japas, M. L.: <span class="cmti-10">Solubilities of inert gases and</span> <span class="cmti-10">methane in</span> H<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">and in</span> D<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">in the temperature range of 300 to 600 </span>K, J. Chem. Phys., 76, 1077–1086, <a href="https://doi.org/10.1063/1.443074" >doi:10.1063/1.443074</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3781',
    'Crozier and Yamamoto (1974)',
    'Crozier, T. E. &#x0026; Yamamoto, S.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">hydrogen in water, seawater and NaCl-solutions</span>, J. Chem. Eng. Data, 19, 242–244, <a href="https://doi.org/10.1021/JE60062A007" >doi:10.1021/JE60062A007</a> (1974). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3962',
    'Cruzeiro et&nbsp;al. (2022)',
    'Cruzeiro, V. W. D., Galib, M., Limmer, D. T., &#x0026; Götz, A. W.: <span class="cmti-10">Uptake of</span> N<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">5</span></sub> <span class="cmti-10">by</span> <span class="cmti-10">aqueous aerosol unveiled using chemically accurate many-body potentials</span>, Nature Commun., 13, 1266, <a href="https://doi.org/10.1038/S41467-022-28697-8" >doi:10.1038/S41467-022-28697-8</a> (2022). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3600',
    'da&nbsp;Silva et&nbsp;al. (1980)',
    'da Silva, A. M., Formosinho, S. J., &#x0026; Martins, C. T.: <span class="cmti-10">Gas chromatographic</span> <span class="cmti-10">determination of the solubility of gases in liquids at low pressures</span>, J. Chromatogr. Sci., 18, 180–182, <a href="https://doi.org/10.1093/CHROMSCI/18.4.180" >doi:10.1093/CHROMSCI/18.4.180</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '512',
    'Dacey et&nbsp;al. (1984)',
    'Dacey, J. W. H., Wakeham, S. G., &#x0026; Howes, B. L.: <span class="cmti-10">Henry&#x2019;s law constants</span> <span class="cmti-10">for dimethylsulfide in freshwater and seawater</span>, Geophys. Res. Lett., 11, 991–994, <a href="https://doi.org/10.1029/GL011I010P00991" >doi:10.1029/GL011I010P00991</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2944',
    'Dallos et&nbsp;al. (1983)',
    'Dallos, A., Ország, I., &#x0026; Ratkovics, F.: <span class="cmti-10">Liquid-liquid and vapour-liquid equilibrium data and</span> <span class="cmti-10">calculations for the system aniline + water in the presence of</span> NaCl<span class="cmti-10">,</span> NaI<span class="cmti-10">,</span> NH<sub><span class="cmr-7">4</span></sub>Cl <span class="cmti-10">and</span> NH<sub><span class="cmr-7">4</span></sub>I, Fluid Phase Equilib., 11, 91–102, <a href="https://doi.org/10.1016/0378-3812(83)85008-0" >doi:10.1016/0378-3812(83)85008-0</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '487',
    'Dasgupta and Dong (1986)',
    'Dasgupta, P. G. &#x0026; Dong, S.: <span class="cmti-10">Solubility of ammonia in liquid water and</span> <span class="cmti-10">generation of trace levels of standard gaseous ammonia</span>, Atmos. Environ., 20, 565–570, <a href="https://doi.org/10.1016/0004-6981(86)90099-5" >doi:10.1016/0004-6981(86)90099-5</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1573',
    'David et&nbsp;al. (2000)',
    'David, M. D., Fendinger, N. J., &#x0026; Hand, V. C.: <span class="cmti-10">Determination of Henry&#x2019;s law constants for</span> <span class="cmti-10">organosilicones in actual and simulated wastewater</span>, Environ. Sci. Technol., 34, 4554–4559, <a href="https://doi.org/10.1021/ES991204M" >doi:10.1021/ES991204M</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3573',
    'De Bruyn and Saltzman (1997)',
    'De Bruyn, W. J. &#x0026; Saltzman, E. S.: <span class="cmti-10">The</span> <span class="cmti-10">solubility of methyl bromide in pure water, 35permil sodium chloride and</span> <span class="cmti-10">seawater</span>, Mar. Chem., 56, 51–57, <a href="https://doi.org/10.1016/S0304-4203(96)00089-8" >doi:10.1016/S0304-4203(96)00089-8</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '389',
    'De Bruyn et&nbsp;al. (1994)',
    'De Bruyn, W. J., Shorter, J. A., Davidovits, P., Worsnop, D. R., Zahniser, M. S., &#x0026; Kolb, C. E.: <span class="cmti-10">Uptake of gas-phase sulfur species</span> <span class="cmti-10">methanesulfonic acid, dimethylsulfoxide, and dimethyl sulfone by aqueous</span> <span class="cmti-10">surfaces</span>, J. Geophys. Res., 99, 16 927–16 932, <a href="https://doi.org/10.1029/94JD00684" >doi:10.1029/94JD00684</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1125',
    'De Bruyn et&nbsp;al. (1995a)',
    'De Bruyn, W. J., Shorter, J. A., Davidovits, P., Worsnop, D. R., Zahniser, M. S., &#x0026; Kolb, C. E.: <span class="cmti-10">Uptake of haloacetyl and carbonyl</span> <span class="cmti-10">halides by water surfaces</span>, Environ. Sci. Technol., 29, 1179–1185, <a href="https://doi.org/10.1021/ES00005A007" >doi:10.1021/ES00005A007</a> (1995a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '586',
    'De Bruyn et&nbsp;al. (1995b)',
    'De Bruyn, W. J., Swartz, E., Hu, J. H., Shorter, J. A., Davidovits, P., Worsnop, D. R., Zahniser, M. S., &#x0026; Kolb, C. E.: <span class="cmti-10">Henry&#x2019;s law</span> <span class="cmti-10">solubilities and </span><span class="cmti-10">Śetchenow coefficients for biogenic reduced sulfur species</span> <span class="cmti-10">obtained from gas-liquid uptake measurements</span>, J. Geophys. Res., 100, 7245–7251, <a href="https://doi.org/10.1029/95JD00217" >doi:10.1029/95JD00217</a> (1995b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '981',
    'De Maagd et&nbsp;al. (1998)',
    'De Maagd, P. G.-J., Ten Hulscher, D. T. E. M., van den Heuvel, H., Opperhuizen, A., &#x0026; Sijm, D. T. H. M.: <span class="cmti-10">Physicochemical properties of</span> <span class="cmti-10">polycyclic aromatic hydrocarbons: Aqueous solubilities, </span><span class="cmmi-10">n</span><span class="cmti-10">-octanol/water</span> <span class="cmti-10">partition coefficients, and Henry&#x2019;s law constants</span>, Environ. Toxicol. Chem., 17, 251–257, <a href="https://doi.org/10.1002/ETC.5620170216" >doi:10.1002/ETC.5620170216</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2950',
    'de Wolf and Lieder (1998)',
    'de Wolf, W. &#x0026; Lieder, P. H.: <span class="cmti-10">A novel method</span> <span class="cmti-10">to determine uptake and elimination kinetics of volatile chemicals in</span> <span class="cmti-10">fish</span>, Chemosphere, 36, 1713–1724, <a href="https://doi.org/10.1016/S0045-6535(97)10062-5" >doi:10.1016/S0045-6535(97)10062-5</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3806',
    'Dean et&nbsp;al. (1973)',
    'Dean, C. R. S., Finch, A., &#x0026; Gardner, P. J.: <span class="cmti-10">The aqueous solubilities of nitrogen trifluoride and</span> <span class="cmti-10">dinitrogen tetrafluoride</span>, J. Chem. Soc. Dalton Trans., pp. 2722–2725, <a href="https://doi.org/10.1039/DT9730002722" >doi:10.1039/DT9730002722</a> (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '379',
    'Dean and Lange (1999)',
    'Dean, J. A. &#x0026; Lange, N. A.: Lange&#x2019;s Handbook of Chemistry, Fifteenth Edition, McGraw-Hill, Inc., ISBN 9780070163843 (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1935',
    'Delgado and Alderete (2002)',
    'Delgado, E. J. &#x0026; Alderete, J.: <span class="cmti-10">On the</span> <span class="cmti-10">calculation of Henry&#x2019;s law constants of chlorinated benzenes in water</span> <span class="cmti-10">from semiempirical quantum chemical methods</span>, J. Chem. Inf. Comput. Sci., 42, 559–563, <a href="https://doi.org/10.1021/CI0101206" >doi:10.1021/CI0101206</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1931',
    'Delgado and Alderete (2003)',
    'Delgado, E. J. &#x0026; Alderete, J. B.: <span class="cmti-10">Prediction</span> <span class="cmti-10">of Henry&#x2019;s law constants of triazine derived herbicides from quantum</span> <span class="cmti-10">chemical continuum solvation models</span>, J. Chem. Inf. Comput. Sci., 43, 1226–1230, <a href="https://doi.org/10.1021/CI0256485" >doi:10.1021/CI0256485</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2220',
    'Della Gatta et&nbsp;al. (1981)',
    'Della Gatta, G., Stradella, L., &#x0026; Venturello, P.: <span class="cmti-10">Enthalpies of solvation</span> <span class="cmti-10">in cyclohexane and in water for homologous aliphatic ketones and esters</span>, J. Solution Chem., 10, 209–220, <a href="https://doi.org/10.1007/BF00653098" >doi:10.1007/BF00653098</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '529',
    'Deno and Berkheimer (1960)',
    'Deno, N. C. &#x0026; Berkheimer, H. E.: <span class="cmti-10">Activity</span> <span class="cmti-10">coefficients as a functon of structure and media</span>, J. Chem. Eng. Data, 5, 1–5, <a href="https://doi.org/10.1021/JE60005A001" >doi:10.1021/JE60005A001</a> (1960). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2497',
    'Destaillats and Charles (2002)',
    'Destaillats, H. &#x0026; Charles, M. J.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">law constants of carbonyl-pentafluorobenzyl hydroxylamine (PFBHA)</span> <span class="cmti-10">derivatives in aqueous solution</span>, J. Chem. Eng. Data, 47, 1481–1487, <a href="https://doi.org/10.1021/JE025545I" >doi:10.1021/JE025545I</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '584',
    'Dewulf et&nbsp;al. (1995)',
    'Dewulf, J., Drijvers, D., &#x0026; van Langenhove, H.: <span class="cmti-10">Measurement of Henry&#x2019;s law</span> <span class="cmti-10">constant as function of temperature and salinity for the low temperature</span> <span class="cmti-10">range</span>, Atmos. Environ., 29, 323–331, <a href="https://doi.org/10.1016/1352-2310(94)00256-K" >doi:10.1016/1352-2310(94)00256-K</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1107',
    'Dewulf et&nbsp;al. (1999)',
    'Dewulf, J., van Langenhove, H., &#x0026; Everaert, P.: <span class="cmti-10">Determination of Henry&#x2019;s law</span> <span class="cmti-10">coefficients by combination of the equilibrium partitioning in closed</span> <span class="cmti-10">systems and solid-phase microextraction techniques</span>, J. Chromatogr. A, 830, 353–363, <a href="https://doi.org/10.1016/S0021-9673(98)00877-2" >doi:10.1016/S0021-9673(98)00877-2</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3118',
    'Diaz et&nbsp;al. (2005)',
    'Diaz, A., Ventura, F., &#x0026; Galceran, M. T.: <span class="cmti-10">Determination of odorous</span> <span class="cmti-10">mixed chloro-bromoanisoles in water by solid-phase micro-extraction and</span> <span class="cmti-10">gas chromatography-mass detection</span>, J. Chromatogr. A, 1064, 97–106, <a href="https://doi.org/10.1016/J.CHROMA.2004.12.027" >doi:10.1016/J.CHROMA.2004.12.027</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '545',
    'Dilling (1977)',
    'Dilling, W. L.: <span class="cmti-10">Interphase transfer processes. II. Evaporation</span> <span class="cmti-10">rates of chloro methanes, ethanes, ethylenes, propanes, and propylenes</span> <span class="cmti-10">from dilute aqueous solutions. Comparisons with theoretical predictions</span>, Environ. Sci. Technol., 11, 405–409, <a href="https://doi.org/10.1021/ES60127A009" >doi:10.1021/ES60127A009</a> (1977). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1916',
    'Dilling et&nbsp;al. (1975)',
    'Dilling, W. L., Tefertiller, N. B., &#x0026; Kallos, G. J.: <span class="cmti-10">Evaporation rates and reactivities of</span> <span class="cmti-10">methylene chloride, chloroform, 1,1,1-trichloroethane, trichloroethylene,</span> <span class="cmti-10">tetrachloroethylene, and other chlorinated compounds in dilute aqueous</span> <span class="cmti-10">solutions</span>, Environ. Sci. Technol., 9, 833–838, <a href="https://doi.org/10.1021/ES60107A008" >doi:10.1021/ES60107A008</a> (1975). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1018',
    'Disselkamp et&nbsp;al. (1999)',
    'Disselkamp, R. S., Chapman, E. G., Barchet, W. R., Colson, S. D., &#x0026; Howd, C. D.: BrCl <span class="cmti-10">production in</span> NaBr<span class="cmti-10">/</span>NaCl<span class="cmti-10">/</span>HNO<sub><span class="cmr-7">3</span></sub><span class="cmti-10">/</span>O<sub><span class="cmr-7">3</span></sub> <span class="cmti-10">solutions</span> <span class="cmti-10">representative of sea-salt aerosols in the marine boundary layer</span>, Geophys. Res. Lett., 26, 2183–2186, <a href="https://doi.org/10.1029/1999GL900251" >doi:10.1029/1999GL900251</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3638',
    'Djerki and Laub (1988)',
    'Djerki, R. A. &#x0026; Laub, R. J.: <span class="cmti-10">Solute retention in column liquid chromatography. X. Determination of</span> <span class="cmti-10">solute infinite-dilution activity coefficients in methanol, water, and their</span> <span class="cmti-10">mixtures, by combined gas-liquid and liquid-liquid chromatography</span>, J. Liq. Chromatogr., 11, 585–612, <a href="https://doi.org/10.1080/01483918808068333" >doi:10.1080/01483918808068333</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2936',
    'Dohányosová et&nbsp;al. (2004)',
    'Dohányosová, P., Sarraute, S., Dohnal, V., Majer, V., &#x0026; Costa Gomes, M.: <span class="cmti-10">Aqueous solubility and related thermodynamic functions of</span> <span class="cmti-10">nonaromatic hydrocarbons as a function of molecular structure</span>, Ind. Eng. Chem. Res., 43, 2805–2815, <a href="https://doi.org/10.1021/IE030800T" >doi:10.1021/IE030800T</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1907',
    'Dohnal and Fenclová (1995)',
    'Dohnal, V. &#x0026; Fenclová, D.: <span class="cmti-10">Air–water partitioning and aqueous solubility of phenols</span>, J. Chem. Eng. Data, 40, 478–483, <a href="https://doi.org/10.1021/JE00018A027" >doi:10.1021/JE00018A027</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2938',
    'Dohnal and Hovorka (1999)',
    'Dohnal, V. &#x0026; Hovorka, Š.: <span class="cmti-10">Exponential saturator:</span> <span class="cmti-10">a novel gas-liquid partitioning technique for measurement of large</span> <span class="cmti-10">limiting activity coefficients</span>, Ind. Eng. Chem. Res., 38, 2036–2043, <a href="https://doi.org/10.1021/IE980743H" >doi:10.1021/IE980743H</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3065',
    'Dohnal et&nbsp;al. (2006)',
    'Dohnal, V., Fenclová, D., &#x0026; Vrbka, P.: <span class="cmti-10">Temperature dependences of limiting activity coefficients,</span> <span class="cmti-10">Henry&#x2019;s law constants, and derivative infinite dilution properties of</span> <span class="cmti-10">lower (</span>C<sub><span class="cmr-7">1</span></sub><span class="cmti-10">-</span>C<sub><span class="cmr-7">5</span></sub><span class="cmti-10">) 1-alkanols in water. critical compilation, correlation,</span> <span class="cmti-10">and recommended data</span>, J. Phys. Chem. Ref. Data, 35, 1621–1651, <a href="https://doi.org/10.1063/1.2203355" >doi:10.1063/1.2203355</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3523',
    'Dohnal et&nbsp;al. (2010)',
    'Dohnal, V., Vrbka, P., Řehák, K., Böhme, A., &#x0026; Paschke, A.: <span class="cmti-10">Activity coefficients and partial molar excess enthalpies at infinite</span> <span class="cmti-10">dilution for four esters in water</span>, Fluid Phase Equilib., 295, 194–200, <a href="https://doi.org/10.1016/J.FLUID.2010.05.010" >doi:10.1016/J.FLUID.2010.05.010</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1337',
    'Donahue and Prinn (1993)',
    'Donahue, N. M. &#x0026; Prinn, R. G.: <span class="cmti-10">In situ</span> <span class="cmti-10">nonmethane hydrocarbon measurements on SAGA 3</span>, J. Geophys. Res., 98, 16 915–16 932, <a href="https://doi.org/10.1029/93JD01780" >doi:10.1029/93JD01780</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '485',
    'Dong and Dasgupta (1986)',
    'Dong, S. &#x0026; Dasgupta, P. G.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">gaseous formaldehyde in liquid water and generation of trace</span> <span class="cmti-10">standard gaseous formaldehyde</span>, Environ. Sci. Technol., 20, 637–640, <a href="https://doi.org/10.1021/ES00148A016" >doi:10.1021/ES00148A016</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '4017',
    'D&#x2019;Orazio and Wood (1963)',
    'D&#x2019;Orazio, L. A. &#x0026; Wood, R. H.: <span class="cmti-10">The thermodynamics of the solution of polar gases in</span> <span class="cmti-10">water; the heat, free energy, and entropy of solution of hydrazoic acid</span>, J. Phys. Chem., 67, 1435–1438, <a href="https://doi.org/10.1021/J100801A007" >doi:10.1021/J100801A007</a> (1963). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1911',
    'Douglas (1967)',
    'Douglas, E.: <span class="cmti-10">Carbon monoxide solubilities in sea water</span>, J. Phys. Chem., 71, 1931–1933, <a href="https://doi.org/10.1021/J100865A064" >doi:10.1021/J100865A064</a> (1967). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '982',
    'Drouillard et&nbsp;al. (1998)',
    'Drouillard, K. G., Tomy, G. T., Muir, D. C. G., &#x0026; Friesen, K. J.: <span class="cmti-10">Volatility of chlorinated </span><span class="cmmi-10">n</span><span class="cmti-10">-alkanes (</span>C<sub><span class="cmr-7">10</span></sub><span class="cmti-10">-</span>C<sub><span class="cmr-7">12</span></sub><span class="cmti-10">): Vapor pressures and</span> <span class="cmti-10">Henry&#x2019;s law constants</span>, Environ. Toxicol. Chem., 17, 1252–1260, <a href="https://doi.org/10.1002/ETC.5620170709" >doi:10.1002/ETC.5620170709</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3645',
    'Druaux et&nbsp;al. (1998)',
    'Druaux, C., Le Thanh, M., Seuvre, A.-M., &#x0026; Voilley, A.: <span class="cmti-10">Application of headspace</span> <span class="cmti-10">analysis to the study of aroma compounds-lipids interactions</span>, J. Am. Oil Chem. Soc., 75, 127–130, <a href="https://doi.org/10.1007/S11746-998-0022-Y" >doi:10.1007/S11746-998-0022-Y</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3426',
    'Du et&nbsp;al. (2017)',
    'Du, Y., Yuan, Y., &#x0026; Rochelle, G. T.: <span class="cmti-10">Volatility of amines for</span> CO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">capture</span>, Int. J. Greenhouse Gas Control, 58, 1–9, <a href="https://doi.org/10.1016/J.IJGGC.2017.01.001" >doi:10.1016/J.IJGGC.2017.01.001</a> (2017). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '759',
    'Dubik et&nbsp;al. (1987)',
    'Dubik, N. A., Titova, G. M., &#x0026; Loshakova, E. I.: <span class="cmti-10">Partition coefficients of bromine and</span> <span class="cmti-10">bromine chloride between air and natural brines</span>, Issled. v Obl. Poluch. Magniya, Ioda, Broma i ikh Soed., M., pp. 53–57, (in Russian, see also Chem. Abstr., 109, 213154j) (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3659',
    'Dubowski (1979)',
    'Dubowski, K. M.: <span class="cmti-10">Breath-alcohol simulators: Scientific</span> <span class="cmti-10">basis and actual performance</span>, J. Anal. Technol., 3, 177–182, <a href="https://doi.org/10.1093/JAT/3.5.177" >doi:10.1093/JAT/3.5.177</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3586',
    'Duchowicz et&nbsp;al. (2020)',
    'Duchowicz, P. R., Aranda, J. F., Bacelo, D. E., &#x0026; Fioressi, S. E.: <span class="cmti-10">QSPR</span> <span class="cmti-10">study of the Henry&#x2019;s law constant for heterogeneous compounds</span>, Chem. Eng. Res. Des., 154, 115–121, <a href="https://doi.org/10.1016/J.CHERD.2019.12.009" >doi:10.1016/J.CHERD.2019.12.009</a> (2020). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1140',
    'Dunnivant and Elzerman (1988)',
    'Dunnivant, F. M. &#x0026; Elzerman, A. W.: <span class="cmti-10">Aqueous solubility and Henry&#x2019;s law constant data for PCB congeners</span> <span class="cmti-10">for evaluation of quantitative structure-property relationships (QSPRs)</span>, Chemosphere, 17, 525–541, <a href="https://doi.org/10.1016/0045-6535(88)90028-8" >doi:10.1016/0045-6535(88)90028-8</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '643',
    'Dunnivant et&nbsp;al. (1988)',
    'Dunnivant, F. M., Coates, J. T., &#x0026; Elzerman, A. W.: <span class="cmti-10">Experimentally</span> <span class="cmti-10">determined Henry&#x2019;s law constants for 17 polychlorobiphenyl congeners</span>, Environ. Sci. Technol., 22, 448–453, <a href="https://doi.org/10.1021/ES00169A013" >doi:10.1021/ES00169A013</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2475',
    'Dunnivant et&nbsp;al. (1992)',
    'Dunnivant, F. M., Elzerman, A. W., Jurs, P. C., &#x0026; Hasan, M. N.: <span class="cmti-10">Quantitative</span> <span class="cmti-10">structure-property relationships for aqueous solubilities and Henry&#x2019;s</span> <span class="cmti-10">law constants of polychlorinated biphenyls</span>, Environ. Sci. Technol., 26, 1567–1573, <a href="https://doi.org/10.1021/ES00032A012" >doi:10.1021/ES00032A012</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3682',
    'Dupeux et&nbsp;al. (2022)',
    'Dupeux, T., Gaudin, T., Marteau-Roussy, C., Aubry, J.-M., &#x0026; Nardello-Rataj, V.: <span class="cmti-10">COSMO-RS as an effective tool for predicting the</span> <span class="cmti-10">physicochemical properties of fragrance raw materials</span>, Flavour Fragrance J., 37, 106–120, <a href="https://doi.org/10.1002/FFJ.3690" >doi:10.1002/FFJ.3690</a> (2022). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '492',
    'Durham et&nbsp;al. (1981)',
    'Durham, J. L., Overton, Jr., J. H., &#x0026; Aneja, V. P.: <span class="cmti-10">Influence of gaseous nitric acid on</span> <span class="cmti-10">sulfate production and acidity in rain</span>, Atmos. Environ., 15, 1059–1068, <a href="https://doi.org/10.1016/0004-6981(81)90106-2" >doi:10.1016/0004-6981(81)90106-2</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2478',
    'Eastcott et&nbsp;al. (1988)',
    'Eastcott, L., Shiu, W. Y., &#x0026; Mackay, D.: <span class="cmti-10">Environmentally relevant physical-chemical properties of</span> <span class="cmti-10">hydrocarbons: A review of data and development of simple correlations</span>, Oil Chem. Pollut., 4, 191–216, <a href="https://doi.org/10.1016/S0269-8579(88)80020-0" >doi:10.1016/S0269-8579(88)80020-0</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3769',
    'Easterbrook et&nbsp;al. (2023)',
    'Easterbrook, K. D., Vona, M. A., Nayebi-Astaneh, K., Miller, A. M., &#x0026; Osthoff, H. D.: <span class="cmti-10">Measurement of Henry&#x2019;s law and liquid-phase</span> <span class="cmti-10">loss rate constants of peroxypropionic nitric anhydride (PPN) in</span> <span class="cmti-10">deionized water and in n-octanol</span>, Atmos. Chem. Phys., 23, 311–322, <a href="https://doi.org/10.5194/ACP-23-311-2023" >doi:10.5194/ACP-23-311-2023</a> (2023). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3766',
    'Ebert et&nbsp;al. (2023)',
    'Ebert, R.-U., Kühne, R., &#x0026; Schüürmann, G.: <span class="cmti-10">Henry&#x2019;s law constant – a general-purpose fragment</span> <span class="cmti-10">model to predict log </span><span class="cmmi-10">K</span><sub><span class="cmr-7">aw</span></sub> <span class="cmti-10">from molecular structure</span>, Environ. Sci. Technol., 57, 160–167, <a href="https://doi.org/10.1021/ACS.EST.2C05623" >doi:10.1021/ACS.EST.2C05623</a> (2023). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3580',
    'Eddingsaas et&nbsp;al. (2010)',
    'Eddingsaas, N. C., VanderVelde, D. G., &#x0026; Wennberg, P. O.: <span class="cmti-10">Kinetics</span> <span class="cmti-10">and products of the acid-catalyzed ring-opening of atmospherically</span> <span class="cmti-10">relevant butyl epoxy alcohols</span>, J. Phys. Chem. A, 114, 8106–8113, <a href="https://doi.org/10.1021/JP103907C" >doi:10.1021/JP103907C</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2442',
    'Edelist et&nbsp;al. (1964)',
    'Edelist, G., Singer, M. M., &#x0026; Eger, E. I., I.: <span class="cmti-10">Solubility</span> <span class="cmti-10">coefficients of teflurane in various biological media</span>, Anesthesiology, 25, 223–225, <a href="https://doi.org/10.1097/00000542-196403000-00015" >doi:10.1097/00000542-196403000-00015</a> (1964). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3540',
    'Edwards et&nbsp;al. (1975)',
    'Edwards, T. J., Newman, J., &#x0026; Prausnitz, J. M.: <span class="cmti-10">Thermodynamics of</span> <span class="cmti-10">aqueous solutions containing volatile weak electrolytes</span>, AIChE J., 21, 248–259, <a href="https://doi.org/10.1002/AIC.690210205" >doi:10.1002/AIC.690210205</a> (1975). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '540',
    'Edwards et&nbsp;al. (1978)',
    'Edwards, T. J., Maurer, G., Newman, J., &#x0026; Prausnitz, J. M.: <span class="cmti-10">Vapor-liquid equilibria in multicomponent aqueous solutions of volatile</span> <span class="cmti-10">weak electrolytes</span>, AIChE J., 24, 966–976, <a href="https://doi.org/10.1002/AIC.690240605" >doi:10.1002/AIC.690240605</a> (1978). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3699',
    'Eger et&nbsp;al. (1963)',
    'Eger, E. I., Shargel, R., &#x0026; Merkel, G.: <span class="cmti-10">Solubility of diethyl ether in water, blood and oil</span>, Anesthesiology, 24, 676–678, <a href="https://doi.org/10.1097/00000542-196309000-00017" >doi:10.1097/00000542-196309000-00017</a> (1963). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3402',
    'Eger et&nbsp;al. (1999)',
    'Eger, II, E. I., Ionescu, P., Laster, M. J., Gong, D., Hudlicky, T., Kendig, J. J., Harris, R. A., Trudell, J. R., &#x0026; Pohorille, A.: <span class="cmti-10">Minimum alveolar anesthetic concentration of fluorinated alkanols in</span> <span class="cmti-10">rats: relevance to theories of narcosis</span>, Anesth. Analg., 88, 867–876, <a href="https://doi.org/10.1213/00000539-199904000-00035" >doi:10.1213/00000539-199904000-00035</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1149',
    'Eguchi et&nbsp;al. (1973)',
    'Eguchi, W., Adachi, M., &#x0026; Yoneda, M.: <span class="cmti-10">Dependency of partition equilibrium of iodine between air</span> <span class="cmti-10">and aqueous solution containing sodium hydroxide upon temperature and</span> <span class="cmti-10">concentration</span>, J. Chem. Eng. Jpn., 6, 389–396, <a href="https://doi.org/10.1252/JCEJ.6.389" >doi:10.1252/JCEJ.6.389</a> (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2631',
    'Elliott (1989)',
    'Elliott, S.: <span class="cmti-10">The solubility of carbon disulfide vapor in natural</span> <span class="cmti-10">aqueous systems</span>, Atmos. Environ., 23, 1977–1980, <a href="https://doi.org/10.1016/0004-6981(89)90523-4" >doi:10.1016/0004-6981(89)90523-4</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1342',
    'Elliott and Rowland (1993)',
    'Elliott, S. &#x0026; Rowland, F. S.: <span class="cmti-10">Nucleophilic</span> <span class="cmti-10">substitution rates and solubilities for methyl halides in seawater</span>, Geophys. Res. Lett., 20, 1043–1046, <a href="https://doi.org/10.1029/93GL01081" >doi:10.1029/93GL01081</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2597',
    'Emel&#x2019;yanenko et&nbsp;al. (2007)',
    'Emel&#x2019;yanenko, V. N., Dabrowska, A., Verevkin, S. P., Hertel, M. O., Scheuren, H., &#x0026; Sommer, K.: <span class="cmti-10">Vapor pressures, enthalpies of vaporization,</span> <span class="cmti-10">and limiting activity coefficients in water at 100</span><span class="cmsy-7">∘</span>C <span class="cmti-10">of 2-furanaldehyde,</span> <span class="cmti-10">benzaldehyde, phenylethanal, and 2-phenylethanol</span>, J. Chem. Eng. Data, 52, 468–471, <a href="https://doi.org/10.1021/JE060406C" >doi:10.1021/JE060406C</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2543',
    'English and Carroll (2001)',
    'English, N. J. &#x0026; Carroll, D. G.: <span class="cmti-10">Prediction of</span> <span class="cmti-10">Henry&#x2019;s law constants by a quantitative structure property relationship</span> <span class="cmti-10">and neural networks</span>, J. Chem. Inf. Comput. Sci., 41, 1150–1161, <a href="https://doi.org/10.1021/CI010361D" >doi:10.1021/CI010361D</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3700',
    'Ercan (1979)',
    'Ercan, M. T.: <span class="cmti-10">Solubility coefficients of</span> <sup><span class="cmr-7">133</span></sup>Xe <span class="cmti-10">in water,</span> <span class="cmti-10">saline, dog blood and organs</span>, Int. J. Appl. Radiat. Isot., 30, 757–759, <a href="https://doi.org/10.1016/0020-708X(79)90156-X" >doi:10.1016/0020-708X(79)90156-X</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2271',
    'Ervin et&nbsp;al. (1980)',
    'Ervin, A. L., Mangone, M. A., &#x0026; Singley, J. E.: <span class="cmti-10">Trace organics removal by air</span> <span class="cmti-10">stripping</span>, in: Proceedings of the Annual Conference of the American Water Works Association, pp. 507–530 (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '858',
    'Ettre et&nbsp;al. (1993)',
    'Ettre, L. S., Welter, C., &#x0026; Kolb, B.: <span class="cmti-10">Determination of gas-liquid partition coefficients by automatic</span> <span class="cmti-10">equilibrium headspace – gas chromatography utilizing the phase ratio</span> <span class="cmti-10">variation method</span>, Chromatographia, 35, 73–84, <a href="https://doi.org/10.1007/BF02278560" >doi:10.1007/BF02278560</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3115',
    'Falabella (2007)',
    'Falabella, J. B.: Air–water partitioning of volatile organic compounds and greenhouse gases in the presence of salts, Ph.D. thesis, Georgia Institute of Technology, USA, URL http://hdl.handle.net/1853/16221 (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3116',
    'Falabella and Teja (2008)',
    'Falabella, J. B. &#x0026; Teja, A. S.: <span class="cmti-10">Air–water partitioning of gasoline components in the presence of sodium</span> <span class="cmti-10">chloride</span>, Energy Fuels, 22, 398–401, <a href="https://doi.org/10.1021/EF700513K" >doi:10.1021/EF700513K</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3112',
    'Falabella et&nbsp;al. (2006)',
    'Falabella, J. B., Nair, A., &#x0026; Teja, A. S.: <span class="cmti-10">Henry&#x2019;s constants of 1-alkanols and 2-ketones in salt</span> <span class="cmti-10">solutions</span>, J. Chem. Eng. Data, 51, 1940–1945, <a href="https://doi.org/10.1021/JE0600956" >doi:10.1021/JE0600956</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2460',
    'Falk et&nbsp;al. (1990)',
    'Falk, A., Gullstrand, E., Löf, A., &#x0026; Wigaeus-Hjelm, E.: <span class="cmti-10">Liquid/air</span> <span class="cmti-10">partition coefficients of four terpenes</span>, Br. J. Ind. Med., 47, 62–64, <a href="https://doi.org/10.1136/OEM.47.1.62" >doi:10.1136/OEM.47.1.62</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2565',
    'Fang et&nbsp;al. (2006)',
    'Fang, F., Chu, S., &#x0026; Hong, C.-S.: <span class="cmti-10">Air–water Henry&#x2019;s law constants for PCB congeners: experimental</span> <span class="cmti-10">determination and modeling of structure-property relationship</span>, Anal. Chem., 78, 5412–5418, <a href="https://doi.org/10.1021/AC0604742" >doi:10.1021/AC0604742</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3506',
    'Feigenbrugel and Le Calvé (2021)',
    'Feigenbrugel, V. &#x0026; Le Calvé, S.: <span class="cmti-10">Temperature dependence of Henry&#x2019;s law</span> <span class="cmti-10">constants of fenpropidin and pyrimethanil: impact on their atmospheric</span> <span class="cmti-10">partitionnings and lifetimes</span>, J. Environ. Sci. Public Health, 5, 180–199, <a href="https://doi.org/10.26502/JESPH.96120124" >doi:10.26502/JESPH.96120124</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2818',
    'Feigenbrugel et&nbsp;al. (2004a)',
    'Feigenbrugel, V., Le Calvé, S., &#x0026; Mirabel, P.: <span class="cmti-10">Temperature dependence</span> <span class="cmti-10">of Henry&#x2019;s law constants of metolachlor and diazinon</span>, Chemosphere, 57, 319–327, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2004.05.013" >doi:10.1016/J.CHEMOSPHERE.2004.05.013</a> (2004a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2488',
    'Feigenbrugel et&nbsp;al. (2004b)',
    'Feigenbrugel, V., Le Calvé, S., Mirabel, P., &#x0026; Louis, F.: <span class="cmti-10">Henry&#x2019;s law constant measurements for phenol,</span> <span class="cmmi-10">o</span><span class="cmti-10">-, </span><span class="cmmi-10">m</span><span class="cmti-10">-, and </span><span class="cmmi-10">p</span><span class="cmti-10">-cresol as a function of temperature</span>, Atmos. Environ., 38, 5577–5588, <a href="https://doi.org/10.1016/J.ATMOSENV.2004.06.025" >doi:10.1016/J.ATMOSENV.2004.06.025</a> (2004b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3943',
    'Felder et&nbsp;al. (1986)',
    'Felder, J. D., Adams, W. J., &#x0026; Saeger, V. W.: <span class="cmti-10">Assessment of the safety of dioctyl adipate</span> <span class="cmti-10">in freshwater environments</span>, Environ. Toxicol. Chem., 5, 777–784, <a href="https://doi.org/10.1002/ETC.5620050809" >doi:10.1002/ETC.5620050809</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3077',
    'Feldhake and Stevens (1963)',
    'Feldhake, C. J. &#x0026; Stevens, C. D.: <span class="cmti-10">The</span> <span class="cmti-10">solubility of tetraethyllead in water</span>, J. Chem. Eng. Data, 8, 196–197, <a href="https://doi.org/10.1021/JE60017A016" >doi:10.1021/JE60017A016</a> (1963). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3527',
    'Fenclová et&nbsp;al. (2007)',
    'Fenclová, D., Dohnal, V., Vrbka, P., &#x0026; Laštovka, V.: <span class="cmti-10">Temperature dependence of</span> <span class="cmti-10">limiting activity coefficients, Henry&#x2019;s law constants, and related infinite</span> <span class="cmti-10">dilution properties of branched (C3 and C4) alkanols in water</span>, J. Chem. Eng. Data, 52, 989–1002, <a href="https://doi.org/10.1021/JE600567Z" >doi:10.1021/JE600567Z</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3633',
    'Fenclová et&nbsp;al. (2010)',
    'Fenclová, D., Dohnal, V., Vrbka, P., &#x0026; ehk, K.: <span class="cmti-10">Temperature dependence</span> <span class="cmti-10">of limiting activity coefficients, Henry&#x2019;s law constants, and related infinite</span> <span class="cmti-10">dilution properties of branched pentanols in water. Measurement, critical</span> <span class="cmti-10">compilation, correlation, and recommended data</span>, J. Chem. Eng. Data, 55, 3032–3043, <a href="https://doi.org/10.1021/JE901063S" >doi:10.1021/JE901063S</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3093',
    'Fenclová et&nbsp;al. (2014)',
    'Fenclová, D., Blahut, A., Vrbka, P., Dohnal, V., &#x0026; Böhme, A.: <span class="cmti-10">Temperature dependence of limiting activity coefficients, Henry&#x2019;s law</span> <span class="cmti-10">constants, and related infinite dilution properties of C4-C6 isomeric</span> <span class="cmmi-10">n</span><span class="cmti-10">-alkyl ethanoates/ethyl </span><span class="cmmi-10">n</span><span class="cmti-10">-alkanoates in water. Measurement, critical</span> <span class="cmti-10">compilation, correlation, and recommended data</span>, Fluid Phase Equilib., 375, 347–359, <a href="https://doi.org/10.1016/J.FLUID.2014.05.023" >doi:10.1016/J.FLUID.2014.05.023</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '900',
    'Fendinger and Glotfelty (1988)',
    'Fendinger, N. J. &#x0026; Glotfelty, D. E.: <span class="cmti-10">A</span> <span class="cmti-10">laboratory method for the experimental determination of air–water</span> <span class="cmti-10">Henry&#x2019;s law constants for several pesticides</span>, Environ. Sci. Technol., 22, 1289–1293, <a href="https://doi.org/10.1021/ES00176A007" >doi:10.1021/ES00176A007</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1144',
    'Fendinger and Glotfelty (1990)',
    'Fendinger, N. J. &#x0026; Glotfelty, D. E.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">law constants for selected pesticides, PAHs and PCBs</span>, Environ. Toxicol. Chem., 9, 731–735, <a href="https://doi.org/10.1002/ETC.5620090606" >doi:10.1002/ETC.5620090606</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '901',
    'Fendinger et&nbsp;al. (1989)',
    'Fendinger, N. J., Glotfelty, D. E., &#x0026; Freeman, H. P.: <span class="cmti-10">Comparison of two experimental</span> <span class="cmti-10">techniques for determining air/water Henry&#x2019;s law constants</span>, Environ. Sci. Technol., 23, 1528–1531, <a href="https://doi.org/10.1021/ES00070A013" >doi:10.1021/ES00070A013</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2105',
    'Fernández-Prini et&nbsp;al. (2003)',
    'Fernández-Prini, R., Alvarez, J. L., &#x0026; Harvey, A. H.: <span class="cmti-10">Henry&#x2019;s constants</span> <span class="cmti-10">and vapor-liquid distribution constants for gaseous solutes in</span> H<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">and</span> D<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">at high temperatures</span>, J. Phys. Chem. Ref. Data, 32, 903–916, <a href="https://doi.org/10.1063/1.1564818" >doi:10.1063/1.1564818</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2532',
    'Ferreira (2001)',
    'Ferreira, M. M. C.: <span class="cmti-10">Polycyclic aromatic hydrocarbons: a QSPR study</span>, Chemosphere, 44, 125–146, <a href="https://doi.org/10.1016/S0045-6535(00)00275-7" >doi:10.1016/S0045-6535(00)00275-7</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2462',
    'Fichan et&nbsp;al. (1999)',
    'Fichan, I., Larroche, C., &#x0026; Gros, J. B.: <span class="cmti-10">Water solubility, vapor pressure, and activity coefficients</span> <span class="cmti-10">of terpenes and terpenoids</span>, J. Chem. Eng. Data, 44, 56–62, <a href="https://doi.org/10.1021/JE980070+" >doi:10.1021/JE980070+</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '910',
    'Fickert (1998)',
    'Fickert, S.: Laboruntersuchungen zur Freisetzung photoreaktiver Halogenverbindungen aus Seesalzaerosol, Ph.D. thesis, Johannes Gutenberg-Universität, Mainz, Germany (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2647',
    'Fischer et&nbsp;al. (2004)',
    'Fischer, A., Müller, M., &#x0026; Klasmeier, J.: <span class="cmti-10">Determination of Henry&#x2019;s law constant for methyl</span> <span class="cmti-10">tert-butyl ether (MTBE) at groundwater temperatures</span>, Chemosphere, 54, 689–694, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2003.08.025" >doi:10.1016/J.CHEMOSPHERE.2003.08.025</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3594',
    'Fischer and Tropsch (1917)',
    'Fischer, F. &#x0026; Tropsch, H.: <span class="cmti-10">Notiz </span><span class="cmti-10">über Farbe</span> <span class="cmti-10">und Oxydationswert einiger Ozonl</span><span class="cmti-10">ösungen</span>, Ber. Dtsch. Chem. Ges., 50, 765–767, <a href="https://doi.org/10.1002/CBER.191705001124" >doi:10.1002/CBER.191705001124</a> (1917). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '978',
    'Fischer and Ballschmiter (1998a)',
    'Fischer, R. G. &#x0026; Ballschmiter, K.: <span class="cmti-10">Determination of vapor pressure, water solubility, gas-water partition</span> <span class="cmti-10">coefficient </span><span class="cmmi-10">P</span><sub><span class="cmr-7">GW</span></sub><span class="cmti-10">, Henry&#x2019;s law constant, and octanol-water partition</span> <span class="cmti-10">coefficient </span><span class="cmmi-10">P</span><sub><span class="cmr-7">OW</span></sub> <span class="cmti-10">of 26 alkyl dinitrates</span>, Chemosphere, 36, 2891–2901, <a href="https://doi.org/10.1016/S0045-6535(97)10246-6" >doi:10.1016/S0045-6535(97)10246-6</a> (1998a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '998',
    'Fischer and Ballschmiter (1998b)',
    'Fischer, R. G. &#x0026; Ballschmiter, K.: <span class="cmti-10">Prediction of the environmental distribution</span> <span class="cmti-10">of alkyl dinitrates – Chromatographic determination of vapor pressure</span> <span class="cmmi-10">p</span><sup><span class="cmr-7">0</span></sup><span class="cmti-10">, water solubility </span><span class="cmmi-10">S</span><sub><span class="cmr-7">H</span><sub><span class="cmr-5">2</span></sub><span class="cmr-7">O</span></sub><span class="cmti-10">, gas-water partition coefficient </span><span class="cmmi-10">K</span><sub><span class="cmr-7">GW</span></sub> <span class="cmti-10">(Henry&#x2019;s</span> <span class="cmti-10">law constant) and octanol-water partition coefficient </span><span class="cmmi-10">K</span><sub><span class="cmr-7">OW</span></sub>, Fresenius J. Anal. Chem., 360, 769–776, <a href="https://doi.org/10.1007/S002160050803" >doi:10.1007/S002160050803</a> (1998b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2959',
    'Fishbein and Albro (1972)',
    'Fishbein, L. &#x0026; Albro, P. W.: <span class="cmti-10">Chromatographic</span> <span class="cmti-10">and biological aspects of the phthalate esters</span>, J. Chromatogr. A, 70, 365–412, <a href="https://doi.org/10.1016/S0021-9673(00)92702-X" >doi:10.1016/S0021-9673(00)92702-X</a> (1972). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1943',
    'Fogg and Sangster (2003)',
    'Fogg, P. &#x0026; Sangster, J.: Chemicals in the Atmosphere: Solubility, Sources and Reactivity, John Wiley &#x0026; Sons, Inc., ISBN 978-0-471-98651-5 (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3863',
    'Fogg and Young (1988)',
    'Fogg, P. G. T. &#x0026; Young, C. L.: IUPAC Solubility Data Series, Volume 32, Hydrogen Sulfide, Deuterium Sulfide and Hydrogen Selenide, Pergamon Press, Oxford, <a href="https://doi.org/10.1016/C2009-0-00348-1" >doi:10.1016/C2009-0-00348-1</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3528',
    'Fogg et&nbsp;al. (2002)',
    'Fogg, P. G. T., Bligh, S.-W. A., Derrick, M. E., Yampol&#x2019;skii, Y. P., Clever, H. L., Skrzecz, A., Young, C. L., &#x0026; Fogg, P. G. T.: <span class="cmti-10">IUPAC-NIST</span> <span class="cmti-10">solubility data series. 76. Solubility of ethyne in liquids</span>, J. Phys. Chem. Ref. Data, 30, 1693–1876, <a href="https://doi.org/10.1063/1.1397768" >doi:10.1063/1.1397768</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'foster94',
    'Foster et&nbsp;al. (1994)',
    'Foster, P., Ferronato, C., &#x0026; Jacob, V.: <span class="cmti-10">Organic-compound transfer between gas-phase and raindrops</span> <span class="cmti-10">– 1st experiments in a simulation chamber</span>, Fresenius Environ. Bull., 3, 318–323 (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3716',
    'Fox (1909)',
    'Fox, C. J. J.: <span class="cmti-10">On the coefficients of absorption of nitrogen and</span> <span class="cmti-10">oxygen in distilled water and sea-water, and of atmospheric carbonic acid</span> <span class="cmti-10">in sea-water</span>, Trans. Faraday Soc., 5, 68–86, <a href="https://doi.org/10.1039/TF9090500068" >doi:10.1039/TF9090500068</a> (1909). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3300',
    'Franco et&nbsp;al. (2021)',
    'Franco, B., Blumenstock, T., Cho, C., Clarisse, L., Clerbaux, C., Coheur, P.-F., De Mazière, M., De Smedt, I., Dorn, H.-P., Emmerichs, T., Fuchs, H., Gkatzelis, G., Griffith, D. W. T., Gromov, S., Hannigan, J. W., Hase, F., Hohaus, T., Jones, N., Kerkweg, A., Kiendler-Scharr, A., Lutsch, E., Mahieu, E., Novelli, A., Ortega, I., Paton-Walsh, C., Pommier, M., Pozzer, A., Reimer, D., Rosanka, S., Sander, R., Schneider, M., Strong, K., Tillmann, R., Van Roozendael, M., Vereecken, L., Vigouroux, C., Wahner, A., &#x0026; Taraborrelli, D.: <span class="cmti-10">Ubiquitous atmospheric</span> <span class="cmti-10">production of organic acids mediated by cloud droplets</span>, Nature, 593, 233–237, <a href="https://doi.org/10.1038/S41586-021-03462-X" >doi:10.1038/S41586-021-03462-X</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1913',
    'Fredenhagen and Liebster (1932)',
    'Fredenhagen, K. &#x0026; Liebster, H.: <span class="cmti-10">Die Teildrucke und Verteilungszahlen der Essigs</span><span class="cmti-10">äure </span><span class="cmti-10">über</span> <span class="cmti-10">ihren w</span><span class="cmti-10">ässerigen L</span><span class="cmti-10">ösungen bei 25</span><span class="cmsy-7">∘</span>C, Z. Phys. Chem., 162A, 449–453, <a href="https://doi.org/10.1515/ZPCH-1932-16234" >doi:10.1515/ZPCH-1932-16234</a> (1932). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1914',
    'Fredenhagen and Wellmann (1932a)',
    'Fredenhagen, K. &#x0026; Wellmann, M.: <span class="cmti-10">Verteilungszahlen des Fluorwasserstoffs </span><span class="cmti-10">über</span> <span class="cmti-10">dem Zweistoffsystem [</span>H<sub><span class="cmr-7">2</span></sub>O<span class="cmti-10">-</span>HF<span class="cmti-10">] bei 25</span><span class="cmsy-7">∘</span>C <span class="cmti-10">und die Siedepunktskurve</span> <span class="cmti-10">dieses Systems bei Atmosph</span><span class="cmti-10">ärendruck</span>, Z. Phys. Chem., 162A, 454–466, <a href="https://doi.org/10.1515/ZPCH-1932-16235" >doi:10.1515/ZPCH-1932-16235</a> (1932a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1915',
    'Fredenhagen and Wellmann (1932b)',
    'Fredenhagen, K. &#x0026; Wellmann, M.: <span class="cmti-10">Verteilungszahlen des Cyanwasserstoffs und des Wassers </span><span class="cmti-10">über dem</span> <span class="cmti-10">Zweistoffsystem [</span>H<sub><span class="cmr-7">2</span></sub>O<span class="cmti-10">-</span>HCN<span class="cmti-10">] bei 18</span><span class="cmsy-7">∘</span>C, Z. Phys. Chem., 162A, 467–470, <a href="https://doi.org/10.1515/ZPCH-1932-16236" >doi:10.1515/ZPCH-1932-16236</a> (1932b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '937',
    'Frenzel et&nbsp;al. (1998)',
    'Frenzel, A., Scheer, V., Sikorski, R., George, C., Behnke, W., &#x0026; Zetzsch, C.: <span class="cmti-10">Heterogeneous interconversion reactions of</span> BrNO<sub><span class="cmr-7">2</span></sub><span class="cmti-10">,</span> ClNO<sub><span class="cmr-7">2</span></sub><span class="cmti-10">,</span> Br<sub><span class="cmr-7">2</span></sub><span class="cmti-10">, and</span> Cl<sub><span class="cmr-7">2</span></sub>, J. Phys. Chem. A, 102, 1329–1337, <a href="https://doi.org/10.1021/JP973044B" >doi:10.1021/JP973044B</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3531',
    'Frenzel et&nbsp;al. (2000)',
    'Frenzel, A., Kutsuna, S., Takeuchi, K., &#x0026; Ibusuki, T.: <span class="cmti-10">Solubility</span> <span class="cmti-10">and reactivity of peroxyacetyl nitrate (PAN) in dilute aqueous salt</span> <span class="cmti-10">solutions and in sulphuric acid</span>, Atmos. Environ., 34, 3641–3644, <a href="https://doi.org/10.1016/S1352-2310(00)00132-1" >doi:10.1016/S1352-2310(00)00132-1</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '729',
    'Friant and Suffet (1979)',
    'Friant, S. L. &#x0026; Suffet, I. H.: <span class="cmti-10">Interactive effects of</span> <span class="cmti-10">temperature, salt concentration, and pH on head space analysis for</span> <span class="cmti-10">isolating volatile trace organics in aqueous environmental samples</span>, Anal. Chem., 51, 2167–2172, <a href="https://doi.org/10.1021/AC50049A027" >doi:10.1021/AC50049A027</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '514',
    'Fried et&nbsp;al. (1994)',
    'Fried, A., Henry, B. E., Calvert, J. G., &#x0026; Mozurkewich, M.: <span class="cmti-10">The reaction probability</span> <span class="cmti-10">of</span> N<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">5</span></sub> <span class="cmti-10">with sulfuric acid aerosols at stratospheric temperatures and</span> <span class="cmti-10">compositions</span>, J. Geophys. Res., 99, 3517–3532, <a href="https://doi.org/10.1029/93JD01907" >doi:10.1029/93JD01907</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3801',
    'Friedman (1954)',
    'Friedman, H. L.: <span class="cmti-10">The solubilities of sulfur hexafluoride</span> <span class="cmti-10">in water and of the rare gases, sulfur hexafluoride and osmium</span> <span class="cmti-10">tetroxide in nitromethane</span>, J. Am. Chem. Soc., 76, 3294–3297, <a href="https://doi.org/10.1021/JA01641A065" >doi:10.1021/JA01641A065</a> (1954). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2945',
    'Friesen et&nbsp;al. (1993)',
    'Friesen, K. J., Loewen, M. D., Fairchild, W. L., Lawrence, S. G., Holoka, M. H., &#x0026; Muir, D. C. G.: <span class="cmti-10">Evidence for particle-mediated</span> <span class="cmti-10">transport of 2,3,7,8-tetrachlorodibenzofuran during gas sparging</span> <span class="cmti-10">of natural water</span>, Environ. Toxicol. Chem., 12, 2037–2044, <a href="https://doi.org/10.1002/ETC.5620121110" >doi:10.1002/ETC.5620121110</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3033',
    'Fu et&nbsp;al. (2013)',
    'Fu, M., Yu, Z., Lu, G., &#x0026; Song, X.: <span class="cmti-10">Henry&#x2019;s law constant for phosphine in seawater: determination and</span> <span class="cmti-10">assessment of influencing factors</span>, Chin. J. Oceanol. Limnol., 31, 860–866, <a href="https://doi.org/10.1007/S00343-013-2212-1" >doi:10.1007/S00343-013-2212-1</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3693',
    'Fukuchi et&nbsp;al. (2002)',
    'Fukuchi, K., Miyoshi, K., Watanabe, T., Yonezawa, S., &#x0026; Arai, Y.: <span class="cmti-10">Measurement and correlation of infinite dilution activity coefficients of</span> <span class="cmti-10">alkanol or ether in aqueous solution</span>, Fluid Phase Equilib., 194-197, 937–945, <a href="https://doi.org/10.1016/S0378-3812(01)00675-6" >doi:10.1016/S0378-3812(01)00675-6</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3701',
    'Gabel and Schultz (1973)',
    'Gabel, R. A. &#x0026; Schultz, B.: <span class="cmti-10">Solubility of nitrous oxide in water, 20–80 C</span>, Anesthesiology, 38, 75–81, <a href="https://doi.org/10.1097/00000542-197301000-00019" >doi:10.1097/00000542-197301000-00019</a> (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'gaffney84',
    'Gaffney and Senum (1984)',
    'Gaffney, J. S. &#x0026; Senum, G. I.: <span class="cmti-10">Peroxides,</span> <span class="cmti-10">peracids, aldehydes, and PANs and their links to natural and</span> <span class="cmti-10">anthropogenic organic sources</span>, in: Gas-Liquid Chemistry of Natural Waters, edited by Newman, L., pp. 5–1–5–7, NTIS TIC-4500, UC-11, BNL 51757 Brookhaven National Laboratory (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '715',
    'Gaffney et&nbsp;al. (1987)',
    'Gaffney, J. S., Streit, G. E., Spall, W. D., &#x0026; Hall, J. H.: <span class="cmti-10">Beyond acid rain. Do soluble oxidants</span> <span class="cmti-10">and organic toxins interact with</span> SO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">and</span> NO<sub><span class="cmr-7">x</span></sub> <span class="cmti-10">to increase ecosystem</span> <span class="cmti-10">effects?</span>, Environ. Sci. Technol., 21, 519–524, <a href="https://doi.org/10.1021/ES00160A001" >doi:10.1021/ES00160A001</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3963',
    'Galib and Limmer (2021)',
    'Galib, M. &#x0026; Limmer, D. T.: <span class="cmti-10">Reactive uptake of</span> N<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">5</span></sub> <span class="cmti-10">by atmospheric aerosol is dominated by interfacial processes</span>, Science, 371, 921–925, <a href="https://doi.org/10.1126/SCIENCE.ABD7716" >doi:10.1126/SCIENCE.ABD7716</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '892',
    'Gan and Yates (1996)',
    'Gan, J. &#x0026; Yates, S. R.: <span class="cmti-10">Degradation and phase</span> <span class="cmti-10">partition of methyl iodide in soil</span>, J. Agric. Food Chem., 44, 4001–4008, <a href="https://doi.org/10.1021/JF960413C" >doi:10.1021/JF960413C</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2902',
    'Garbarini and Lion (1985)',
    'Garbarini, D. R. &#x0026; Lion, L. W.: <span class="cmti-10">Evaluation</span> <span class="cmti-10">of sorptive partitioning of nonionic pollutants in closed systems</span> <span class="cmti-10">by headspace analysis</span>, Environ. Sci. Technol., 19, 1122–1128, <a href="https://doi.org/10.1021/ES00141A018" >doi:10.1021/ES00141A018</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3748',
    'Gatterer (1926)',
    'Gatterer, A.: <span class="cmti-10">XXXIX.—The absorption of gases by colloidal solutions</span>, J. Chem. Soc., 129, 299–316, <a href="https://doi.org/10.1039/JR9262900299" >doi:10.1039/JR9262900299</a> (1926). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2438',
    'Gautier et&nbsp;al. (2003)',
    'Gautier, C., Le Calvé, S., &#x0026; Mirabel, P.: <span class="cmti-10">Henry&#x2019;s law constants measurements of alachlor and</span> <span class="cmti-10">dichlorvos between 283 and 298 </span>K, Atmos. Environ., 37, 2347–2353, <a href="https://doi.org/10.1016/S1352-2310(03)00155-9" >doi:10.1016/S1352-2310(03)00155-9</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3711',
    'Geffcken (1904)',
    'Geffcken, G.: <span class="cmti-10">Beitr</span><span class="cmti-10">äge zur Kenntnis der L</span><span class="cmti-10">öslichkeitsbeeinflussung</span>, Z. Phys. Chem., 49, 257–302, <a href="https://doi.org/10.1515/ZPCH-1904-4925" >doi:10.1515/ZPCH-1904-4925</a> (1904). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'george93',
    'George et&nbsp;al. (1993)',
    'George, C., Ponche, J. L., &#x0026; Mirabel, P.: <span class="cmti-10">Experimental determination of uptake coefficients for acid</span> <span class="cmti-10">halides</span>, in: Proceedings of Workshop on STEP-HALOCSIDE, AFEAS, Dublin, 23–25 March (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1126',
    'George et&nbsp;al. (1994a)',
    'George, C., Lagrange, J., Lagrange, P., Mirabel, P., Pallares, C., &#x0026; Ponche, J. L.: <span class="cmti-10">Heterogeneous chemistry of trichloroacetyl chloride in the</span> <span class="cmti-10">atmosphere</span>, J. Geophys. Res., 99, 1255–1262, <a href="https://doi.org/10.1029/93JD02915" >doi:10.1029/93JD02915</a> (1994a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1127',
    'George et&nbsp;al. (1994b)',
    'George, C., Saison, J. Y., Ponche, J. L., &#x0026; Mirabel, P.: <span class="cmti-10">Kinetics of mass</span> <span class="cmti-10">transfer of carbonyl fluoride, trifluoroacetyl fluoride, and trifluoroacetyl</span> <span class="cmti-10">chloride at the air/water interface</span>, J. Phys. Chem., 98, 10 857–10 862, <a href="https://doi.org/10.1021/J100093A029" >doi:10.1021/J100093A029</a> (1994b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2512',
    'Gershenzon et&nbsp;al. (2001)',
    'Gershenzon, M., Davidovits, P., Jayne, J. T., Kolb, C. E., &#x0026; Worsnop, D. R.: <span class="cmti-10">Simultaneous uptake of DMS and ozone on water</span>, J. Phys. Chem. A, 105, 7031–7036, <a href="https://doi.org/10.1021/JP010696Y" >doi:10.1021/JP010696Y</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3579',
    'Gharagheizi et&nbsp;al. (2010)',
    'Gharagheizi, F., Abbasi, R., &#x0026; Tirandazi, B.: <span class="cmti-10">Prediction of Henry&#x2019;s law constant of</span> <span class="cmti-10">organic compounds in water from a new group-contribution-based model</span>, Ind. Eng. Chem. Res., 49, 10 149–10 152, <a href="https://doi.org/10.1021/IE101532E" >doi:10.1021/IE101532E</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3406',
    'Gharagheizi et&nbsp;al. (2012)',
    'Gharagheizi, F., Eslamimanesh, A., Mohammadi, A. H., &#x0026; Richon, D.: <span class="cmti-10">Empirical method for estimation of Henry&#x2019;s law constant of</span> <span class="cmti-10">non-electrolyte organic compounds in water</span>, J. Chem. Thermodyn., 47, 295–299, <a href="https://doi.org/10.1016/J.JCT.2011.11.015" >doi:10.1016/J.JCT.2011.11.015</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3078',
    'Giardino et&nbsp;al. (1988)',
    'Giardino, N. J., Andelman, J. B., Borrazzo, J. E., &#x0026; Davidson, C. I.: <span class="cmti-10">Sulfur hexafluoride as a surrogate for volatilization of organics</span> <span class="cmti-10">from indoor water uses</span>, J. Air Pollut. Control Assoc., 38, 278–279, <a href="https://doi.org/10.1080/08940630.1988.10466379" >doi:10.1080/08940630.1988.10466379</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2014',
    'Gibbs et&nbsp;al. (1991)',
    'Gibbs, P., Radzicka, A., &#x0026; Wolfenden, R.: <span class="cmti-10">The anomalous hydrophilic character of proline</span>, J. Am. Chem. Soc., 113, 4714–4715, <a href="https://doi.org/10.1021/JA00012A068" >doi:10.1021/JA00012A068</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2224',
    'Gill et&nbsp;al. (1976)',
    'Gill, S. J., Nichols, N. F., &#x0026; Wadsö, I.: <span class="cmti-10">Calorimetric determination of enthalpies of solution of slightly</span> <span class="cmti-10">soluble liquids II. Enthalpy of solution of some hydrocarbons in water and</span> <span class="cmti-10">their use in establishing the temperature dependence of their solubilities</span>, J. Chem. Thermodyn., 8, 445–452, <a href="https://doi.org/10.1016/0021-9614(76)90065-3" >doi:10.1016/0021-9614(76)90065-3</a> (1976). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'giona69',
    'Giona et&nbsp;al. (1969)',
    'Giona, A. R., Passino, A., &#x0026; Turriziani, R.: <span class="cmti-10">Assorbimento del bromo in acqua</span>, La Ricerca Scientifica, 30, 2047–2056 (1969). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2578',
    'Glew and Hames (1971)',
    'Glew, D. N. &#x0026; Hames, D. A.: <span class="cmti-10">Aqueous nonelectrolyte solutions. Part X. Mercury solubility in</span> <span class="cmti-10">water</span>, Can. J. Chem., 49, 3114–3118, <a href="https://doi.org/10.1139/V71-520" >doi:10.1139/V71-520</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2222',
    'Glew and Moelwyn-Hughes (1953)',
    'Glew, D. N. &#x0026; Moelwyn-Hughes, E. A.: <span class="cmti-10">Chemical statics of the methyl halides in water</span>, Discuss. Faraday Soc., 15, 150–161, <a href="https://doi.org/10.1039/DF9531500150" >doi:10.1039/DF9531500150</a> (1953). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2452',
    'Glotfelty et&nbsp;al. (1987)',
    'Glotfelty, D. E., Seiber, J. N., &#x0026; Liljedahl, A.: <span class="cmti-10">Pesticides in fog</span>, Nature, 325, 602–605, <a href="https://doi.org/10.1038/325602A0" >doi:10.1038/325602A0</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '752',
    'Gmitro and Vermeulen (1964)',
    'Gmitro, J. I. &#x0026; Vermeulen, T.: <span class="cmti-10">Vapor-liquid</span> <span class="cmti-10">equilibria for aqueous sulfuric acid</span>, AIChE J., 10, 740–746, <a href="https://doi.org/10.1002/AIC.690100531" >doi:10.1002/AIC.690100531</a> (1964). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'usepa82',
    'Goldstein (1982)',
    'Goldstein, D. J.: <span class="cmti-10">Air and steam</span> <span class="cmti-10">stripping of toxic pollutants, Appendix 3: Henry&#x2019;s law constants</span>, Tech. Rep. EPA-68-03-002, Industrial Environmental Research Laboratory, Cincinnati, OH, USA (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3729',
    'Goodarzi et&nbsp;al. (2010)',
    'Goodarzi, M., Ortiz, E. V., Coelho, L. D. S., &#x0026; Duchowicz, P. R.: <span class="cmti-10">Linear and non-linear relationships mapping the Henry&#x2019;s law</span> <span class="cmti-10">parameters of organic pesticides</span>, Atmos. Environ., 44, 3179–3186, <a href="https://doi.org/10.1016/J.ATMOSENV.2010.05.025" >doi:10.1016/J.ATMOSENV.2010.05.025</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3935',
    'Goodwin (1883)',
    'Goodwin, W. L.: <span class="cmti-10">XXVI.—On the nature of solution. Part</span> <span class="cmti-10">I.—On the solubility of chlorine in water, and in aqueous solutions of</span> <span class="cmti-10">soluble chlorides</span>, Earth Environ. Sci. Trans. R. Soc. Edinburgh, 30, 597–618, <a href="https://doi.org/10.1017/S0080456800025096" >doi:10.1017/S0080456800025096</a> (1883). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3782',
    'Gordon et&nbsp;al. (1977)',
    'Gordon, L. I., Cohen, Y., &#x0026; Standley, D. R.: <span class="cmti-10">The solubility of molecular hydrogen in seawater</span>, Deep-Sea Res., 24, 937–941, <a href="https://doi.org/10.1016/0146-6291(77)90563-X" >doi:10.1016/0146-6291(77)90563-X</a> (1977). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'gordon1895',
    'Gordon (1895)',
    'Gordon, V.: <span class="cmti-10">Ueber die Absorption des Stickoxyduls</span> <span class="cmti-10">in Wasser und in Salzl</span><span class="cmti-10">ösungen</span>, Z. Phys. Chem., 18, 1–16, <a href="https://doi.org/10.1515/ZPCH-1895-1802" >doi:10.1515/ZPCH-1895-1802</a> (1895). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2953',
    'Görgényi et&nbsp;al. (2002)',
    'Görgényi, M., Dewulf, J., &#x0026; Van Langenhove, H.: <span class="cmti-10">Temperature dependence of</span> <span class="cmti-10">Henry&#x2019;s law constant in an extended temperature range</span>, Chemosphere, 48, 757–762, <a href="https://doi.org/10.1016/S0045-6535(02)00131-5" >doi:10.1016/S0045-6535(02)00131-5</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3780',
    'Goss (2005)',
    'Goss, K.-U.: <span class="cmti-10">Predicting the equilibrium partitioning of organic</span> <span class="cmti-10">compounds using just one linear solvation energy relationship (LSER)</span>, Fluid Phase Equilib., 233, 19–22, <a href="https://doi.org/10.1016/J.FLUID.2005.04.006" >doi:10.1016/J.FLUID.2005.04.006</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2993',
    'Goss et&nbsp;al. (2006)',
    'Goss, K. U., Bronner, G., Harner, T., Hertel, M., &#x0026; Schmidt, T.: <span class="cmti-10">The partition</span> <span class="cmti-10">behavior of fluorotelomer alcohols and olefins</span>, Environ. Sci. Technol., 40, 3572–3577, <a href="https://doi.org/10.1021/ES060004P" >doi:10.1021/ES060004P</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2951',
    'Gossett (1980)',
    'Gossett, J. M.: <span class="cmti-10">Packed tower air</span> <span class="cmti-10">stripping of trichloroethylene from dilute aqueous solution</span>, Final Report ESL-TR-81-38, Engineering and Services Laboratory, Tyndall Air Force Base, FL (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '532',
    'Gossett (1987)',
    'Gossett, J. M.: <span class="cmti-10">Measurement of Henry&#x2019;s law constants for</span> C<sub><span class="cmr-7">1</span></sub> <span class="cmti-10">and</span> C<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">chlorinated hydrocarbons</span>, Environ. Sci. Technol., 21, 202–208, <a href="https://doi.org/10.1021/ES00156A012" >doi:10.1021/ES00156A012</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2269',
    'Gossett et&nbsp;al. (1985)',
    'Gossett, J. M., Cameron, C. E., Eckstrom, B. P., Goodman, C., &#x0026; Lincoff, A. H.: <span class="cmti-10">Mass transfer coefficients and Henry&#x2019;s constants</span> <span class="cmti-10">for packed-tower air stripping of volatile organics: Measurements and</span> <span class="cmti-10">Correlations</span>, Final Report ESL-TR-85-18, Engineering and Services Laboratory, Tyndall Air Force Base, FL (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1112',
    'Govers and Krop (1998)',
    'Govers, H. A. J. &#x0026; Krop, H. B.: <span class="cmti-10">Partition constants</span> <span class="cmti-10">of chlorinated dibenzofurans and dibenzo-p-dioxins</span>, Chemosphere, 37, 2139–2152, <a href="https://doi.org/10.1016/S0045-6535(98)00276-8" >doi:10.1016/S0045-6535(98)00276-8</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '82',
    'Graedel and Goldberg (1983)',
    'Graedel, T. E. &#x0026; Goldberg, K. I.: <span class="cmti-10">Kinetic</span> <span class="cmti-10">studies of raindrop chemistry 1.</span><span class="cmti-10"> Inorganic and organic processes</span>, J. Geophys. Res., 88, 10 865–10 882, <a href="https://doi.org/10.1029/JC088IC15P10865" >doi:10.1029/JC088IC15P10865</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2942',
    'Green and Frank (1979)',
    'Green, W. J. &#x0026; Frank, H. S.: <span class="cmti-10">The state of</span> <span class="cmti-10">dissolved benzene in aqueous solution</span>, J. Solution Chem., 8, 187–196, <a href="https://doi.org/10.1007/BF00648878" >doi:10.1007/BF00648878</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3715',
    'Grollman (1929)',
    'Grollman, A.: <span class="cmti-10">The solubility of gases in blood and blood</span> <span class="cmti-10">fluids</span>, J. Biol. Chem., 82, 317–325, <a href="https://doi.org/10.1016/S0021-9258(20)78278-5" >doi:10.1016/S0021-9258(20)78278-5</a> (1929). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2924',
    'Guitart et&nbsp;al. (1989)',
    'Guitart, R., Puigdemont, F., &#x0026; Arboix, M.: <span class="cmti-10">Rapid headspace gas chromatographic</span> <span class="cmti-10">method for the determination of liquid/gas partition coefficients</span>, J. Chromatogr., 491, 271–280, <a href="https://doi.org/10.1016/S0378-4347(00)82845-5" >doi:10.1016/S0378-4347(00)82845-5</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2183',
    'Guo and Brimblecombe (2007)',
    'Guo, X. X. &#x0026; Brimblecombe, P.: <span class="cmti-10">Henry&#x2019;s law constants of phenol and</span> <span class="cmti-10">mononitrophenols in water and aqueous sulfuric acid</span>, Chemosphere, 68, 436–444, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2007.01.011" >doi:10.1016/J.CHEMOSPHERE.2007.01.011</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2961',
    'Gupta et&nbsp;al. (2000)',
    'Gupta, A. K., Teja, A. S., Chai, X. S., &#x0026; Zhu, J. Y.: <span class="cmti-10">Henry&#x2019;s constants of </span><span class="cmmi-10">n</span><span class="cmti-10">-alkanols (methanol</span> <span class="cmti-10">through </span><span class="cmmi-10">n</span><span class="cmti-10">-hexanol) in water at temperatures between 40</span><span class="cmsy-7">∘</span>C <span class="cmti-10">and 90</span><span class="cmsy-7">∘</span>C, Fluid Phase Equilib., 170, 183–192, <a href="https://doi.org/10.1016/S0378-3812(00)00350-2" >doi:10.1016/S0378-3812(00)00350-2</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3591',
    'Gurol and Singer (1982)',
    'Gurol, M. D. &#x0026; Singer, P. C.: <span class="cmti-10">Kinetics of ozone</span> <span class="cmti-10">decomposition: a dynamic approach</span>, Environ. Sci. Technol., 16, 377–383, <a href="https://doi.org/10.1021/ES00101A003" >doi:10.1021/ES00101A003</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '794',
    'Guthrie (1973)',
    'Guthrie, J. P.: <span class="cmti-10">Hydration of carboxylic acids and esters.</span> <span class="cmti-10">Evaluation of the free energy change for addition of water to acetic and</span> <span class="cmti-10">formic acids and their methyl esters</span>, J. Am. Chem. Soc., 95, 6999–7003, <a href="https://doi.org/10.1021/JA00802A021" >doi:10.1021/JA00802A021</a> (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3304',
    'Haimi et&nbsp;al. (2006)',
    'Haimi, P., Uusi-Kyyny, P., Pokki, J.-P., Aittamaa, J., &#x0026; Keskinen, K. I.: <span class="cmti-10">Infinite dilution activity coefficient</span> <span class="cmti-10">measurements by inert gas stripping method</span>, Fluid Phase Equilib., 243, 126–132, <a href="https://doi.org/10.1016/J.FLUID.2006.02.022" >doi:10.1016/J.FLUID.2006.02.022</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '482',
    'Hales and Drewes (1979)',
    'Hales, J. M. &#x0026; Drewes, D. R.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">ammonia in water at low concentrations</span>, Atmos. Environ., 13, 1133–1147, <a href="https://doi.org/10.1016/0004-6981(79)90037-4" >doi:10.1016/0004-6981(79)90037-4</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3955',
    'Haller and Northgraves (1955)',
    'Haller, J. F. &#x0026; Northgraves, W. W.: <span class="cmti-10">Chlorine dioxide and safety</span>, TAPPI, 38, 199–202 (1955). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2456',
    'Hamelink et&nbsp;al. (1996)',
    'Hamelink, J. L., Simon, P. B., &#x0026; Silberhorn, E. M.: <span class="cmti-10">Henry&#x2019;s law constant, volatilization</span> <span class="cmti-10">rate, and aquatic half-life of octamethylcyclotetrasiloxane</span>, Environ. Sci. Technol., 30, 1946–1952, <a href="https://doi.org/10.1021/ES950634J" >doi:10.1021/ES950634J</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '632',
    'Hamm et&nbsp;al. (1984)',
    'Hamm, S., Hahn, J., Helas, G., &#x0026; Warneck, P.: <span class="cmti-10">Acetonitrile in the troposphere: residence</span> <span class="cmti-10">time due to rainout and uptake by the ocean</span>, Geophys. Res. Lett., 11, 1207–1210, <a href="https://doi.org/10.1029/GL011I012P01207" >doi:10.1029/GL011I012P01207</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '903',
    'Hansen et&nbsp;al. (1993)',
    'Hansen, K. C., Zhou, Z., Yaws, C. L., &#x0026; Aminabhavi, T. M.: <span class="cmti-10">Determination of Henry&#x2019;s</span> <span class="cmti-10">law constants of organics in dilute aqueous solutions</span>, J. Chem. Eng. Data, 38, 546–550, <a href="https://doi.org/10.1021/JE00012A017" >doi:10.1021/JE00012A017</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '618',
    'Hansen et&nbsp;al. (1995)',
    'Hansen, K. C., Zhou, Z., Yaws, C. L., &#x0026; Aminabhavi, T. M.: <span class="cmti-10">A laboratory method for</span> <span class="cmti-10">the determination of Henry&#x2019;s law constants of volatile organic chemicals</span>, J. Chem. Educ., 72, 93–96, <a href="https://doi.org/10.1021/ED072P93" >doi:10.1021/ED072P93</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '183',
    'Hanson and Ravishankara (1991)',
    'Hanson, D. R. &#x0026; Ravishankara, A. R.: <span class="cmti-10">The reaction probabilities of</span> ClONO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">and</span> N<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">5</span></sub> <span class="cmti-10">on 40 to 75% sulfuric acid solutions</span>, J. Geophys. Res., 96, 17 307–17 314, <a href="https://doi.org/10.1029/91JD01750" >doi:10.1029/91JD01750</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '515',
    'Hanson et&nbsp;al. (1992)',
    'Hanson, D. R., Burkholder, J. B., Howard, C. J., &#x0026; Ravishankara, A. R.: <span class="cmti-10">Measurement of</span> OH <span class="cmti-10">and</span> HO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">radical uptake coefficients on</span> <span class="cmti-10">water and sulfuric acid surfaces</span>, J. Phys. Chem., 96, 4979–4985, <a href="https://doi.org/10.1021/J100191A046" >doi:10.1021/J100191A046</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3660',
    'Harger et&nbsp;al. (1950)',
    'Harger, R. N., Raney, B. B., Bridwell, E. G., &#x0026; Kitchel, M. F.: <span class="cmti-10">The partition ratio of alcohol between air and water, urine and</span> <span class="cmti-10">blood; estimation and identification of alcohol in these liquids from</span> <span class="cmti-10">analysis of air equilibrated with them</span>, J. Biol. Chem., 183, 197–213, <a href="https://doi.org/10.1016/S0021-9258(18)56458-9" >doi:10.1016/S0021-9258(18)56458-9</a> (1950). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'harris51',
    'Harris (1951)',
    'Harris, T. A. B.: The Mode of Action of Anaesthetics, E. &#x0026; S. Livingstone (1951). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1912',
    'Harrison et&nbsp;al. (1993)',
    'Harrison, D. P., Valsaraj, K. T., &#x0026; Wetzel, D. M.: <span class="cmti-10">Air stripping of organics from ground</span> <span class="cmti-10">water</span>, Waste Manage., 13, 417–429, <a href="https://doi.org/10.1016/0956-053X(93)90074-7" >doi:10.1016/0956-053X(93)90074-7</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1513',
    'Harrison et&nbsp;al. (2002)',
    'Harrison, M. A. J., Cape, J. N., &#x0026; Heal, M. R.: <span class="cmti-10">Experimentally determined</span> <span class="cmti-10">Henry&#x2019;s Law coefficients of phenol, 2-methylphenol and 2-nitrophenol</span> <span class="cmti-10">in the temperature range 281-302 </span>K, Atmos. Environ., 36, 1843–1851, <a href="https://doi.org/10.1016/S1352-2310(02)00137-1" >doi:10.1016/S1352-2310(02)00137-1</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '640',
    'Hartkopf and Karger (1973)',
    'Hartkopf, A. &#x0026; Karger, B. L.: <span class="cmti-10">Study of the</span> <span class="cmti-10">interfacial properties of water by gas chromatography</span>, Acc. Chem. Res., 6, 209–216, <a href="https://doi.org/10.1021/AR50066A006" >doi:10.1021/AR50066A006</a> (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3524',
    'Haruta et&nbsp;al. (2011)',
    'Haruta, S., Jiao, W., Chen, W., Chang, A. C., &#x0026; Gan, J.: <span class="cmti-10">Evaluating Henry&#x2019;s law constant</span> <span class="cmti-10">of N-Nitrosodimethylamine (NDMA)</span>, Wat. Sci. Tech., 64, 1636–1641, <a href="https://doi.org/10.2166/WST.2011.742" >doi:10.2166/WST.2011.742</a> (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1024',
    'Hauff et&nbsp;al. (1998)',
    'Hauff, K., Fischer, R. G., &#x0026; Ballschmiter, K.: <span class="cmti-10">Determination of</span> C<sub><span class="cmr-7">1</span></sub><span class="cmti-10">-</span>C<sub><span class="cmr-7">5</span></sub> <span class="cmti-10">alkyl nitrates in rain, snow, white frost, and tap water by a combined</span> <span class="cmti-10">codistillation head-space gas chromatography technique. Determination of</span> <span class="cmti-10">Henry&#x2019;s law constants by head-space GC</span>, Chemosphere, 37, 2599–2615, <a href="https://doi.org/10.1016/S0045-6535(98)00159-3" >doi:10.1016/S0045-6535(98)00159-3</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2232',
    'Hawthorne et&nbsp;al. (1985)',
    'Hawthorne, S. B., Sievers, R. E., &#x0026; Barkley, R. M.: <span class="cmti-10">Organic emissions from shale oil</span> <span class="cmti-10">wastewaters and their implications for air quality</span>, Environ. Sci. Technol., 19, 992–997, <a href="https://doi.org/10.1021/ES00140A018" >doi:10.1021/ES00140A018</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3855',
    'Hayduk (1982)',
    'Hayduk, W.: IUPAC Solubility Data Series, Volume 9, Ethane, Pergamon Press, Oxford, ISBN 0080262309 (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3860',
    'Hayduk (1986)',
    'Hayduk, W.: IUPAC Solubility Data Series, Volume 24, Propane, Butane and 2-Methylpropane, Pergamon Press, Oxford, ISBN 008029202X (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3869',
    'Hayduk (1994)',
    'Hayduk, W.: IUPAC Solubility Data Series, Volume 57, Ethene, Pergamon Press, Oxford (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3728',
    'Hayer et&nbsp;al. (2022)',
    'Hayer, N., Jirasek, F., &#x0026; Hasse, H.: <span class="cmti-10">Prediction of Henry&#x2019;s law constants by matrix completion</span>, AIChE J., 68, e17 753, <a href="https://doi.org/10.1002/AIC.17753" >doi:10.1002/AIC.17753</a> (2022). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3034',
    'Haynes (2014)',
    'Haynes, W. M.: CRC Handbook of Chemistry and Physics, 95th Edition (Internet Version 2015), Taylor and Francis Group, <a href="https://doi.org/10.1201/b17118" >doi:10.1201/b17118</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1484',
    'Heal et&nbsp;al. (1995)',
    'Heal, M. R., Pilling, M. J., Titcombe, P. E., &#x0026; Whitaker, B. J.: <span class="cmti-10">Mass accommodation</span> <span class="cmti-10">of aniline, phenol and toluene on aqueous droplets</span>, Geophys. Res. Lett., 22, 3043–3046, <a href="https://doi.org/10.1029/95GL02944" >doi:10.1029/95GL02944</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2255',
    'Hedgecock and Pirrone (2004)',
    'Hedgecock, I. M. &#x0026; Pirrone, N.: <span class="cmti-10">Chasing</span> <span class="cmti-10">quicksilver: Modeling the atmospheric lifetime of</span> Hg<sub><span class="cmr-7">(g)</span></sub><sup><span class="cmr-7">0</span></sup> <span class="cmti-10">in the marine</span> <span class="cmti-10">boundary layer at various latitudes</span>, Environ. Sci. Technol., 38, 69–76, <a href="https://doi.org/10.1021/ES034623Z" >doi:10.1021/ES034623Z</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2169',
    'Hedgecock et&nbsp;al. (2005)',
    'Hedgecock, I. M., Trunfio, G. A., Pirrone, N., &#x0026; Sprovieri, F.: <span class="cmti-10">Mercury</span> <span class="cmti-10">chemistry in the MBL: Mediterranean case and sensitivity studies using</span> <span class="cmti-10">the AMCOTS (Atmospheric Mercury Chemistry over the Sea) model</span>, Atmos. Environ., 39, 7217–7230, <a href="https://doi.org/10.1016/J.ATMOSENV.2005.09.002" >doi:10.1016/J.ATMOSENV.2005.09.002</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2912',
    'Heidman et&nbsp;al. (1985)',
    'Heidman, J. L., Tsonopoulos, C., Brady, C. J., &#x0026; Wilson, G. M.: <span class="cmti-10">High-temperature mutual solubilities of hydrocarbons and water. Part II:</span> <span class="cmti-10">Ethylbenzene, ethylcyclohexane, and </span><span class="cmmi-10">n</span><span class="cmti-10">-octane</span>, AIChE J., 31, 376–384, <a href="https://doi.org/10.1002/AIC.690310304" >doi:10.1002/AIC.690310304</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2955',
    'Helburn et&nbsp;al. (2008)',
    'Helburn, R., Albritton, J., Howe, G., Michael, L., &#x0026; Franke, D.: <span class="cmti-10">Henry&#x2019;s law constants for fragrance and organic solvent compounds in</span> <span class="cmti-10">aqueous industrial surfactants</span>, J. Chem. Eng. Data, 53, 1071–1079, <a href="https://doi.org/10.1021/JE700418A" >doi:10.1021/JE700418A</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3775',
    'Heller et&nbsp;al. (2015)',
    'Heller, S. R., McNaught, A., Pletnev, I., Stein, S., &#x0026; Tchekhovskoi, D.: <span class="cmti-10">InChI, the IUPAC international chemical identifier</span>, J. Cheminformatics, 7, 23, <a href="https://doi.org/10.1186/S13321-015-0068-4" >doi:10.1186/S13321-015-0068-4</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2448',
    'Hellmann (1987)',
    'Hellmann, H.: <span class="cmti-10">Model tests on volatilization of organic</span> <span class="cmti-10">trace substances in surfaces waters</span>, Fresenius J. Anal. Chem., 328, 475–479, <a href="https://doi.org/10.1007/BF00475967" >doi:10.1007/BF00475967</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '869',
    'Hempel (1901)',
    'Hempel, W.: <span class="cmti-10">Ueber Kohlenoxysulfid</span>, Z. Angew. Chem., 14, 865–868, <a href="https://doi.org/10.1002/ANGE.19010143502" >doi:10.1002/ANGE.19010143502</a> (1901). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2606',
    'Henry (1803)',
    'Henry, W.: <span class="cmti-10">Experiments on the quantity of gases absorbed</span> <span class="cmti-10">by water, at different temperatures, and under different pressures</span>, Phil. Trans. R. Soc. Lond., 93, 29–43, <a href="https://doi.org/10.1098/RSTL.1803.0004" >doi:10.1098/RSTL.1803.0004</a> (1803). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1046',
    'Heron et&nbsp;al. (1998)',
    'Heron, G., Christensen, T. H., &#x0026; Enfield, C. G.: <span class="cmti-10">Henry&#x2019;s law constant for trichloroethylene</span> <span class="cmti-10">between 10 and 95</span><span class="cmsy-7">∘</span>C, Environ. Sci. Technol., 32, 1433–1437, <a href="https://doi.org/10.1021/ES9707015" >doi:10.1021/ES9707015</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2598',
    'Hertel and Sommer (2005)',
    'Hertel, M. O. &#x0026; Sommer, K.: <span class="cmti-10">Limiting separation</span> <span class="cmti-10">factors and limiting activity coefficients for 2-phenylethanol and</span> <span class="cmti-10">2-phenylethanal in water at 100</span><span class="cmsy-7">∘</span>C, J. Chem. Eng. Data, 50, 1905–1906, <a href="https://doi.org/10.1021/JE050171P" >doi:10.1021/JE050171P</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2599',
    'Hertel and Sommer (2006)',
    'Hertel, M. O. &#x0026; Sommer, K.: <span class="cmti-10">Limiting separation</span> <span class="cmti-10">factors and limiting activity coefficients for 2-furfural, </span><span class="cmmi-10">γ</span><span class="cmti-10">-nonalactone,</span> <span class="cmti-10">benzaldehyde, and linalool in water at 100</span><span class="cmsy-7">∘</span>C, J. Chem. Eng. Data, 51, 1283–1285, <a href="https://doi.org/10.1021/JE0600404" >doi:10.1021/JE0600404</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2600',
    'Hertel et&nbsp;al. (2007)',
    'Hertel, M. O., Scheuren, H., Sommer, K., &#x0026; Glas, K.: <span class="cmti-10">Limiting separation</span> <span class="cmti-10">factors and limiting activity coefficients for hexanal, 2-methylbutanal,</span> <span class="cmti-10">3-methylbutanal, and dimethylsulfide in water at (98.1 to 99.0)</span><span class="cmsy-7">∘</span>C, J. Chem. Eng. Data, 52, 148–150, <a href="https://doi.org/10.1021/JE060324O" >doi:10.1021/JE060324O</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2891',
    'Hiatt (2013)',
    'Hiatt, M. H.: <span class="cmti-10">Determination of Henry&#x2019;s law constants using</span> <span class="cmti-10">internal standards with benchmark values</span>, J. Chem. Eng. Data, 58, 902–908, <a href="https://doi.org/10.1021/JE3010535" >doi:10.1021/JE3010535</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2521',
    'Hilal et&nbsp;al. (2008)',
    'Hilal, S. H., Ayyampalayam, S. N., &#x0026; Carreira, L. A.: <span class="cmti-10">Air-liquid</span> <span class="cmti-10">partition coefficient for a diverse set of organic compounds: Henry&#x2019;s law</span> <span class="cmti-10">constant in water and hexadecane</span>, Environ. Sci. Technol., 42, 9231–9236, <a href="https://doi.org/10.1021/ES8005783" >doi:10.1021/ES8005783</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '627',
    'Hill et&nbsp;al. (1968)',
    'Hill, J. O., Worsley, I. G., &#x0026; Hepler, L. G.: <span class="cmti-10">Calorimetric determination of the distribution coefficient</span> <span class="cmti-10">and thermodynamic properties of bromine in water and carbon</span> <span class="cmti-10">tetrachloride</span>, J. Phys. Chem., 72, 3695–3697, <a href="https://doi.org/10.1021/J100856A066" >doi:10.1021/J100856A066</a> (1968). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3431',
    'Himmelblau (1960)',
    'Himmelblau, D. M.: <span class="cmti-10">Solubilities of inert gases in water.</span> <span class="cmti-10">0</span><span class="cmsy-7">∘</span>C<span class="cmti-10">. to near the critical point of water</span>, J. Chem. Eng. Data, 5, 10–15, <a href="https://doi.org/10.1021/JE60005A003" >doi:10.1021/JE60005A003</a> (1960). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '642',
    'Hine and Mookerjee (1975)',
    'Hine, J. &#x0026; Mookerjee, P. K.: <span class="cmti-10">The intrinsic hydrophilic character of organic</span> <span class="cmti-10">compounds. Correlations in terms of structural contributions</span>, J. Org. Chem., 40, 292–298, <a href="https://doi.org/10.1021/JO00891A006" >doi:10.1021/JO00891A006</a> (1975). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '714',
    'Hine and Weimar (1965)',
    'Hine, J. &#x0026; Weimar, Jr., R. D.: <span class="cmti-10">Carbon basicity</span>, J. Am. Chem. Soc., 87, 3387–3396, <a href="https://doi.org/10.1021/JA01093A018" >doi:10.1021/JA01093A018</a> (1965). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3964',
    'Hirshberg et&nbsp;al. (2018)',
    'Hirshberg, B., Rossich Molina, E., Götz, A. W., Hammerich, A. D., Nathanson, G. M., Bertram, T. H., Johnson, M. A., &#x0026; Gerber, R. B.: N<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">5</span></sub> <span class="cmti-10">at water surfaces: binding forces, charge separation, energy</span> <span class="cmti-10">accommodation and atmospheric implications</span>, Phys. Chem. Chem. Phys., 20, 17 961–17 976, <a href="https://doi.org/10.1039/C8CP03022G" >doi:10.1039/C8CP03022G</a> (2018). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '637',
    'Hoff et&nbsp;al. (1993)',
    'Hoff, J. T., Mackay, D., Gillham, R., &#x0026; Shiu, W. Y.: <span class="cmti-10">Partitioning of organic chemicals at the</span> <span class="cmti-10">air–water interface in environmental systems</span>, Environ. Sci. Technol., 27, 2174–2180, <a href="https://doi.org/10.1021/ES00047A026" >doi:10.1021/ES00047A026</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '140',
    'Hoffmann and Calvert (1985)',
    'Hoffmann, M. R. &#x0026; Calvert, J. G.: <span class="cmti-10">Chemical</span> <span class="cmti-10">transformation modules for Eulerian acid deposition models. Volume II.</span> <span class="cmti-10">The aqueous-phase chemistry</span>, Tech. rep., NCAR, Box 3000, Boulder, CO 80307 (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '103',
    'Hoffmann and Jacob (1984)',
    'Hoffmann, M. R. &#x0026; Jacob, D. J.: <span class="cmti-10">Kinetics and</span> <span class="cmti-10">mechanisms of the catalytic oxidation of dissolved sulfur dioxide in</span> <span class="cmti-10">aqueous solution: An application to nighttime fog water chemistry</span>, in: SO<sub><span class="cmr-7">2</span></sub>, NO and NO<sub><span class="cmr-7">2</span></sub> Oxidation Mechanisms: Atmospheric Considerations, edited by Calvert, J. G., pp. 101–172, Butterworth Publishers, Boston, MA, ISBN 0250405687 (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '513',
    'Holdren et&nbsp;al. (1984)',
    'Holdren, M. W., Spicer, C. W., &#x0026; Hales, J. M.: <span class="cmti-10">Peroxyacetyl nitrate solubility</span> <span class="cmti-10">and decomposition rate in acidic water</span>, Atmos. Environ., 18, 1171–1173, <a href="https://doi.org/10.1016/0004-6981(84)90148-3" >doi:10.1016/0004-6981(84)90148-3</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3969',
    'Holst (1944)',
    'Holst, G.: <span class="cmti-10">Die Grundlagen einer technischen Methode zur</span> <span class="cmti-10">Darstellung von Chlordioxyd</span>, Svensk Papperstidn., 47, 537–546 (1944). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '287',
    'Holzwarth et&nbsp;al. (1984)',
    'Holzwarth, G., Balmer, R. G., &#x0026; Soni, L.: <span class="cmti-10">The fate of chlorine</span> <span class="cmti-10">and chloramines in cooling towers</span>, Wat. Res., 18, 1421–1427, <a href="https://doi.org/10.1016/0043-1354(84)90012-5" >doi:10.1016/0043-1354(84)90012-5</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3653',
    'Horvath and Getzen (1999)',
    'Horvath, A. L. &#x0026; Getzen, F. W.: <span class="cmti-10">IUPAC-NIST</span> <span class="cmti-10">Solubility Data Series 68. Halogenated Aliphatic Hydrocarbon Compounds</span> C<sub><span class="cmr-7">3</span></sub><span class="cmti-10">-</span>C<sub><span class="cmr-7">14</span></sub>, J. Phys. Chem. Ref. Data, 28, 649–777, <a href="https://doi.org/10.1063/1.556051" >doi:10.1063/1.556051</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '443',
    'Hough (1991)',
    'Hough, A. M.: <span class="cmti-10">Development of a two-dimensional global</span> <span class="cmti-10">tropospheric model: Model chemistry</span>, J. Geophys. Res., 96, 7325–7362, <a href="https://doi.org/10.1029/90JD01327" >doi:10.1029/90JD01327</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2450',
    'Hovorka and Dohnal (1997)',
    'Hovorka, Š. &#x0026; Dohnal, V.: <span class="cmti-10">Determination of</span> <span class="cmti-10">air–water partitioning of volatile halogenated hydrocarbons by the</span> <span class="cmti-10">inert gas stripping method</span>, J. Chem. Eng. Data, 42, 924–933, <a href="https://doi.org/10.1021/JE970046G" >doi:10.1021/JE970046G</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3609',
    'Hovorka et&nbsp;al. (2002)',
    'Hovorka, Š., Dohnal, V., Roux, A. H., &#x0026; Roux-Desgranges, G.: <span class="cmti-10">Determination of temperature dependence of limiting activity coefficients</span> <span class="cmti-10">for a group of moderately hydrophobic organic solutes in water</span>, Fluid Phase Equilib., 201, 135–164, <a href="https://doi.org/10.1016/S0378-3812(02)00087-0" >doi:10.1016/S0378-3812(02)00087-0</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3632',
    'Hovorka et&nbsp;al. (2019)',
    'Hovorka, Š., Vrbka, P., Bermúdez-Salguero, C., Böhme, A., &#x0026; Dohnal, V.: <span class="cmti-10">Air–water partitioning of C5 and C6 alkanones: measurement,</span> <span class="cmti-10">critical compilation, correlation, and recommended data</span>, J. Chem. Eng. Data, 64, 5765–5774, <a href="https://doi.org/10.1021/ACS.JCED.9B00726" >doi:10.1021/ACS.JCED.9B00726</a> (2019). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'howard89',
    'Howard (1989)',
    'Howard, P. H.: Handbook of Environmental fate and exposure data for organic chemicals. Vol. I: Large production and priority pollutants, Lewis Publishers Inc. Chelsea, Michigan, ISBN 0873711513 (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'howard90',
    'Howard (1990)',
    'Howard, P. H.: Handbook of Environmental fate and exposure data for organic chemicals. Vol. II: Solvents, Lewis Publishers Inc. Chelsea, Michigan, <a href="https://doi.org/10.1201/9781003418863" >doi:10.1201/9781003418863</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'howard91b',
    'Howard (1991)',
    'Howard, P. H.: Handbook of Environmental fate and exposure data for organic chemicals. Vol. III: Pesticides, Lewis Publishers Inc. Chelsea, Michigan, <a href="https://doi.org/10.1201/9780203719305" >doi:10.1201/9780203719305</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'howard93',
    'Howard (1993)',
    'Howard, P. H.: Handbook of Environmental fate and exposure data for organic chemicals. Vol. IV: Solvents 2, Lewis Publishers Inc. Chelsea, Michigan, <a href="https://doi.org/10.1201/9781003418887" >doi:10.1201/9781003418887</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'howard97',
    'Howard and Meylan (1997)',
    'Howard, P. H. &#x0026; Meylan, W. M.: Handbook of physical properties of organic chemicals, CRC Press, Lewis Publisher, Boca Raton, FL, ISBN 1566702275 (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'howard91',
    'Howard et&nbsp;al. (1991)',
    'Howard, P. H., Boethling, R. S., Jarvis, W. F., Meylan, W. M., &#x0026; Michalenko, E. M.: Handbook of Environmental Degradation Rates, Lewis Publishers Inc. Chelsea, Michigan, <a href="https://doi.org/10.1201/9780203719329" >doi:10.1201/9780203719329</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2894',
    'Howe et&nbsp;al. (1987)',
    'Howe, G. B., Mullins, M. E., &#x0026; Rogers, T. N.: <span class="cmti-10">Evaluation and prediction of Henry&#x2019;s law constants</span> <span class="cmti-10">and aqueous solubilities for solvents and hydrocarbon fuel components.</span> <span class="cmti-10">Vol II: Experimental Henry&#x2019;s law data</span>, Tech. Rep. NTIS AD-A202 262, Research Triangle Institute, Research Triangle Park, NC, 27709, USA (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3119',
    'Hoyt (1982)',
    'Hoyt, S. D.: The ocean-air exchange of carbonyl sulfide (OCS) and halocarbons, Ph.D. thesis, Oregon Graduate Center, USA, URL https://digitalcollections.ohsu.edu/record/67/files/67_etd.pdf (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3088',
    'HSDB (2015)',
    'HSDB: <span class="cmti-10">Hazardous Substances Data Bank, TOXicology</span> <span class="cmti-10">data NETwork (TOXNET), National Library of Medicine (US)</span>, URL https://www.nlm.nih.gov/toxnet/Accessing_HSDB_Content_from_PubChem.html (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2827',
    'Huang and Chen (2010)',
    'Huang, D. &#x0026; Chen, Z.: <span class="cmti-10">Reinvestigation of the Henry&#x2019;s law constant for</span> <span class="cmti-10">hydrogen peroxide with temperature and acidity variation</span>, J. Environ. Sci., 22, 570–574, <a href="https://doi.org/10.1016/S1001-0742(09)60147-9" >doi:10.1016/S1001-0742(09)60147-9</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '608',
    'Hunter-Smith et&nbsp;al. (1983)',
    'Hunter-Smith, R. J., Balls, P. W., &#x0026; Liss, P. S.: <span class="cmti-10">Henry&#x2019;s law constants and the air-sea</span> <span class="cmti-10">exchange of various low molecular weight halocarbon gases</span>, Tellus, 35B, 170–176, <a href="https://doi.org/10.1111/J.1600-0889.1983.TB00021.X" >doi:10.1111/J.1600-0889.1983.TB00021.X</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '315',
    'Huthwelker et&nbsp;al. (1995)',
    'Huthwelker, T., Clegg, S. L., Peter, T., Carslaw, K., Luo, B. P., &#x0026; Brimblecombe, P.: <span class="cmti-10">Solubility of</span> HOCl <span class="cmti-10">in water and aqueous</span> H<sub><span class="cmr-7">2</span></sub>SO<sub><span class="cmr-7">4</span></sub> <span class="cmti-10">to stratospheric temperatures</span>, J. Atmos. Chem., 21, 81–95, <a href="https://doi.org/10.1007/BF00712439" >doi:10.1007/BF00712439</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '488',
    'Hwang and Dasgupta (1985)',
    'Hwang, H. &#x0026; Dasgupta, P. G.: <span class="cmti-10">Thermodynamics of the hydrogen peroxide-water</span> <span class="cmti-10">system</span>, Environ. Sci. Technol., 19, 255–258, <a href="https://doi.org/10.1021/ES00133A006" >doi:10.1021/ES00133A006</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3691',
    'Hwang et&nbsp;al. (2010)',
    'Hwang, I.-C., Kwak, H.-Y., &#x0026; Park, S.-J.: <span class="cmti-10">Determination and prediction of </span><span class="cmmi-10">K</span><sub><span class="cmmi-7">ow</span></sub> <span class="cmti-10">and dimensionless</span> <span class="cmti-10">Henry&#x2019;s constant (</span><span class="cmmi-10">H</span><span class="cmti-10">) for 6 ether compounds at several temperatures</span>, J. Ind. Eng. Chem., 16, 629–633, <a href="https://doi.org/10.1016/J.JIEC.2010.03.003" >doi:10.1016/J.JIEC.2010.03.003</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2439',
    'Hwang et&nbsp;al. (1992)',
    'Hwang, Y.-L., Olson, J. D., &#x0026; Keller, II, G. E.: <span class="cmti-10">Steam stripping for removal of organic pollutants</span> <span class="cmti-10">from water. 2. Vapor-liquid equilibrium data</span>, Ind. Eng. Chem. Res., 31, 1759–1768, <a href="https://doi.org/10.1021/IE00007A022" >doi:10.1021/IE00007A022</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1936',
    'Iliuta and Larachi (2005a)',
    'Iliuta, M. C. &#x0026; Larachi, F.: <span class="cmti-10">Gas-liquid partition</span> <span class="cmti-10">coefficients and Henry&#x2019;s law constants of DMS in aqueous solutions of</span> Fe(II) <span class="cmti-10">chelate complexes using the static headspace method</span>, J. Chem. Eng. Data, 50, 1700–1705, <a href="https://doi.org/10.1021/JE0501686" >doi:10.1021/JE0501686</a> (2005a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3610',
    'Iliuta and Larachi (2005b)',
    'Iliuta, M. C. &#x0026; Larachi, F.: <span class="cmti-10">Solubility of dimethyldisulfide (DMDS) in aqueous</span> <span class="cmti-10">solutions of Fe(III) complexes of trans-1,2-cyclohexanediaminetetraacetic</span> <span class="cmti-10">acid (CDTA) using the static headspace method</span>, Fluid Phase Equilib., 233, 184–189, <a href="https://doi.org/10.1016/J.FLUID.2005.05.004" >doi:10.1016/J.FLUID.2005.05.004</a> (2005b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2630',
    'Iliuta and Larachi (2007)',
    'Iliuta, M. C. &#x0026; Larachi, F.: <span class="cmti-10">Solubility of total</span> <span class="cmti-10">reduced sulfurs (hydrogen sulfide, methyl mercaptan, dimethyl sulfide,</span> <span class="cmti-10">and dimethyl disulfide) in liquids</span>, J. Chem. Eng. Data, 52, 2–19, <a href="https://doi.org/10.1021/JE060263U" >doi:10.1021/JE060263U</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3957',
    'Imagawa (1950)',
    'Imagawa, H.: <span class="cmti-10">Chemical reactions in the</span> <span class="cmti-10">chlorate manufacturing electrolytic cell (part 1) The vapour pressure of</span> <span class="cmti-10">hypochlorous acid on its aquous solution</span>, J. Electrochem. Soc. Jpn., 18, 382–385, <a href="https://doi.org/10.5796/DENKA.18.382" >doi:10.5796/DENKA.18.382</a> (1950). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3100',
    'Inga and McKetta (1961)',
    'Inga, R. F. &#x0026; McKetta, J. J.: <span class="cmti-10">Solubility of propyne</span> <span class="cmti-10">in water</span>, J. Chem. Eng. Data, 6, 337–338, <a href="https://doi.org/10.1021/JE00103A008" >doi:10.1021/JE00103A008</a> (1961). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3640',
    'Ioffe et&nbsp;al. (1984)',
    'Ioffe, B. V., Kostkina, M. I., &#x0026; Vitenberg, A. G.: <span class="cmti-10">Preparation of standard vapor-gas mixtures</span> <span class="cmti-10">for gas chromatography: discontinuous gas extraction</span>, Anal. Chem., 56, 2500–2503, <a href="https://doi.org/10.1021/AC00277A053" >doi:10.1021/AC00277A053</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2401',
    'Ip et&nbsp;al. (2009)',
    'Ip, H. S. S., Huang, X. H. H., &#x0026; Yu, J. Z.: <span class="cmti-10">Effective Henry&#x2019;s law constants of</span> <span class="cmti-10">glyoxal, glyoxylic acid, and glycolic acid</span>, Geophys. Res. Lett., 36, L01802, <a href="https://doi.org/10.1029/2008GL036212" >doi:10.1029/2008GL036212</a> (2009). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1004',
    'Iraci et&nbsp;al. (1999)',
    'Iraci, L. T., Baker, B. M., Tyndall, G. S., &#x0026; Orlando, J. J.: <span class="cmti-10">Measurements of the Henry&#x2019;s</span> <span class="cmti-10">law coefficients of 2-methyl-3-buten-2-ol, methacrolein, and methylvinyl</span> <span class="cmti-10">ketone</span>, J. Atmos. Chem., 33, 321–330, <a href="https://doi.org/10.1023/A:1006169029230" >doi:10.1023/A:1006169029230</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2550',
    'Irmann (1965)',
    'Irmann, F.: <span class="cmti-10">Eine einfache Korrelation zwischen Wasserl</span><span class="cmti-10">öslichkeit und Struktur von</span> <span class="cmti-10">Kohlenwasserstoffen und Halogenkohlenwasserstoffen</span>, Chem.-Ing.-Tech., 37, 789–798, <a href="https://doi.org/10.1002/CITE.330370802" >doi:10.1002/CITE.330370802</a> (1965). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3413',
    'Isaacman-VanWertz et&nbsp;al. (2016)',
    'Isaacman-VanWertz, G., Yee, L. D., Kreisberg, N. M., Wernis, R., Moss, J. A., Hering, S. V., de Sá, S. S., Martin, S. T., Alexander, M. L., Palm, B. B., Hu, W., Campuzano-Jost, P., Day, D. A., Jimenez, J. L., Riva, M., Surratt, J. D., Viegas, J., Manzi, A., Edgerton, E., Baumann, K., Souza, R., Artaxo, P., &#x0026; Goldstein, A. H.: <span class="cmti-10">Ambient</span> <span class="cmti-10">gas-particle partitioning of tracers for biogenic oxidation</span>, Environ. Sci. Technol., 50, 9952–9962, <a href="https://doi.org/10.1021/ACS.EST.6B01674" >doi:10.1021/ACS.EST.6B01674</a> (2016). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3965',
    'Ishi (1958)',
    'Ishi, G.: <span class="cmti-10">Solubility of chlorine dioxide</span>, Chem. Eng. (Japan), 22, 153–154, <a href="https://doi.org/10.1252/KAKORONBUNSHU1953.22.153" >doi:10.1252/KAKORONBUNSHU1953.22.153</a> (1958). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2288',
    'Iverfeldt and Lindqvist (1982)',
    'Iverfeldt, Å. &#x0026; Lindqvist, O.: <span class="cmti-10">Distribution</span> <span class="cmti-10">equilibrium of methyl mercury chloride between water and air</span>, Atmos. Environ., 16, 2917–2925, <a href="https://doi.org/10.1016/0004-6981(82)90042-7" >doi:10.1016/0004-6981(82)90042-7</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2576',
    'Iverfeldt and Persson (1985)',
    'Iverfeldt, Å. &#x0026; Persson, I.: <span class="cmti-10">The solvation thermodynamics of methylmercury(II) species derived from</span> <span class="cmti-10">measurements of the heat of solution and the Henry&#x2019;s law constant</span>, Inorg. Chim. Acta, 103, 113–119, <a href="https://doi.org/10.1016/S0020-1693(00)87476-9" >doi:10.1016/S0020-1693(00)87476-9</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '46',
    'Jacob (1986)',
    'Jacob, D. J.: <span class="cmti-10">Chemistry of</span> OH <span class="cmti-10">in remote clouds and its role</span> <span class="cmti-10">in the production of formic acid and peroxymonosulfate</span>, J. Geophys. Res., 91, 9807–9826, <a href="https://doi.org/10.1029/JD091ID09P09807" >doi:10.1029/JD091ID09P09807</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '76',
    'Jacob et&nbsp;al. (1989)',
    'Jacob, D. J., Gottlieb, E. W., &#x0026; Prather, M. J.: <span class="cmti-10">Chemistry of a</span> <span class="cmti-10">polluted cloudy boundary layer</span>, J. Geophys. Res., 94, 12 975–13 002, <a href="https://doi.org/10.1029/JD094ID10P12975" >doi:10.1029/JD094ID10P12975</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3751',
    'Jadkar and Chaudhari (1980)',
    'Jadkar, P. B. &#x0026; Chaudhari, R. V.: <span class="cmti-10">Solubility</span> <span class="cmti-10">of acetylene in aqueous solutions of formaldehyde and 2-butyne-1,4-diol</span>, J. Chem. Eng. Data, 25, 115–117, <a href="https://doi.org/10.1021/JE60085A021" >doi:10.1021/JE60085A021</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1536',
    'Jaeglé et&nbsp;al. (1996)',
    'Jaeglé, L., Yung, Y. L., Toon, G. C., Sen, B., &#x0026; Blavier, J.-F.: <span class="cmti-10">Balloon observations or</span> <span class="cmti-10">organic and inorganic chlorine in the stratosphere: The role of</span> HClO<sub><span class="cmr-7">4</span></sub> <span class="cmti-10">production on sulfate aerosols</span>, Geophys. Res. Lett., 23, 1749–1752, <a href="https://doi.org/10.1029/96GL01543" >doi:10.1029/96GL01543</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1154',
    'Janini and Quaddora (1986)',
    'Janini, G. M. &#x0026; Quaddora, L. A.: <span class="cmti-10">Determination of activity coefficients of oxygenated</span> <span class="cmti-10">hydrocarbons by liquid-liquid chromatography</span>, J. Liq. Chromatogr., 9, 39–53, <a href="https://doi.org/10.1080/01483918608076621" >doi:10.1080/01483918608076621</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2492',
    'Jantunen and Bidleman (2006)',
    'Jantunen, L. M. &#x0026; Bidleman, T. F.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">law</span> <span class="cmti-10">constants for hexachlorobenzene, p,p&#x2019;-DDE and components of technical</span> <span class="cmti-10">chlordane and estimates of gas exchange for Lake Ontario</span>, Chemosphere, 62, 1689–1696, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2005.06.035" >doi:10.1016/J.CHEMOSPHERE.2005.06.035</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3664',
    'Järnberg and Johanson (1995)',
    'Järnberg, J. &#x0026; Johanson, G.: <span class="cmti-10">Liquid/air</span> <span class="cmti-10">partition coefficients of the trimethylbenzenes</span>, Toxicol. Ind. Health, 11, 81–88, <a href="https://doi.org/10.1177/074823379501100107" >doi:10.1177/074823379501100107</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2525',
    'Jayasinghe et&nbsp;al. (1992)',
    'Jayasinghe, D. S., Brownawell, B. J., Chen, H., &#x0026; Westall, J. C.: <span class="cmti-10">Determination of Henry&#x2019;s constants of organic compounds of low</span> <span class="cmti-10">volatility: methylanilines in methanol-water</span>, Environ. Sci. Technol., 26, 2275–2281, <a href="https://doi.org/10.1021/ES00035A028" >doi:10.1021/ES00035A028</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '753',
    'Jenkins and King (1965)',
    'Jenkins, J. &#x0026; King, M. B.: <span class="cmti-10">Vapor-liquid equilibria</span> <span class="cmti-10">for the system bromine/water at low bromine concentrations</span>, Chem. Eng. Sci., 20, 921–922, <a href="https://doi.org/10.1016/0009-2509(65)80089-6" >doi:10.1016/0009-2509(65)80089-6</a> (1965). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2524',
    'Ji and Evans (2007)',
    'Ji, C. &#x0026; Evans, E. M.: <span class="cmti-10">Using an internal standard</span> <span class="cmti-10">method to determine Henry&#x2019;s law constants</span>, Environ. Toxicol. Chem., 26, 231–236, <a href="https://doi.org/10.1897/06-339R.1" >doi:10.1897/06-339R.1</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3724',
    'Ji et&nbsp;al. (2008)',
    'Ji, C., Day, S. E., Ortega, S. A., &#x0026; Beall, G. W.: <span class="cmti-10">Henry&#x2019;s law constants of some aromatic aldehydes and</span> <span class="cmti-10">ketones measured by an internal standard method</span>, J. Chem. Eng. Data, 53, 1093–1097, <a href="https://doi.org/10.1021/JE700612B" >doi:10.1021/JE700612B</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3079',
    'Johanson and Dynésius (1988)',
    'Johanson, G. &#x0026; Dynésius, B.: <span class="cmti-10">Liquid/air</span> <span class="cmti-10">partition coefficients of six commonly used glycol ethers</span>, Br. J. Ind. Med., 45, 561–564, <a href="https://doi.org/10.1136/OEM.45.8.561" >doi:10.1136/OEM.45.8.561</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '4023',
    'Johnson (1990)',
    'Johnson, B. J.: The carbon isotope content and concentration of ambient formic acid and acetic acid, Ph.D. thesis, University of Arizona, Tucson, AZ, USA, URL https://hdl.handle.net/10150/185355 (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '511',
    'Johnson et&nbsp;al. (1996)',
    'Johnson, B. J., Betterton, E. A., &#x0026; Craig, D.: <span class="cmti-10">Henry&#x2019;s law coefficients of formic</span> <span class="cmti-10">and acetic acids</span>, J. Atmos. Chem., 24, 113–119, <a href="https://doi.org/10.1007/BF00162406" >doi:10.1007/BF00162406</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '524',
    'Johnson and Harrison (1986)',
    'Johnson, J. E. &#x0026; Harrison, H.: <span class="cmti-10">Carbonyl sulfide</span> <span class="cmti-10">concentrations in the surface waters and above the Pacific Ocean</span>, J. Geophys. Res., 91, 7883–7888, <a href="https://doi.org/10.1029/JD091ID07P07883" >doi:10.1029/JD091ID07P07883</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1072',
    'Johnstone and Leppla (1934)',
    'Johnstone, H. F. &#x0026; Leppla, P. W.: <span class="cmti-10">The</span> <span class="cmti-10">solubility of sulfur-dioxide at low partial pressures</span>, J. Am. Chem. Soc., 56, 2233–2238, <a href="https://doi.org/10.1021/JA01326A009" >doi:10.1021/JA01326A009</a> (1934). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3661',
    'Jones (1983)',
    'Jones, A. W.: <span class="cmti-10">Determination of liquid/air partition coefficients for dilute solutions of</span> <span class="cmti-10">ethanol in water, whole blood, and plasma</span>, J. Anal. Technol., 7, 193–197, <a href="https://doi.org/10.1093/JAT/7.4.193" >doi:10.1093/JAT/7.4.193</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3934',
    'Jones (1911)',
    'Jones, W. J.: <span class="cmti-10">XLIX.—The determination of solubility</span> <span class="cmti-10">coefficients by aspiration</span>, J. Chem. Soc. Trans., 99, 392–404, <a href="https://doi.org/10.1039/CT9119900392" >doi:10.1039/CT9119900392</a> (1911). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3649',
    'Jones et&nbsp;al. (1988)',
    'Jones, W. J., Egoville, M. J., Strolle, E. O., &#x0026; Dellamonica, E. S.: <span class="cmti-10">Determination of</span> <span class="cmti-10">partition coefficients by headspace gas chromatography</span>, J. Chromatogr. A, 455, 45–51, <a href="https://doi.org/10.1016/S0021-9673(01)82105-1" >doi:10.1016/S0021-9673(01)82105-1</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2909',
    'Jönsson et&nbsp;al. (1982)',
    'Jönsson, J. Å., Vejrosta, J., &#x0026; Novák, J.: <span class="cmti-10">Air/water partition coefficients for normal</span> <span class="cmti-10">alkanes (</span><span class="cmmi-10">n</span><span class="cmti-10">-pentane to </span><span class="cmmi-10">n</span><span class="cmti-10">-nonane)</span>, Fluid Phase Equilib., 9, 279–286, <a href="https://doi.org/10.1016/0378-3812(82)80023-X" >doi:10.1016/0378-3812(82)80023-X</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2833',
    'Joosten and Danckwerts (1972)',
    'Joosten, G. E. H. &#x0026; Danckwerts, P. V.: <span class="cmti-10">Solubility and diffusivity of nitrous oxide in equimolar potassium</span> <span class="cmti-10">carbonate-potassium bicarbonate solutions at 25</span><span class="cmsy-7">∘</span>C <span class="cmti-10">and 1atm</span>, J. Chem. Eng. Data, 17, 452–454, <a href="https://doi.org/10.1021/JE60055A016" >doi:10.1021/JE60055A016</a> (1972). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2910',
    'Jou and Mather (2000)',
    'Jou, F.-Y. &#x0026; Mather, A. E.: <span class="cmti-10">Vapor-liquid-liquid locus</span> <span class="cmti-10">of the system pentane + water</span>, J. Chem. Eng. Data, 45, 728–729, <a href="https://doi.org/10.1021/JE000065H" >doi:10.1021/JE000065H</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3556',
    'Jou et&nbsp;al. (2021)',
    'Jou, F.-Y., Mather, A. E., &#x0026; Schmidt, K. A. G.: <span class="cmti-10">Solubility of ethanethiol in water</span>, J. Chem. Eng. Data, 66, 4174–4179, <a href="https://doi.org/10.1021/ACS.JCED.1C00358" >doi:10.1021/ACS.JCED.1C00358</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '541',
    'Kames and Schurath (1992)',
    'Kames, J. &#x0026; Schurath, U.: <span class="cmti-10">Alkyl nitrates and</span> <span class="cmti-10">bifunctional nitrates of atmospheric interest: Henry&#x2019;s law constants</span> <span class="cmti-10">and their temperature dependencies</span>, J. Atmos. Chem., 15, 79–95, <a href="https://doi.org/10.1007/BF00053611" >doi:10.1007/BF00053611</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '624',
    'Kames and Schurath (1995)',
    'Kames, J. &#x0026; Schurath, U.: <span class="cmti-10">Henry&#x2019;s law</span> <span class="cmti-10">and hydrolysis-rate constants for peroxyacyl nitrates (</span>PAN<span class="cmti-10">s) using</span> <span class="cmti-10">a homogeneous gas-phase source</span>, J. Atmos. Chem., 21, 151–164, <a href="https://doi.org/10.1007/BF00696578" >doi:10.1007/BF00696578</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '502',
    'Kames et&nbsp;al. (1991)',
    'Kames, J., Schweighoefer, S., &#x0026; Schurath, U.: <span class="cmti-10">Henry&#x2019;s law constant and</span> <span class="cmti-10">hydrolysis of peroxyacetyl nitrate (</span>PAN<span class="cmti-10">)</span>, J. Atmos. Chem., 12, 169–180, <a href="https://doi.org/10.1007/BF00115778" >doi:10.1007/BF00115778</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3010',
    'Kampf et&nbsp;al. (2013)',
    'Kampf, C. J., Waxman, E. M., Slowik, J. G., Dommen, J., Pfaffenberger, L., Praplan, A. P., Prévôt, A. S. H., Baltensperger, U., Hoffmann, T., &#x0026; Volkamer, R.: <span class="cmti-10">Effective Henry&#x2019;s law partitioning and</span> <span class="cmti-10">the salting constant of glyoxal in aerosols containing sulfate</span>, Environ. Sci. Technol., 47, 4236–4244, <a href="https://doi.org/10.1021/ES400083D" >doi:10.1021/ES400083D</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '498',
    'Kanakidou et&nbsp;al. (1995)',
    'Kanakidou, M., Dentener, F. J., &#x0026; Crutzen, P. J.: <span class="cmti-10">A global three-dimensional study of</span> <span class="cmti-10">the fate of</span> HCFC<span class="cmti-10">s and</span> HFC<span class="cmsy-10">−</span>134a <span class="cmti-10">in the troposphere</span>, J. Geophys. Res., 100, 18 781–18 801, <a href="https://doi.org/10.1029/95JD01919" >doi:10.1029/95JD01919</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2571',
    'Kanefke (2008)',
    'Kanefke, R.: Durch Quecksilberbromierung verbesserte Quecksilberabscheidung aus den Abgasen von Kohlekraftwerken und Abfallverbrennungsanlagen, Ph.D. thesis, Martin-Luther-Universität Halle-Wittenberg, Germany, ISBN 978-3-8322-7241-8 (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3631',
    'Kaneko et&nbsp;al. (1994)',
    'Kaneko, T., Wang, P. Y., &#x0026; Sato, A.: <span class="cmti-10">Partition coefficients of some acetate esters and alcohols in</span> <span class="cmti-10">water, blood, olive oil, and rat tissues</span>, Occup. Environ. Med., 51, 68–72, <a href="https://doi.org/10.1136/OEM.51.1.68" >doi:10.1136/OEM.51.1.68</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3753',
    'Karagodin-Doyennel et&nbsp;al. (2021)',
    'Karagodin-Doyennel, A., Rozanov, E., Sukhodolov, T., Egorova, T., Saiz-Lopez, A., Cuevas, C. A., Fernandez, R. P., Sherwen, T., Volkamer, R., Koenig, T. K., Giroud, T., &#x0026; Peter, T.: <span class="cmti-10">Iodine chemistry in the</span> <span class="cmti-10">chemistry–climate model SOCOL-AERv2-I</span>, Geosci. Model Dev., 14, 6623–6645, <a href="https://doi.org/10.5194/GMD-14-6623-2021" >doi:10.5194/GMD-14-6623-2021</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1922',
    'Karl et&nbsp;al. (2003)',
    'Karl, T., Yeretzian, C., Jordan, A., &#x0026; Lindinger, W.: <span class="cmti-10">Dynamic</span> <span class="cmti-10">measurements of partition coefficients using proton-transfer-reaction</span> <span class="cmti-10">mass spectrometry (PTR-MS)</span>, Int. J. Mass Spectrom., 223-224, 383–395, <a href="https://doi.org/10.1016/S1387-3806(02)00927-2" >doi:10.1016/S1387-3806(02)00927-2</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3555',
    'Karl et&nbsp;al. (2008)',
    'Karl, T., Guenther, A., Turnipseed, A., Patton, E. G., &#x0026; Jardine, K.: <span class="cmti-10">Chemical</span> <span class="cmti-10">sensing of plant stress at the ecosystem scale</span>, Biogeosci., 5, 1287–1294, <a href="https://doi.org/10.5194/BG-5-1287-2008" >doi:10.5194/BG-5-1287-2008</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1323',
    'Katrib et&nbsp;al. (2001)',
    'Katrib, Y., Deiber, G., Schweitzer, F., Mirabel, P., &#x0026; George, C.: <span class="cmti-10">Chemical transformation of bromine chloride at the air/water interface</span>, J. Aerosol Sci., 32, 893–911, <a href="https://doi.org/10.1016/S0021-8502(00)00114-2" >doi:10.1016/S0021-8502(00)00114-2</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1940',
    'Katrib et&nbsp;al. (2003)',
    'Katrib, Y., Calve, S. L., &#x0026; Mirabel, P.: <span class="cmti-10">Uptake measurements of dibasic esters by water droplets and</span> <span class="cmti-10">determination of their Henry&#x2019;s law constants</span>, J. Phys. Chem. A, 107, 11 433–11 439, <a href="https://doi.org/10.1021/JP0368132" >doi:10.1021/JP0368132</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3764',
    'Katritzky et&nbsp;al. (1998)',
    'Katritzky, A. R., Wang, Y., Sild, S., Tamm, T., &#x0026; Karelson, M.: <span class="cmti-10">QSPR</span> <span class="cmti-10">studies on vapor pressure, aqueous solubility, and the prediction of</span> <span class="cmti-10">water-air partition coefficients</span>, J. Chem. Inf. Comput. Sci., 38, 720–725, <a href="https://doi.org/10.1021/CI980022T" >doi:10.1021/CI980022T</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2229',
    'Kawamoto and Urano (1989)',
    'Kawamoto, K. &#x0026; Urano, K.: <span class="cmti-10">Parameters for</span> <span class="cmti-10">predicting fate of organochlorine pesticides in the environment (I)</span> <span class="cmti-10">Octanol-water and air–water partition coefficients</span>, Chemosphere, 18, 1987–1996, <a href="https://doi.org/10.1016/0045-6535(89)90482-7" >doi:10.1016/0045-6535(89)90482-7</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2906',
    'Keeley et&nbsp;al. (1988)',
    'Keeley, D. F., Hoffpauir, M. A., &#x0026; Meriwether, J. R.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">aromatic hydrocarbons in water and sodium chloride solutions of different</span> <span class="cmti-10">ionic strengths: benzene and toluene</span>, Environ. Sci. Technol., 33, 87–89, <a href="https://doi.org/10.1021/JE00052A006" >doi:10.1021/JE00052A006</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '508',
    'Keene and Galloway (1986)',
    'Keene, W. C. &#x0026; Galloway, J. N.: <span class="cmti-10">Considerations regarding sources for formic</span> <span class="cmti-10">and acetic acids in the troposphere</span>, J. Geophys. Res., 91, 14 466–14 474, <a href="https://doi.org/10.1029/JD091ID13P14466" >doi:10.1029/JD091ID13P14466</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '538',
    'Keene et&nbsp;al. (1995)',
    'Keene, W. C., Mosher, B. W., Jacob, D. J., Munger, J. W., Talbot, R. W., Artz, R. S., Maben, J. R., Daube, B. C., &#x0026; Galloway, J. N.: <span class="cmti-10">Carboxylic acids in a high-elevation forested site in central Virginia</span>, J. Geophys. Res., 100, 9345–9357, <a href="https://doi.org/10.1029/94JD01247" >doi:10.1029/94JD01247</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '626',
    'Kelley and Tartar (1956)',
    'Kelley, C. M. &#x0026; Tartar, H. V.: <span class="cmti-10">On the system: bromine-water</span>, J. Am. Chem. Soc., 78, 5752–5756, <a href="https://doi.org/10.1021/JA01603A010" >doi:10.1021/JA01603A010</a> (1956). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3956',
    'Kepinski and Trzeszczynski (1964)',
    'Kepinski, J. &#x0026; Trzeszczynski, J.: <span class="cmti-10">Absorption equilibria of chlorine dioxide. Solubility</span> <span class="cmti-10">in water, carbon tetrachloride, sulphuric and acetic acid solutions</span>, Rocz. Chem., 38, 201–211 (1964). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3730',
    'Keshavarz et&nbsp;al. (2022)',
    'Keshavarz, M. H., Rezaei, M., &#x0026; Hosseini, S. H.: <span class="cmti-10">A simple approach</span> <span class="cmti-10">for prediction of Henry&#x2019;s law constant of pesticides, solvents, aromatic</span> <span class="cmti-10">hydrocarbons, and persistent pollutants without using complex computer</span> <span class="cmti-10">codes and descriptors</span>, Process Saf. Environ. Prot., 162, 867–877, <a href="https://doi.org/10.1016/J.PSEP.2022.04.045" >doi:10.1016/J.PSEP.2022.04.045</a> (2022). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3248',
    'Keßel et&nbsp;al. (2017)',
    'Keßel, S., Cabrera-Perez, D., Horowitz, A., Veres, P. R., Sander, R., Taraborrelli, D., Tucceri, M., Crowley, J. N., Pozzer, A., Stönner, C., Vereecken, L., Lelieveld, J., &#x0026; Williams, J.: <span class="cmti-10">Atmospheric chemistry,</span> <span class="cmti-10">sources and sinks of carbon suboxide,</span> C<sub><span class="cmr-7">3</span></sub>O<sub><span class="cmr-7">2</span></sub>, Atmos. Chem. Phys., 17, 8789–8804, <a href="https://doi.org/10.5194/ACP-17-8789-2017" >doi:10.5194/ACP-17-8789-2017</a> (2017). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2895',
    'Khalfaoui and Newsham (1994a)',
    'Khalfaoui, B. &#x0026; Newsham, D. M. T.: <span class="cmti-10">Phase equilibria in very dilute mixtures of</span> <span class="cmti-10">water and brominated hydrocarbons</span>, Fluid Phase Equilib., 98, 213–223, <a href="https://doi.org/10.1016/0378-3812(94)80120-7" >doi:10.1016/0378-3812(94)80120-7</a> (1994a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2931',
    'Khalfaoui and Newsham (1994b)',
    'Khalfaoui, B. &#x0026; Newsham, D. M. T.: <span class="cmti-10">Determination of infinite dilution activity coefficients and second virial</span> <span class="cmti-10">coefficients using gas-liquid chromatography I. The dilute mixtures of</span> <span class="cmti-10">water and unsaturated chlorinated</span> <span class="cmti-10">hydrocarbons and of water and benzene</span>, J. Chromatogr. A, 673, 85–92, <a href="https://doi.org/10.1016/0021-9673(94)87060-8" >doi:10.1016/0021-9673(94)87060-8</a> (1994b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2230',
    'Khan and Brimblecombe (1992)',
    'Khan, I. &#x0026; Brimblecombe, P.: <span class="cmti-10">Henry&#x2019;s law</span> <span class="cmti-10">constants of low molecular weight (</span><span class="cmmi-10">&#x003C;</span><span class="cmti-10">130) organic acids</span>, J. Aerosol Sci., 23, S897–S900, <a href="https://doi.org/10.1016/0021-8502(92)90556-B" >doi:10.1016/0021-8502(92)90556-B</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1063',
    'Khan et&nbsp;al. (1992)',
    'Khan, I., Brimblecombe, P., &#x0026; Clegg, S. L.: <span class="cmti-10">The Henry&#x2019;s law constants</span> <span class="cmti-10">of pyruvic and methacrylic acids</span>, Environ. Technol., 13, 587–593, <a href="https://doi.org/10.1080/09593339209385187" >doi:10.1080/09593339209385187</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '493',
    'Khan et&nbsp;al. (1995)',
    'Khan, I., Brimblecombe, P., &#x0026; Clegg, S. L.: <span class="cmti-10">Solubilities of pyruvic acid</span> <span class="cmti-10">and the lower (</span>C<sub><span class="cmr-7">1</span></sub><span class="cmti-10">-</span>C<sub><span class="cmr-7">6</span></sub><span class="cmti-10">) carboxylic acids. Experimental determination of</span> <span class="cmti-10">equilibrium vapour pressures above pure aqueous and salt solutions</span>, J. Atmos. Chem., 22, 285–302, <a href="https://doi.org/10.1007/BF00696639" >doi:10.1007/BF00696639</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3588',
    'Kieckbusch and King (1979a)',
    'Kieckbusch, T. G. &#x0026; King, C. J.: <span class="cmti-10">Partition-coefficients for acetates in food systems</span>, J. Agric. Food Chem., 27, 504–507, <a href="https://doi.org/10.1021/JF60223A033" >doi:10.1021/JF60223A033</a> (1979a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '628',
    'Kieckbusch and King (1979b)',
    'Kieckbusch, T. G. &#x0026; King, C. J.: <span class="cmti-10">An</span> <span class="cmti-10">improved method of determining vapor liquid equilibria for dilute</span> <span class="cmti-10">organics in aqueous solution</span>, J. Chromatogr. Sci., 17, 273–276, <a href="https://doi.org/10.1093/CHROMSCI/17.5.273" >doi:10.1093/CHROMSCI/17.5.273</a> (1979b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3593',
    'Kilpatrick et&nbsp;al. (1956)',
    'Kilpatrick, M. L., Herrick, C. C., &#x0026; Kilpatrick, M.: <span class="cmti-10">The decomposition</span> <span class="cmti-10">of ozone in aqueous solution</span>, J. Am. Chem. Soc., 78, 1784–1789, <a href="https://doi.org/10.1021/JA01590A003" >doi:10.1021/JA01590A003</a> (1956). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2837',
    'Kim et&nbsp;al. (2000)',
    'Kim, B. R., Kalis, E. M., DeWulf, T., &#x0026; Andrews, K. M.: <span class="cmti-10">Henry&#x2019;s Law</span> <span class="cmti-10">constants for paint solvents and their implications on volatile organic</span> <span class="cmti-10">compound emissions from automotive painting</span>, Water Environ. Res., 72, 65–74, <a href="https://doi.org/10.2175/106143000X137121" >doi:10.2175/106143000X137121</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3654',
    'Kim et&nbsp;al. (2008)',
    'Kim, I., Svendsen, H. F., &#x0026; Børresen, E.: <span class="cmti-10">Ebulliometric determination of vapor-liquid equilibria for</span> <span class="cmti-10">pure water, monoethanolamine, N-methyldiethanolamine,</span> <span class="cmti-10">3-(methylamino)-propylamine, and their binary and ternary solutions</span>, J. Chem. Eng. Data, 53, 2521–2531, <a href="https://doi.org/10.1021/JE800290K" >doi:10.1021/JE800290K</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3032',
    'Kim and Kim (2014)',
    'Kim, Y.-H. &#x0026; Kim, K.-H.: <span class="cmti-10">Recent advances in thermal</span> <span class="cmti-10">desorption-gas chromatography-mass spectrometery method to eliminate</span> <span class="cmti-10">the matrix effect between air and water samples: Application to the</span> <span class="cmti-10">accurate determination of Henry&#x2019;s law constant</span>, J. Chromatogr. A, 1342, 78–85, <a href="https://doi.org/10.1016/J.CHROMA.2014.03.040" >doi:10.1016/J.CHROMA.2014.03.040</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3684',
    'Kim and Kim (2016)',
    'Kim, Y.-H. &#x0026; Kim, K.-H.: <span class="cmti-10">A simple method for the</span> <span class="cmti-10">accurate determination of the Henry&#x2019;s law constant for highly sorptive,</span> <span class="cmti-10">semivolatile organic compounds</span>, Anal. Bioanal. Chem., 408, 775–784, <a href="https://doi.org/10.1007/S00216-015-9159-3" >doi:10.1007/S00216-015-9159-3</a> (2016). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3007',
    'Kish et&nbsp;al. (2013)',
    'Kish, J. D., Leng, C. B., Kelley, J., Hiltner, J., Zhang, Y. H., &#x0026; Liu, Y.: <span class="cmti-10">An improved</span> <span class="cmti-10">approach for measuring Henry&#x2019;s law coefficients of atmospheric organics</span>, Atmos. Environ., 79, 561–565, <a href="https://doi.org/10.1016/J.ATMOSENV.2013.07.023" >doi:10.1016/J.ATMOSENV.2013.07.023</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3080',
    'Klein (1982)',
    'Klein, R. G.: <span class="cmti-10">Calculations and measurements on the volatility</span> <span class="cmti-10">of N-nitrosamines and their aqueous solutions</span>, Toxicology, 23, 135–147, <a href="https://doi.org/10.1016/0300-483X(82)90093-2" >doi:10.1016/0300-483X(82)90093-2</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3692',
    'Knauss et&nbsp;al. (2000)',
    'Knauss, K. G., Dibley, M. J., Leif, R. N., Mew, D. A., &#x0026; Aines, R. D.: <span class="cmti-10">The aqueous solubility of trichloroethene (TCE) and tetrachloroethene</span> <span class="cmti-10">(PCE) as a function of temperature</span>, Appl. Geochem., 15, 501–512, <a href="https://doi.org/10.1016/S0883-2927(99)00058-X" >doi:10.1016/S0883-2927(99)00058-X</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3081',
    'Kochetkov et&nbsp;al. (2001)',
    'Kochetkov, A., Smith, J. S., Ravikrishna, R., Valsaraj, K. T., &#x0026; Thibodeaux, L. J.: <span class="cmti-10">Air–water partition constants for volatile</span> <span class="cmti-10">methyl siloxanes</span>, Environ. Toxicol. Chem., 20, 2184–2188, <a href="https://doi.org/10.1002/ETC.5620201008" >doi:10.1002/ETC.5620201008</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '542',
    'Koga (1995)',
    'Koga, Y.: <span class="cmti-10">Vapor pressures of dilute aqueous </span><span class="cmmi-10">t</span><span class="cmti-10">-butyl alcohol:</span> <span class="cmti-10">How dilute is the Henry&#x2019;s law region?</span>, J. Phys. Chem., 99, 6231–6233, <a href="https://doi.org/10.1021/J100016A069" >doi:10.1021/J100016A069</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '857',
    'Kolb et&nbsp;al. (1992)',
    'Kolb, B., Welter, C., &#x0026; Bichler, C.: <span class="cmti-10">Determination of partition coefficients by automatic</span> <span class="cmti-10">equilibrium headspace gas chromatography by vapor phase calibration</span>, Chromatographia, 34, 235–240, <a href="https://doi.org/10.1007/BF02268351" >doi:10.1007/BF02268351</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1138',
    'Komiyama and Inoue (1978)',
    'Komiyama, H. &#x0026; Inoue, H.: <span class="cmti-10">Reaction and</span> <span class="cmti-10">transport of nitrogen oxides in nitrous acid solutions</span>, J. Chem. Eng. Jpn., 11, 25–32, <a href="https://doi.org/10.1252/JCEJ.11.25" >doi:10.1252/JCEJ.11.25</a> (1978). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3120',
    'Komiyama and Inoue (1980)',
    'Komiyama, H. &#x0026; Inoue, H.: <span class="cmti-10">Absorption</span> <span class="cmti-10">of nitrogen oxides into water</span>, Chem. Eng. Sci., 35, 154–161, <a href="https://doi.org/10.1016/0009-2509(80)80082-0" >doi:10.1016/0009-2509(80)80082-0</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2904',
    'Kondoh and Nakajima (1997)',
    'Kondoh, H. &#x0026; Nakajima, T.: <span class="cmti-10">Optimization</span> <span class="cmti-10">of headspace cryofocus gas chromatography/mass spectrometry for the</span> <span class="cmti-10">analysis of 54 volatile organic compounds, and the measurement of their</span> <span class="cmti-10">Henry&#x2019;s constants</span>, J. Environ. Chem., 7, 81–89, <a href="https://doi.org/10.5985/JEC.7.81" >doi:10.5985/JEC.7.81</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '480',
    'Kosak-Channing and Helz (1983)',
    'Kosak-Channing, L. F. &#x0026; Helz, G. R.: <span class="cmti-10">Solubility of ozone in aqueous solutions of 0–0.6 </span>M <span class="cmti-10">ionic strength at</span> <span class="cmti-10">5–30</span><span class="cmsy-7">∘</span>C, Environ. Sci. Technol., 17, 145–149, <a href="https://doi.org/10.1021/ES00109A005" >doi:10.1021/ES00109A005</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'kotlik74',
    'Kotlik and Lebedeva (1974)',
    'Kotlik, S. B. &#x0026; Lebedeva, G. N.: <span class="cmti-10">Equilibrium</span> <span class="cmti-10">pressures of</span> HCN <span class="cmti-10">and</span> NH<sub><span class="cmr-7">3</span></sub> <span class="cmti-10">over aqueous solutions</span>, Zh. Prikl. Khim., 47, 444–446 (1974). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3123',
    'Kramers et&nbsp;al. (1961)',
    'Kramers, H., Blind, M. P. P., &#x0026; Snoeck, E.: <span class="cmti-10">Absorption of nitrogen tetroxide by water jets</span>, Chem. Eng. Sci., 14, 115–123, <a href="https://doi.org/10.1016/0009-2509(61)85062-8" >doi:10.1016/0009-2509(61)85062-8</a> (1961). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2436',
    'Krause and Benson (1989)',
    'Krause, Jr., D. &#x0026; Benson, B. B.: <span class="cmti-10">The solubility</span> <span class="cmti-10">and isotopic fractionation of gases in dilute aqueous solution. IIa.</span> <span class="cmti-10">solubilities of the noble gases</span>, J. Solution Chem., 18, 823–873, <a href="https://doi.org/10.1007/BF00685062" >doi:10.1007/BF00685062</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3749',
    'Kremann and Hönel (1913)',
    'Kremann, R. &#x0026; Hönel, H.: <span class="cmti-10">Über die L</span><span class="cmti-10">öslichkeit</span> <span class="cmti-10">von Acetylen in Aceton und Aceton-Wassergemischen</span>, Monatsh. Chem. – Chem. Mon., 34, 1089–1094, <a href="https://doi.org/10.1007/BF01517552" >doi:10.1007/BF01517552</a> (1913). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2321',
    'Kroll et&nbsp;al. (2005)',
    'Kroll, J. H., Ng, N. L., Murphy, S. M., Varutbangkul, V., Flagan, R. C., &#x0026; Seinfeld, J. H.: <span class="cmti-10">Chamber studies of secondary organic aerosol</span> <span class="cmti-10">growth by reactive uptake of simple carbonyl compounds</span>, J. Geophys. Res., 110, D23207, <a href="https://doi.org/10.1029/2005JD006004" >doi:10.1029/2005JD006004</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '977',
    'Krop et&nbsp;al. (1997)',
    'Krop, H. B., van Velzen, M. J. M., Parsons, J. R., &#x0026; Govers, H. A. J.: <span class="cmti-10">n-Octanol-water</span> <span class="cmti-10">partition coefficients, aqueous</span> <span class="cmti-10">solubilities and Henry&#x2019;s law constants of fatty acid esters</span>, Chemosphere, 34, 107–119, <a href="https://doi.org/10.1016/S0045-6535(96)00371-2" >doi:10.1016/S0045-6535(96)00371-2</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2927',
    'Krysztofiak et&nbsp;al. (2012)',
    'Krysztofiak, G., Catoire, V., Poulet, G., Marécal, V., Pirre, M., Louis, F., Canneaux, S., &#x0026; Josse, B.: <span class="cmti-10">Detailed modeling of the atmospheric</span> <span class="cmti-10">degradation mechanism of very-short lived brominated species</span>, Atmos. Environ., 59, 514–532, <a href="https://doi.org/10.1016/J.ATMOSENV.2012.05.026" >doi:10.1016/J.ATMOSENV.2012.05.026</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1152',
    'Kucklick et&nbsp;al. (1991)',
    'Kucklick, J. R., Hinckley, D. A., &#x0026; Bidleman, T. F.: <span class="cmti-10">Determination of</span> <span class="cmti-10">Henry&#x2019;s law constants for hexachlorocyclohexanes in distilled water and</span> <span class="cmti-10">artificial seawater as a function of temperature</span>, Mar. Chem., 34, 197–209, <a href="https://doi.org/10.1016/0304-4203(91)90003-F" >doi:10.1016/0304-4203(91)90003-F</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1917',
    'Kühne et&nbsp;al. (2005)',
    'Kühne, R., Ebert, R.-U., &#x0026; Schüürmann, G.: <span class="cmti-10">Prediction of the temperature dependency of</span> <span class="cmti-10">Henry&#x2019;s law constant from chemical structure</span>, Environ. Sci. Technol., 39, 6705–6711, <a href="https://doi.org/10.1021/ES050527H" >doi:10.1021/ES050527H</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3709',
    'Kunerth (1922)',
    'Kunerth, W.: <span class="cmti-10">Solubility of</span> CO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">and</span> N<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">in certain</span> <span class="cmti-10">solvents</span>, Phys. Rev., 19, 512–524, <a href="https://doi.org/10.1103/PHYSREV.19.512" >doi:10.1103/PHYSREV.19.512</a> (1922). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1932',
    'Kuramochi et&nbsp;al. (2004)',
    'Kuramochi, H., Maeda, K., &#x0026; Kawamoto, K.: <span class="cmti-10">Measurements of water solubilities and</span> <span class="cmti-10">1-octanol/water partition coefficients and estimations of Henry&#x2019;s law</span> <span class="cmti-10">constants for brominated benzenes</span>, J. Chem. Eng. Data, 49, 720–724, <a href="https://doi.org/10.1021/JE0342724" >doi:10.1021/JE0342724</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3024',
    'Kuramochi et&nbsp;al. (2014)',
    'Kuramochi, H., Takigami, H., Scheringer, M., &#x0026; Sakai, S.: <span class="cmti-10">Measurement</span> <span class="cmti-10">of vapor pressures of selected PBDEs, hexabromobenzene, and</span> <span class="cmti-10">1,2-bis(2,4,6-tribromophenoxy)ethane at elevated temperatures</span>, J. Chem. Eng. Data, 59, 8–15, <a href="https://doi.org/10.1021/JE400520E" >doi:10.1021/JE400520E</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3099',
    'Kurtén et&nbsp;al. (2015)',
    'Kurtén, T., Elm, J., Prisle, N. L., Mikkelsen, K. V., Kampf, C. J., Waxman, E. M., &#x0026; Volkamer, R.: <span class="cmti-10">Computational study of the effect of</span> <span class="cmti-10">glyoxal-sulfate clustering on the Henry&#x2019;s law coefficient of glyoxal</span>, J. Phys. Chem. A, 119, 4509–4514, <a href="https://doi.org/10.1021/JP510304C" >doi:10.1021/JP510304C</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1096',
    'Kurz and Ballschmiter (1999)',
    'Kurz, J. &#x0026; Ballschmiter, K.: <span class="cmti-10">Vapour pressures,</span> <span class="cmti-10">aqueous solubilities, Henry&#x2019;s law constants, partition coefficients between</span> <span class="cmti-10">gas/water (</span><span class="cmmi-10">K</span><sub><span class="cmr-7">gw</span></sub><span class="cmti-10">), </span><span class="cmmi-10">n</span><span class="cmti-10">-octanol/water (</span><span class="cmmi-10">K</span><sub><span class="cmr-7">ow</span></sub><span class="cmti-10">) and gas/n-octanol (</span><span class="cmmi-10">K</span><sub><span class="cmr-7">go</span></sub><span class="cmti-10">) of</span> <span class="cmti-10">106 polychlorinated diphenyl ethers (PCDE)</span>, Chemosphere, 38, 573–586, <a href="https://doi.org/10.1016/S0045-6535(98)00212-4" >doi:10.1016/S0045-6535(98)00212-4</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2939',
    'Kutsuna (2013)',
    'Kutsuna, S.: <span class="cmti-10">Determination of rate constants for aqueous</span> <span class="cmti-10">reactions of HCFC-123 and HCFC-225ca with</span> OH<sup><span class="cmsy-7">−</span></sup> <span class="cmti-10">along with Henry&#x2019;s</span> <span class="cmti-10">law constants of several HCFCs</span>, Int. J. Chem. Kinet., 45, 440–451, <a href="https://doi.org/10.1002/KIN.20780" >doi:10.1002/KIN.20780</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3272',
    'Kutsuna (2017)',
    'Kutsuna, S.: <span class="cmti-10">Experimental determination of Henry&#x2019;s law</span> <span class="cmti-10">constants of difluoromethane (HFC-32) and the salting-out effects in</span> <span class="cmti-10">aqueous salt solutions relevant to seawater</span>, Atmos. Chem. Phys., 17, 7495–7507, <a href="https://doi.org/10.5194/ACP-17-7495-2017" >doi:10.5194/ACP-17-7495-2017</a> (2017). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2406',
    'Kutsuna and Hori (2008a)',
    'Kutsuna, S. &#x0026; Hori, H.: <span class="cmti-10">Experimental determination of Henry&#x2019;s law constants</span> <span class="cmti-10">of trifluoroacetic acid at 278-298 </span>K, Atmos. Environ., 42, 1399–1412, <a href="https://doi.org/10.1016/J.ATMOSENV.2007.11.009" >doi:10.1016/J.ATMOSENV.2007.11.009</a> (2008a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2816',
    'Kutsuna and Hori (2008b)',
    'Kutsuna, S. &#x0026; Hori, H.: <span class="cmti-10">Experimental</span> <span class="cmti-10">determination of Henry&#x2019;s law constant of perfluorooctanoic acid (PFOA)</span> <span class="cmti-10">at 298 </span>K <span class="cmti-10">by means of an inert-gas stripping method with a helical plate</span>, Atmos. Environ., 42, 8883–8892, <a href="https://doi.org/10.1016/J.ATMOSENV.2008.09.008" >doi:10.1016/J.ATMOSENV.2008.09.008</a> (2008b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3507',
    'Kutsuna and Kaneyasu (2021)',
    'Kutsuna, S. &#x0026; Kaneyasu, N.: <span class="cmti-10">Henry&#x2019;s law</span> <span class="cmti-10">constants and hydration equilibrium constants of </span><span class="cmmi-10">n</span><span class="cmti-10">-hexanal and their</span> <span class="cmti-10">temperature dependence as determined by the rectangular pulse method</span>, Chem. Eng. Sci., 239, 116 639, <a href="https://doi.org/10.1016/J.CES.2021.116639" >doi:10.1016/J.CES.2021.116639</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2803',
    'Kutsuna et&nbsp;al. (2004)',
    'Kutsuna, S., Chen, L., Ohno, K., Tokuhashi, K., &#x0026; Sekiya, A.: <span class="cmti-10">Henry&#x2019;s law constants and hydrolysis rate constants of 2,2,2-trifluoroethyl</span> <span class="cmti-10">acetate and methyl trifluoroacetate</span>, Atmos. Environ., 38, 725–732, <a href="https://doi.org/10.1016/J.ATMOSENV.2003.10.019" >doi:10.1016/J.ATMOSENV.2003.10.019</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2802',
    'Kutsuna et&nbsp;al. (2005)',
    'Kutsuna, S., Chen, L., Abe, T., Mizukado, J., Uchimaru, T., Tokuhashi, K., &#x0026; Sekiya, A.: <span class="cmti-10">Henry&#x2019;s law constants of 2,2,2-trifluoroethyl formate,</span> <span class="cmti-10">ethyl trifluoroacetate, and non-fluorinated analogous esters</span>, Atmos. Environ., 39, 5884–5892, <a href="https://doi.org/10.1016/J.ATMOSENV.2005.06.021" >doi:10.1016/J.ATMOSENV.2005.06.021</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3940',
    'Kwan (2001)',
    'Kwan, W. C.: Physical property determination of perfluorinated surfactants, Master&#x2019;s thesis, University of Toronto, URL https://hdl.handle.net/1807/16278 (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2901',
    'Lamarche and Droste (1989)',
    'Lamarche, P. &#x0026; Droste, R. L.: <span class="cmti-10">Air stripping</span> <span class="cmti-10">mass transfer correlations for volatile organics</span>, J. Am. Water Works Assoc., 81, 78–89, <a href="https://doi.org/10.1002/J.1551-8833.1989.TB03326.X" >doi:10.1002/J.1551-8833.1989.TB03326.X</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3647',
    'Landy et&nbsp;al. (1995)',
    'Landy, P., Druaux, C., &#x0026; A.Voilley: <span class="cmti-10">Retention of aroma compounds by proteins in aqueous</span> <span class="cmti-10">solution</span>, Food Chem., 54, 387–392, <a href="https://doi.org/10.1016/0308-8146(95)00069-U" >doi:10.1016/0308-8146(95)00069-U</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3651',
    'Landy et&nbsp;al. (1996)',
    'Landy, P., Courthaudon, J.-L., Dubois, C., &#x0026; Voilley, A.: <span class="cmti-10">Effect of interface in</span> <span class="cmti-10">model food emulsions on the volatility of aroma compounds</span>, J. Agric. Food Chem., 44, 526–530, <a href="https://doi.org/10.1021/JF950279G" >doi:10.1021/JF950279G</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3789',
    'Lannung (1930)',
    'Lannung, A.: <span class="cmti-10">The solubilities of helium, neon and argon</span> <span class="cmti-10">in water and some organic solvents</span>, J. Am. Chem. Soc., 52, 68–80, <a href="https://doi.org/10.1021/JA01364A011" >doi:10.1021/JA01364A011</a> (1930). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '499',
    'Latimer (1952)',
    'Latimer, W. M.: The Oxidation States of the Elements and their Potentials in Aqueous Solutions, Prentice-Hall, Englewood Cliffs, NJ (1952). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2564',
    'Lau et&nbsp;al. (2006)',
    'Lau, F. K., Charles, M. J., &#x0026; Cahill, T. M.: <span class="cmti-10">Evaluation of gas-stripping methods for</span> <span class="cmti-10">the determination of Henry&#x2019;s law constants for polybrominated diphenyl</span> <span class="cmti-10">ethers and polychlorinated biphenyls</span>, J. Chem. Eng. Data, 51, 871–878, <a href="https://doi.org/10.1021/JE050308B" >doi:10.1021/JE050308B</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2982',
    'Lau et&nbsp;al. (2010)',
    'Lau, K., Rogers, T. N., &#x0026; Chesney, D. J.: <span class="cmti-10">Measuring the aqueous Henry&#x2019;s law constant at elevated</span> <span class="cmti-10">temperatures using an extended EPICS technique</span>, J. Chem. Eng. Data, 55, 5144–5148, <a href="https://doi.org/10.1021/JE100701W" >doi:10.1021/JE100701W</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3947',
    'Lau et&nbsp;al. (1995)',
    'Lau, Y. L., Liu, D. L. S., Pacepavicius, G. J., &#x0026; Maguire, R. J.: <span class="cmti-10">Volatilization of</span> <span class="cmti-10">metolachlor from water</span>, J. Environ. Sci. Health B, 30, 605–620, <a href="https://doi.org/10.1080/03601239509372956" >doi:10.1080/03601239509372956</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3929',
    'Leaist (1986)',
    'Leaist, D. G.: <span class="cmti-10">Absorption of chlorine into water</span>, J. Solution Chem., 15, 827–838, <a href="https://doi.org/10.1007/BF00646090" >doi:10.1007/BF00646090</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '481',
    'Ledbury and Blair (1925)',
    'Ledbury, W. &#x0026; Blair, E. W.: <span class="cmti-10">The partial</span> <span class="cmti-10">formaldehyde vapour pressures of aqueous solutions of formaldehyde.</span> <span class="cmti-10">Part II</span>, J. Chem. Soc., 127, 2832–2839, <a href="https://doi.org/10.1039/CT9252702832" >doi:10.1039/CT9252702832</a> (1925). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2932',
    'Lee (2007)',
    'Lee, F. F.: Comprehensive analysis, Henry&#x2019;s law constant determination, and photocatalytic degradation of polychlorinated biphenyls (PCBs) and/or other persistent organic pollutants (POPs), Ph.D. thesis, University at Albany, State University of New York, USA, ISBN 978-0-549-42141-2 (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3014',
    'Lee et&nbsp;al. (2012)',
    'Lee, H., Kim, H.-J., &#x0026; Kwon, J.-H.: <span class="cmti-10">Determination of Henry&#x2019;s law constant using diffusion in air</span> <span class="cmti-10">and water boundary layers</span>, J. Chem. Eng. Data, 57, 3296–3302, <a href="https://doi.org/10.1021/JE300954S" >doi:10.1021/JE300954S</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2987',
    'Lee et&nbsp;al. (2013)',
    'Lee, S.-H., Mukherjee, S., Brewer, B., Ryan, R., Yu, H., &#x0026; Gangoda, M.: <span class="cmti-10">A laboratory experiment to measure Henry&#x2019;s law constants of volatile</span> <span class="cmti-10">organic compounds with a bubble column and a gas chromatography</span> <span class="cmti-10">flame ionization detector (GC-FID)</span>, J. Chem. Educ., 90, 495–499, <a href="https://doi.org/10.1021/ED200303X" >doi:10.1021/ED200303X</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '59',
    'Lee and Schwartz (1981)',
    'Lee, Y.-N. &#x0026; Schwartz, S. E.: <span class="cmti-10">Reaction kinetics of</span> <span class="cmti-10">nitrogen dioxide with liquid water at low partial pressure</span>, J. Phys. Chem., 85, 840–848, <a href="https://doi.org/10.1021/J150607A022" >doi:10.1021/J150607A022</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2527',
    'Lee and Zhou (1993)',
    'Lee, Y.-N. &#x0026; Zhou, X.: <span class="cmti-10">Method for the determination</span> <span class="cmti-10">of some soluble atmospheric carbonyl compounds</span>, Environ. Sci. Technol., 27, 749–756, <a href="https://doi.org/10.1021/ES00041A020" >doi:10.1021/ES00041A020</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1124',
    'Lee and Zhou (1994)',
    'Lee, Y.-N. &#x0026; Zhou, X.: <span class="cmti-10">Aqueous reaction kinetics of</span> <span class="cmti-10">ozone and dimethylsulfide and its atmospheric implications</span>, J. Geophys. Res., 99, 3597–3605, <a href="https://doi.org/10.1029/93JD02919" >doi:10.1029/93JD02919</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2960',
    'Lei et&nbsp;al. (1999)',
    'Lei, Y. D., Wania, F., Shiu, W. Y., &#x0026; Boocock, D. G. B.: <span class="cmti-10">Temperature dependent vapor pressures</span> <span class="cmti-10">of chlorinated catechols, syringols, and syringaldehydes</span>, J. Chem. Eng. Data, 44, 200–202, <a href="https://doi.org/10.1021/JE9801819" >doi:10.1021/JE9801819</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2996',
    'Lei et&nbsp;al. (2004)',
    'Lei, Y. D., Wania, F., Mathers, D., &#x0026; Mabury, S. A.: <span class="cmti-10">Determination of vapor pressures,</span> <span class="cmti-10">octanol-air, and water-air partition coefficients for polyfluorinated</span> <span class="cmti-10">sulfonamide, sulfonamidoethanols, and telomer alcohols</span>, J. Chem. Eng. Data, 49, 1013–1022, <a href="https://doi.org/10.1021/JE049949H" >doi:10.1021/JE049949H</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2997',
    'Lei et&nbsp;al. (2007)',
    'Lei, Y. D., Shunthirasingham, C., &#x0026; Wania, F.: <span class="cmti-10">Comparison of headspace</span> <span class="cmti-10">and gas-stripping techniques for measuring the air–water partititioning</span> <span class="cmti-10">of normal alkanols (C4 to C10) – effect of temperature, chain length</span> <span class="cmti-10">and adsorption to the water surface</span>, J. Chem. Eng. Data, 52, 168–179, <a href="https://doi.org/10.1021/JE060344Q" >doi:10.1021/JE060344Q</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1150',
    'Leighton and Calo (1981)',
    'Leighton, D. T. &#x0026; Calo, J. M.: <span class="cmti-10">Distribution</span> <span class="cmti-10">coefficients of chlorinated hydrocarbons in dilute air–water systems</span> <span class="cmti-10">for groundwater contamination applications</span>, J. Chem. Eng. Data, 26, 382–385, <a href="https://doi.org/10.1021/JE00026A010" >doi:10.1021/JE00026A010</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3086',
    'Leistra (1970)',
    'Leistra, M.: <span class="cmti-10">Distribution of 1,3-dichloropropene over the phases in soil</span>, J. Agric. Food Chem., 18, 1124–1126, <a href="https://doi.org/10.1021/JF60172A004" >doi:10.1021/JF60172A004</a> (1970). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3126',
    'Lekvam and Bishnoi (1997)',
    'Lekvam, K. &#x0026; Bishnoi, P. R.: <span class="cmti-10">Dissolution of</span> <span class="cmti-10">methane in water at low temperatures and intermediate pressures</span>, Fluid Phase Equilib., 131, 297–309, <a href="https://doi.org/10.1016/S0378-3812(96)03229-3" >doi:10.1016/S0378-3812(96)03229-3</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '31',
    'Lelieveld and Crutzen (1991)',
    'Lelieveld, J. &#x0026; Crutzen, P. J.: <span class="cmti-10">The role of</span> <span class="cmti-10">clouds in tropospheric photochemistry</span>, J. Atmos. Chem., 12, 229–267, <a href="https://doi.org/10.1007/BF00048075" >doi:10.1007/BF00048075</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3008',
    'Leng et&nbsp;al. (2013)',
    'Leng, C., Kish, J. D., Kelley, J., Mach, M., Hiltner, J., Zhang, Y., &#x0026; Liu, Y.: <span class="cmti-10">Temperature-dependent Henry&#x2019;s law constants of atmospheric</span> <span class="cmti-10">organics of biogenic origin</span>, J. Phys. Chem. A, 117, 10 359–10 367, <a href="https://doi.org/10.1021/JP403603Z" >doi:10.1021/JP403603Z</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3685',
    'Leng et&nbsp;al. (2015a)',
    'Leng, C., Kish, J. D., Roberts, J. E., Dwebi, I., Chon, N., &#x0026; Liu, Y.: <span class="cmti-10">Temperature-dependent Henry&#x2019;s law constants of atmospheric amines</span>, J. Phys. Chem. A, 119, 8884–8891, <a href="https://doi.org/10.1021/ACS.JPCA.5B05174" >doi:10.1021/ACS.JPCA.5B05174</a> (2015a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3686',
    'Leng et&nbsp;al. (2015b)',
    'Leng, C.-B., Roberts, J. E., Zeng, G., Zhang, Y.-H., &#x0026; Liu, Y.: <span class="cmti-10">Effects of temperature,</span> <span class="cmti-10">pH, and ionic strength on the Henry&#x2019;s law constant of triethylamine</span>, Geophys. Res. Lett., 42, 3569–3575, <a href="https://doi.org/10.1002/2015GL063840" >doi:10.1002/2015GL063840</a> (2015b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1442',
    'Leriche et&nbsp;al. (2000)',
    'Leriche, M., Voisin, D., Chaumerliac, N., Monod, A., &#x0026; Aumont, B.: <span class="cmti-10">A</span> <span class="cmti-10">model for tropospheric multiphase chemistry: application to one cloudy</span> <span class="cmti-10">event during the CIME experiment</span>, Atmos. Environ., 34, 5015–5036, <a href="https://doi.org/10.1016/S1352-2310(00)00329-0" >doi:10.1016/S1352-2310(00)00329-0</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2444',
    'Lerman et&nbsp;al. (1983)',
    'Lerman, J., Willis, M. M., Gregory, G. A., &#x0026; Eger, E. I.: <span class="cmti-10">Osmolarity determines the</span> <span class="cmti-10">solubility of anesthetics in aqueous solutions at 37</span><span class="cmsy-7">∘</span>C, Anesthesiology, 59, 554–558, <a href="https://doi.org/10.1097/00000542-198312000-00013" >doi:10.1097/00000542-198312000-00013</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1182',
    'Leu and Zhang (1999)',
    'Leu, M.-T. &#x0026; Zhang, R.: <span class="cmti-10">Solubilities</span> <span class="cmti-10">of</span> CH<sub><span class="cmr-7">3</span></sub>C(O)O<sub><span class="cmr-7">2</span></sub>NO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">and</span> HO<sub><span class="cmr-7">2</span></sub>NO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">in water and liquid</span> H<sub><span class="cmr-7">2</span></sub>SO<sub><span class="cmr-7">4</span></sub>, Geophys. Res. Lett., 26, 1129–1132, <a href="https://doi.org/10.1029/1999GL900158" >doi:10.1029/1999GL900158</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1068',
    'Leuenberger et&nbsp;al. (1985)',
    'Leuenberger, C., Ligocki, M. P., &#x0026; Pankow, J. F.: <span class="cmti-10">Trace organic compounds in</span> <span class="cmti-10">rain: 4. Identities, concentrations, and scavenging mechanisms for</span> <span class="cmti-10">phenols in urban air and rain</span>, Environ. Sci. Technol., 19, 1053–1058, <a href="https://doi.org/10.1021/ES00141A005" >doi:10.1021/ES00141A005</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3392',
    'Levanov et&nbsp;al. (2008)',
    'Levanov, A. V., Kuskov, I. V., Antipenko, E. E., &#x0026; Lunin, V. V.: <span class="cmti-10">The solubility</span> <span class="cmti-10">of ozone in aqueous solutions of sulfuric, phosphoric, and perchloric</span> <span class="cmti-10">acids</span>, Russ. J. Phys. Chem. A, 82, 1126–1131 (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3545',
    'Lewis et&nbsp;al. (1987)',
    'Lewis, C., Hopke, P. K., &#x0026; Stukel, J. J.: <span class="cmti-10">Solubility of radon in selected perfluorocarbon compounds</span> <span class="cmti-10">and water</span>, Ind. Eng. Chem. Res., 26, 356–359, <a href="https://doi.org/10.1021/IE00062A030" >doi:10.1021/IE00062A030</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2566',
    'Li et&nbsp;al. (2007)',
    'Li, H., Ellis, D., &#x0026; Mackay, D.: <span class="cmti-10">Measurement of low air–water partition coefficients of organic acids by</span> <span class="cmti-10">evaporation from a water surface</span>, J. Chem. Eng. Data, 52, 1580–1584, <a href="https://doi.org/10.1021/JE600556D" >doi:10.1021/JE600556D</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3689',
    'Li et&nbsp;al. (2014)',
    'Li, H., Wang, X., Yi, T., Xu, Z., &#x0026; Liu, X.: <span class="cmti-10">Prediction of Henry&#x2019;s law constants for organic compounds</span> <span class="cmti-10">using multilayer feedforward neural networks based on linear salvation</span> <span class="cmti-10">energy relationship</span>, J. Chem. Pharm. Res., 6, 1557–1564 (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2457',
    'Li and Carr (1993)',
    'Li, J. &#x0026; Carr, P. W.: <span class="cmti-10">Measurement of water-hexadecane</span> <span class="cmti-10">partition coefficients by headspace gas chromatography and calculation</span> <span class="cmti-10">of limiting activity coefficients in water</span>, Anal. Chem., 65, 1443–1450, <a href="https://doi.org/10.1021/AC00058A023" >doi:10.1021/AC00058A023</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2493',
    'Li et&nbsp;al. (1993)',
    'Li, J., Dallas, A. J., Eikens, D. I., Carr, P. W., Bergmann, D. L., Hait, M. J., &#x0026; Eckert, C. A.: <span class="cmti-10">Measurement of large infinite dilution activity</span> <span class="cmti-10">coefficients of nonelectrolytes in water by inert gas stripping and gas</span> <span class="cmti-10">chromatography</span>, Anal. Chem., 65, 3212–3218, <a href="https://doi.org/10.1021/AC00070A008" >doi:10.1021/AC00070A008</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2464',
    'Li et&nbsp;al. (1998)',
    'Li, J., Perdue, E. M., Pavlostathis, S. G., &#x0026; Araujo, R.: <span class="cmti-10">Physicochemical properties of selected</span> <span class="cmti-10">monoterpenes</span>, Environ. Int., 24, 353–358, <a href="https://doi.org/10.1016/S0160-4120(98)00013-0" >doi:10.1016/S0160-4120(98)00013-0</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2970',
    'Li et&nbsp;al. (2008)',
    'Li, J.-Q., Shen, C.-Y., Xu, G.-H., Wang, H.-M., Jiang, H.-H., Han, H.-Y., Chu, Y.-N., &#x0026; Zheng, P.-C.: <span class="cmti-10">Dynamic measurements of Henry&#x2019;s law constant</span> <span class="cmti-10">of aromatic compounds using proton transfer reaction mass spectrometry</span>, Acta Phys. Chim. Sin., 24, 705–708 (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2540',
    'Li et&nbsp;al. (2003)',
    'Li, N., Wania, F., Lei, Y. D., &#x0026; Daly, G. L.: <span class="cmti-10">A comprehensive and critical</span> <span class="cmti-10">compilation, evaluation, and selection of physical-chemical property data</span> <span class="cmti-10">for selected polychlorinated biphenyls</span>, J. Phys. Chem. Ref. Data, 32, 1545–1590, <a href="https://doi.org/10.1063/1.1562632" >doi:10.1063/1.1562632</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3570',
    'Li et&nbsp;al. (2019)',
    'Li, P., Mühle, J., Montzka, S. A., Oram, D. E., Miller, B. R., Weiss, R. F., Fraser, P. J., &#x0026; Tanhua, T.: <span class="cmti-10">Atmospheric histories, growth rates</span> <span class="cmti-10">and solubilities in seawater and other natural waters of the potential</span> <span class="cmti-10">transient tracers HCFC-22, HCFC-141b, HCFC-142b, HFC-134a,</span> <span class="cmti-10">HFC-125, HFC-23, PFC-14 and PFC-116</span>, Ocean Sci., 15, 33–60, <a href="https://doi.org/10.5194/OS-15-33-2019" >doi:10.5194/OS-15-33-2019</a> (2019). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2913',
    'Li et&nbsp;al. (2004)',
    'Li, S., Chen, Z., &#x0026; Shi, F.: <span class="cmti-10">Determination</span> <span class="cmti-10">of Henry&#x2019;s Law constant for methyl hydroperoxide by long path FTIR</span>, Prog. Nat. Sci., 14, 765–769, <a href="https://doi.org/10.1080/10020070412331344291" >doi:10.1080/10020070412331344291</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3667',
    'Lichtenbelt and Schram (1985)',
    'Lichtenbelt, J. H. &#x0026; Schram, B. J.: <span class="cmti-10">Vapor-liquid equilibrium of water-acetone-air at ambient temperatures</span> <span class="cmti-10">and pressures. An analysis of different VLE-fitting methods</span>, Ind. Eng. Chem. Process Des. Dev., 24, 391–397, <a href="https://doi.org/10.1021/I200029A029" >doi:10.1021/I200029A029</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '477',
    'Lide and Frederikse (1995)',
    'Lide, D. R. &#x0026; Frederikse, H. P. R.: CRC Handbook of Chemistry and Physics, 76th Edition, CRC Press, Inc., Boca Raton, FL, ISBN 0849304768 (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1230',
    'Lin and Pehkonen (1998)',
    'Lin, C.-J. &#x0026; Pehkonen, S. O.: <span class="cmti-10">Oxidation of</span> <span class="cmti-10">elemental mercury by aqueous chlorine (</span>HOCl<span class="cmti-10">/</span>OCl<sup><span class="cmsy-7">−</span></sup><span class="cmti-10">): Implications for</span> <span class="cmti-10">tropospheric mercury chemistry</span>, J. Geophys. Res., 103, 28 093–28 102, <a href="https://doi.org/10.1029/98JD02304" >doi:10.1029/98JD02304</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3694',
    'Lin and Chou (2006)',
    'Lin, J.-H. &#x0026; Chou, M.-S.: <span class="cmti-10">Temperature effects on</span> <span class="cmti-10">Henry&#x2019;s law constants for four VOCs in air-activated sludge systems</span>, Atmos. Environ., 40, 2469–2477, <a href="https://doi.org/10.1016/J.ATMOSENV.2005.12.037" >doi:10.1016/J.ATMOSENV.2005.12.037</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '601',
    'Lincoff and Gossett (1984)',
    'Lincoff, A. H. &#x0026; Gossett, J. M.: <span class="cmti-10">The</span> <span class="cmti-10">determination of Henry&#x2019;s law constant for volatile organics by equilibrium</span> <span class="cmti-10">partitioning in closed systems</span>, in: Gas transfer at water surfaces, edited by Brutsaert, W. &#x0026; Jirka, G. H., pp. 17–25, D. Reidel Publishing Company, Dordrecht-Holland, <a href="https://doi.org/10.1007/978-94-017-1660-4_2" >doi:10.1007/978-94-017-1660-4_2</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '258',
    'Lind and Kok (1986)',
    'Lind, J. A. &#x0026; Kok, G. L.: <span class="cmti-10">Henry&#x2019;s law determinations</span> <span class="cmti-10">for aqueous solutions of hydrogen</span> <span class="cmti-10">peroxide, methylhydroperoxide, and peroxyacetic acid</span>, J. Geophys. Res., 91, 7889–7895, <a href="https://doi.org/10.1029/JD091ID07P07889" >doi:10.1029/JD091ID07P07889</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '311',
    'Lind and Kok (1994)',
    'Lind, J. A. &#x0026; Kok, G. L.: <span class="cmti-10">Correction to “Henry&#x2019;s</span> <span class="cmti-10">law determinations for aqueous solutions of hydrogen peroxide,</span> <span class="cmti-10">methylhydroperoxide, and peroxyacetic acid” by John A.</span><span class="cmti-10"> Lind and</span> <span class="cmti-10">Gregory L.</span><span class="cmti-10"> Kok</span>, J. Geophys. Res., 99, 21 119, <a href="https://doi.org/10.1029/94JD01155" >doi:10.1029/94JD01155</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '796',
    'Lindinger et&nbsp;al. (1998)',
    'Lindinger, W., Hansel, A., &#x0026; Jordan, A.: <span class="cmti-10">On-line monitoring of volatile organic compounds</span> <span class="cmti-10">at pptv levels by means of proton-transfer-reaction mass spectrometry</span> <span class="cmti-10">(PTR-MS) medical applications, food control and</span> <span class="cmti-10">environmental research</span>, Int. J. Mass Spectrom. Ion Proc., 173, 191–241, <a href="https://doi.org/10.1016/S0168-1176(97)00281-4" >doi:10.1016/S0168-1176(97)00281-4</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2290',
    'Lindqvist and Rodhe (1985)',
    'Lindqvist, O. &#x0026; Rodhe, H.: <span class="cmti-10">Atmospheric mercury – a review</span>, Tellus, 37B, 136–159, <a href="https://doi.org/10.1111/J.1600-0889.1985.TB00062.X" >doi:10.1111/J.1600-0889.1985.TB00062.X</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3517',
    'Linnemann et&nbsp;al. (2020)',
    'Linnemann, M., Nikolaychuk, P. A., noz Muñoz, Y. M. M., Baumhögger, E., &#x0026; Vrabec, J.: <span class="cmti-10">Henry&#x2019;s law constant of noble</span> <span class="cmti-10">gases in water, methanol, ethanol, and isopropanol by experiment</span> <span class="cmti-10">and molecular simulation</span>, J. Chem. Eng. Data, 65, 1180–1188, <a href="https://doi.org/10.1021/ACS.JCED.9B00565" >doi:10.1021/ACS.JCED.9B00565</a> (2020). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '563',
    'Liss and Slater (1974)',
    'Liss, P. S. &#x0026; Slater, P. G.: <span class="cmti-10">Flux of gases across the</span> <span class="cmti-10">air-sea interface</span>, Nature, 247, 181–184, <a href="https://doi.org/10.1038/247181A0" >doi:10.1038/247181A0</a> (1974). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3695',
    'Liu et&nbsp;al. (2021)',
    'Liu, G.-H., Wen, M.-M., Deng, L.-T., Cui, H.-N., Jia, Y.-Y., Cheng, S.-H., Cao, J., &#x0026; Li, C.: <span class="cmti-10">The determination of Henry&#x2019;s law constant</span> <span class="cmti-10">of methane in test water by A.R.M/headspace</span>, Acta Geosci. Sinica, 4, 565–571, <a href="https://doi.org/10.3975/CAGSB.2020.111202" >doi:10.3975/CAGSB.2020.111202</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3111',
    'Liu et&nbsp;al. (2015)',
    'Liu, X., Guo, Z., Roache, N. F., Mocka, C. A., Allen, M. R., &#x0026; Mason, M. A.: <span class="cmti-10">Henry&#x2019;s law constant and overall mass transfer coefficient for</span> <span class="cmti-10">formaldehyde emission from small water pools under simulated indoor</span> <span class="cmti-10">environmental conditions</span>, Environ. Sci. Technol., 49, 1603–1610, <a href="https://doi.org/10.1021/ES504540C" >doi:10.1021/ES504540C</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2954',
    'Lodge and Danso (2007)',
    'Lodge, K. B. &#x0026; Danso, D.: <span class="cmti-10">The measurement</span> <span class="cmti-10">of fugacity and the Henry&#x2019;s law constant for volatile organic</span> <span class="cmti-10">compounds containing chromophores</span>, Fluid Phase Equilib., 253, 74–79, <a href="https://doi.org/10.1016/J.FLUID.2007.01.010" >doi:10.1016/J.FLUID.2007.01.010</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3683',
    'Long et&nbsp;al. (2017)',
    'Long, J., Youli, Q., &#x0026; Yu, L.: <span class="cmti-10">Effect analysis of quantum chemical descriptors and substituent</span> <span class="cmti-10">characteristics on Henry&#x2019;s law constants of polybrominated diphenyl</span> <span class="cmti-10">ethers at different temperatures</span>, Ecotoxicol. Environ. Saf., 145, 176–183, <a href="https://doi.org/10.1016/J.ECOENV.2017.07.024" >doi:10.1016/J.ECOENV.2017.07.024</a> (2017). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3786',
    'Longo et&nbsp;al. (1970)',
    'Longo, L. D., Delivoria-Papadopoulos, M., Power, G. G., Hill, E. P., &#x0026; Forster, R. E.: <span class="cmti-10">Diffusion equilibration of inert gases between</span> <span class="cmti-10">maternal and fetal placental capillaires</span>, Am. J. Physiol., 219, 561–569, <a href="https://doi.org/10.1152/AJPLEGACY.1970.219.3.561" >doi:10.1152/AJPLEGACY.1970.219.3.561</a> (1970). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '732',
    'Loomis (1928)',
    'Loomis, A. G.: <span class="cmti-10">Solubilities of gases in water</span>, in: International Critical Tables of Numerical Data, Physics, Chemistry and Technology, Vol. III, edited by Washburn, E. W., West, C. J., Dorsey, N. E., Bichowsky, F. R., &#x0026; Klemenc, A., pp. 255–261, McGraw-Hill, Inc. (1928). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '726',
    'Lovelock et&nbsp;al. (1972)',
    'Lovelock, J. E., Maggs, R. J., &#x0026; Rasmussen, R. A.: <span class="cmti-10">Atmospheric dimethyl sulphide and</span> <span class="cmti-10">the natural sulphur cycle</span>, Nature, 237, 452–453, <a href="https://doi.org/10.1038/237452A0" >doi:10.1038/237452A0</a> (1972). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '533',
    'Luke et&nbsp;al. (1989)',
    'Luke, W. T., Dickerson, R. R., &#x0026; Nunnermacker, L. J.: <span class="cmti-10">Direct</span> <span class="cmti-10">measurements of the photolysis rate coefficients and Henry&#x2019;s law</span> <span class="cmti-10">constants of several alkyl nitrates</span>, J. Geophys. Res., 94, 14 905–14 921, <a href="https://doi.org/10.1029/JD094ID12P14905" >doi:10.1029/JD094ID12P14905</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3595',
    'Luther (1905)',
    'Luther, R.: <span class="cmti-10">Zur Kenntnis des Ozons</span>, Z. Elektrochem., 11, 832–835, <a href="https://doi.org/10.1002/BBPC.19050114704" >doi:10.1002/BBPC.19050114704</a> (1905). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3690',
    'Lutsyk et&nbsp;al. (2005)',
    'Lutsyk, A., Portnanskij, V., Sujkov, S., &#x0026; Tchuprina, V.: <span class="cmti-10">A new set of gas/water</span> <span class="cmti-10">partition coefficients for the chloromethanes</span>, Monatsh. Chem. – Chem. Mon., 136, 1183–1189, <a href="https://doi.org/10.1007/S00706-005-0319-6" >doi:10.1007/S00706-005-0319-6</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3533',
    'Ma et&nbsp;al. (2010a)',
    'Ma, J., Dasgupta, P. K., Blackledge, W., &#x0026; Boss, G. R.: <span class="cmti-10">Temperature dependence of</span> <span class="cmti-10">Henry&#x2019;s law constant for hydrogen cyanide. Generation of trace standard</span> <span class="cmti-10">gaseous hydrogen cyanide</span>, Environ. Sci. Technol., 44, 3028–3034, <a href="https://doi.org/10.1021/ES1001192" >doi:10.1021/ES1001192</a> (2010a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2989',
    'Ma et&nbsp;al. (2010b)',
    'Ma, Y.-G., Lei, Y. D., Xiao, H., Wania, F., &#x0026; Wang, W.-H.: <span class="cmti-10">Critical review and recommended</span> <span class="cmti-10">values for the physical-chemical property data of 15 polycyclic</span> <span class="cmti-10">aromatic hydrocarbons at 25</span><span class="cmsy-7">∘</span>C, J. Chem. Eng. Data, 55, 819–825, <a href="https://doi.org/10.1021/JE900477X" >doi:10.1021/JE900477X</a> (2010b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '834',
    'Maahs (1982)',
    'Maahs, H. G.: <span class="cmti-10">Sulfur-dioxide/water equilibria between 0</span><span class="cmsy-7">∘</span> <span class="cmti-10">and 50</span><span class="cmsy-7">∘</span>C<span class="cmti-10">. An examination of data at low concentrations</span>, in: Heterogeneous Atmospheric Chemistry, Geophysical Monograph 26, edited by Schryer, D. R., pp. 187–195, Am. Geophys. Union, Washington, D.C., <a href="https://doi.org/10.1029/GM026P0187" >doi:10.1029/GM026P0187</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '923',
    'Maaßen (1995)',
    'Maaßen, S.: Experimentelle Bestimmung und Korrelierung von Verteilungskoeffizienten in verdünnten Lösungen, Ph.D. thesis, Technische Universität Berlin, Germany, ISBN 3826511042 (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2972',
    'Mabury and Crosby (1996)',
    'Mabury, S. A. &#x0026; Crosby, D. G.: <span class="cmti-10">Pesticide</span> <span class="cmti-10">reactivity toward hydroxyl and its relationship to field persistence</span>, J. Agric. Food Chem., 44, 1920–1924, <a href="https://doi.org/10.1021/JF950423Y" >doi:10.1021/JF950423Y</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3087',
    'MacBean (2012a)',
    'MacBean, C.: <span class="cmti-10">The Pesticide Manual, 16th Edition,</span> <span class="cmti-10">Supplementary Entries – Extended</span>, Tech. rep., British Crop Production Council, ISBN 190139686X (2012a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'macbean2012',
    'MacBean (2012b)',
    'MacBean, C.: The Pesticide Manual, British Crop Production Council, ISBN 9781901396867 (2012b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'macintosh58',
    'Macintosh et&nbsp;al. (1958)',
    'Macintosh, R. R., Mushin, W. W., &#x0026; Epstein, H. G.: Physics for the Anaesthetist: Including a Section on Explosions, Charles C. Thomas Publisher, Ltd. (1958). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2471',
    'Mackay and Leinonen (1975)',
    'Mackay, D. &#x0026; Leinonen, P. J.: <span class="cmti-10">Rate of evaporation of low-solubility contaminants</span> <span class="cmti-10">from water bodies to atmosphere</span>, Environ. Sci. Technol., 9, 1178–1180, <a href="https://doi.org/10.1021/ES60111A012" >doi:10.1021/ES60111A012</a> (1975). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '479',
    'Mackay and Shiu (1981)',
    'Mackay, D. &#x0026; Shiu, W. Y.: <span class="cmti-10">A critical review of</span> <span class="cmti-10">Henry&#x2019;s law constants for chemicals of environmental interest</span>, J. Phys. Chem. Ref. Data, 10, 1175–1199, <a href="https://doi.org/10.1063/1.555654" >doi:10.1063/1.555654</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2233',
    'Mackay and Yeun (1983)',
    'Mackay, D. &#x0026; Yeun, A. T. K.: <span class="cmti-10">Mass transfer</span> <span class="cmti-10">coefficient correlations for volatilization of organic solutes from water</span>, Environ. Sci. Technol., 17, 211–217, <a href="https://doi.org/10.1021/ES00110A006" >doi:10.1021/ES00110A006</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '490',
    'Mackay et&nbsp;al. (1979)',
    'Mackay, D., Shiu, W. Y., &#x0026; Sutherland, R. P.: <span class="cmti-10">Determination of air–water Henry&#x2019;s law</span> <span class="cmti-10">constants for hydrophobic pollutants</span>, Environ. Sci. Technol., 13, 333–337, <a href="https://doi.org/10.1021/ES60151A012" >doi:10.1021/ES60151A012</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '633',
    'Mackay et&nbsp;al. (1992a)',
    'Mackay, D., Shiu, W. Y., &#x0026; Ma, K. C.: Illustrated Handbook of Physical-Chemical Properties and Environmental Fate for Organic Chemicals, vol. I of <span class="cmti-10">Monoaromatic</span> <span class="cmti-10">Hydrocarbons, Chlorobenzenes, and PCBs</span>, Lewis Publishers, Boca Raton, ISBN 0873715136 (1992a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '634',
    'Mackay et&nbsp;al. (1992b)',
    'Mackay, D., Shiu, W. Y., &#x0026; Ma, K. C.: Illustrated Handbook of Physical-Chemical Properties and Environmental Fate for Organic Chemicals, vol. II of <span class="cmti-10">Polynuclear</span> <span class="cmti-10">Aromatic Hydrocarbons, Polychlorinated Dioxins, and Dibenzofurans</span>, Lewis Publishers, Boca Raton, ISBN 0873715837 (1992b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '635',
    'Mackay et&nbsp;al. (1993)',
    'Mackay, D., Shiu, W. Y., &#x0026; Ma, K. C.: Illustrated Handbook of Physical-Chemical Properties and Environmental Fate for Organic Chemicals, vol. III of <span class="cmti-10">Volatile Organic</span> <span class="cmti-10">Chemicals</span>, Lewis Publishers, Boca Raton, ISBN 0873719735 (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '636',
    'Mackay et&nbsp;al. (1995)',
    'Mackay, D., Shiu, W. Y., &#x0026; Ma, K. C.: Illustrated Handbook of Physical-Chemical Properties and Environmental Fate for Organic Chemicals, vol. IV of <span class="cmti-10">Oxygen, Nitrogen,</span> <span class="cmti-10">and Sulfur Containing Compounds</span>, Lewis Publishers, Boca Raton, ISBN 1566700353 (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2611',
    'Mackay et&nbsp;al. (2006a)',
    'Mackay, D., Shiu, W. Y., Ma, K. C., &#x0026; Lee, S. C.: Handbook of Physical-Chemical Properties and Environmental Fate for Organic Chemicals, vol. I of <span class="cmti-10">Introduction and Hydrocarbons</span>, CRC/Taylor &#x0026; Francis Group, <a href="https://doi.org/10.1201/9781420044393" >doi:10.1201/9781420044393</a> (2006a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2612',
    'Mackay et&nbsp;al. (2006b)',
    'Mackay, D., Shiu, W. Y., Ma, K. C., &#x0026; Lee, S. C.: Handbook of Physical-Chemical Properties and Environmental Fate for Organic Chemicals, vol. II of <span class="cmti-10">Halogenated Hydrocarbons</span>, CRC/Taylor &#x0026; Francis Group, <a href="https://doi.org/10.1201/9781420044393" >doi:10.1201/9781420044393</a> (2006b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2613',
    'Mackay et&nbsp;al. (2006c)',
    'Mackay, D., Shiu, W. Y., Ma, K. C., &#x0026; Lee, S. C.: Handbook of Physical-Chemical Properties and Environmental Fate for Organic Chemicals, vol. III of <span class="cmti-10">Oxygen Containing</span> <span class="cmti-10">Compounds</span>, CRC/Taylor &#x0026; Francis Group, <a href="https://doi.org/10.1201/9781420044393" >doi:10.1201/9781420044393</a> (2006c). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2614',
    'Mackay et&nbsp;al. (2006d)',
    'Mackay, D., Shiu, W. Y., Ma, K. C., &#x0026; Lee, S. C.: Handbook of Physical-Chemical Properties and Environmental Fate for Organic Chemicals, vol. IV of <span class="cmti-10">Nitrogen and</span> <span class="cmti-10">Sulfur Containing Compounds and Pesticides</span>, CRC/Taylor &#x0026; Francis Group, <a href="https://doi.org/10.1201/9781420044393" >doi:10.1201/9781420044393</a> (2006d). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '844',
    'Magi et&nbsp;al. (1997)',
    'Magi, L., Schweitzer, F., Pallares, C., Cherif, S., Mirabel, P., &#x0026; George, C.: <span class="cmti-10">Investigation of the uptake rate of ozone and methyl</span> <span class="cmti-10">hydroperoxide by water surfaces</span>, J. Phys. Chem. A, 101, 4943–4949, <a href="https://doi.org/10.1021/JP970646M" >doi:10.1021/JP970646M</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3596',
    'Mailfert (1894)',
    'Mailfert, M.: <span class="cmti-10">Sur la solubilit</span><span class="cmti-10">é de l&#x2019;ozone</span>, C. R. Hebd. Séances Acad. Sci., 119, 951–953 (1894). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3750',
    'Maillard and Rosenthal (1952)',
    'Maillard, A. &#x0026; Rosenthal, W.: <span class="cmti-10">Solubilit</span><span class="cmti-10">é de</span> <span class="cmti-10">l&#x2019;ac</span><span class="cmti-10">étyl</span><span class="cmti-10">ène dans divers liquides organiques</span>, C. R. Hebd. Séances Acad. Sci., 234, 2546–2548 (1952). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3549',
    'Maniere et&nbsp;al. (2011)',
    'Maniere, I., Bouneb, F., Fastier, A., Courty, B., Dumenil, J., Poupard, M., &#x0026; Mercier, T.: <span class="cmti-10">AGRITOX-Database on pesticide active substances</span>, Toxicol. Lett., 205S, S231–S232, <a href="https://doi.org/10.1016/J.TOXLET.2011.05.792" >doi:10.1016/J.TOXLET.2011.05.792</a>, URL https://www.data.gouv.fr/fr/datasets/base-de-donnees-agritox (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1992',
    'Manogue and Pigford (1960)',
    'Manogue, W. H. &#x0026; Pigford, R. L.: <span class="cmti-10">The kinetics</span> <span class="cmti-10">of the absorption of phosgene into water and aqueous solutions</span>, AIChE J., 6, 494–500, <a href="https://doi.org/10.1002/AIC.690060329" >doi:10.1002/AIC.690060329</a> (1960). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3563',
    'Mansfield (2020)',
    'Mansfield, M. L.: <span class="cmti-10">Mass transport of gases across the</span> <span class="cmti-10">air–water interface: implications for aldehyde emissions in the Uinta</span> <span class="cmti-10">Basin, Utah, USA</span>, Atmos., 11, 1057, <a href="https://doi.org/10.3390/ATMOS11101057" >doi:10.3390/ATMOS11101057</a> (2020). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2964',
    'Marin et&nbsp;al. (1999)',
    'Marin, M., Baek, I., &#x0026; Taylor, A. J.: <span class="cmti-10">Volatile release from aqueous solutions</span> <span class="cmti-10">under dynamic headspace dilution conditions</span>, J. Agric. Food Chem., 47, 4750–4755, <a href="https://doi.org/10.1021/JF990470G" >doi:10.1021/JF990470G</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2607',
    'Markham and Kobe (1941)',
    'Markham, A. E. &#x0026; Kobe, K. A.: <span class="cmti-10">The solubility</span> <span class="cmti-10">of gases in liquids</span>, Chem. Rev., 28, 519–588, <a href="https://doi.org/10.1021/CR60091A003" >doi:10.1021/CR60091A003</a> (1941). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '167',
    'Marsh and McElroy (1985)',
    'Marsh, A. R. W. &#x0026; McElroy, W. J.: <span class="cmti-10">The dissociation constant and Henry&#x2019;s law</span> <span class="cmti-10">constant of</span> HCl <span class="cmti-10">in aqueous solution</span>, Atmos. Environ., 19, 1075–1080, <a href="https://doi.org/10.1016/0004-6981(85)90192-1" >doi:10.1016/0004-6981(85)90192-1</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '769',
    'Marti et&nbsp;al. (1997)',
    'Marti, J. J., Jefferson, A., Cai, X. P., Richert, C., McMurry, P. H., &#x0026; Eisele, F.: H<sub><span class="cmr-7">2</span></sub>SO<sub><span class="cmr-7">4</span></sub> <span class="cmti-10">vapor pressure of sulfuric acid and ammonium sulfate</span> <span class="cmti-10">solutions</span>, J. Geophys. Res., 102, 3725–3735, <a href="https://doi.org/10.1029/96JD03064" >doi:10.1029/96JD03064</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '74',
    'Martin (1984)',
    'Martin, L. R.: <span class="cmti-10">Kinetic studies of sulfite oxidation in aqueous</span> <span class="cmti-10">solution</span>, in: SO<sub><span class="cmr-7">2</span></sub>, NO and NO<sub><span class="cmr-7">2</span></sub> Oxidation Mechanisms: Atmospheric Considerations, edited by Calvert, J. G., pp. 63–100, Butterworth Publishers, Boston, MA, ISBN 0250405687 (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '43',
    'Martin and Damschen (1981)',
    'Martin, L. R. &#x0026; Damschen, D. E.: <span class="cmti-10">Aqueous</span> <span class="cmti-10">oxidation of sulfur dioxide by hydrogen peroxide at low pH</span>, Atmos. Environ., 15, 1615–1621, <a href="https://doi.org/10.1016/0004-6981(81)90146-3" >doi:10.1016/0004-6981(81)90146-3</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3678',
    'Martins et&nbsp;al. (2017)',
    'Martins, M. A. R., Silva, L. P., Ferreira, O., Schröder, B., Coutinho, J. A. P., &#x0026; Pinho, S. P.: <span class="cmti-10">Terpenes solubility in water</span> <span class="cmti-10">and their environmental distribution</span>, J. Mol. Liq., 241, 996–1002, <a href="https://doi.org/10.1016/J.MOLLIQ.2017.06.099" >doi:10.1016/J.MOLLIQ.2017.06.099</a> (2017). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3951',
    'Mavu (2011)',
    'Mavu, D. M.: Pharmaceutical formulation and in-vitro testing of Dioxy MP 14 (stabilised chlorine dioxide) against mycobacteria tuberculosis, Master&#x2019;s thesis, University of the Western Cape, South Africa, URL https://etd.uwc.ac.za/xmlui/handle/11394/5372 (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3669',
    'Mazza (1980)',
    'Mazza, G.: <span class="cmti-10">Relative volatilities of some onion flavour components</span>, Int. J. Food Sci. Technol., 15, 35–41, <a href="https://doi.org/10.1111/J.1365-2621.1980.TB00916.X" >doi:10.1111/J.1365-2621.1980.TB00916.X</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3095',
    'Mazzoni et&nbsp;al. (1997)',
    'Mazzoni, S. M., Roy, S., &#x0026; Grigoras, S.: <span class="cmti-10">Eco-relevant properties of selected organosilicon</span> <span class="cmti-10">materials</span>, in: The Handbook of Environmental Chemistry, Vol. 3. Part H. Organosilicon Materials, edited by Chandra, G., pp. 53–81, Springer Verlag, Berlin, <a href="https://doi.org/10.1007/978-3-540-68331-5" >doi:10.1007/978-3-540-68331-5</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3923',
    'McAuliffe (1963)',
    'McAuliffe, C.: <span class="cmti-10">Solubility in water of</span> C<sub><span class="cmr-7">1</span></sub><span class="cmti-10">-</span>C<sub><span class="cmr-7">9</span></sub> <span class="cmti-10">hydrocarbons</span>, Nature, 200, 1092–1093, <a href="https://doi.org/10.1038/2001092A0" >doi:10.1038/2001092A0</a> (1963). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3924',
    'McAuliffe (1966)',
    'McAuliffe, C.: <span class="cmti-10">Solubility in water of paraffin,</span> <span class="cmti-10">cycloparaffin, olefin, acetylene, cycloolefin, and aromatic hydrocarbons</span>, J. Phys. Chem., 70, 1267–1275, <a href="https://doi.org/10.1021/J100876A049" >doi:10.1021/J100876A049</a> (1966). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'mcauliffe71',
    'McAuliffe (1971)',
    'McAuliffe, C.: <span class="cmti-10">Gas chromatographic determination of</span> <span class="cmti-10">solutes by multiple phase equilibrium</span>, Chem. Technol., 1, 46–51 (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'mccarty80',
    'McCarty (1980)',
    'McCarty, P. L.: <span class="cmti-10">Organics in water – an engineering</span> <span class="cmti-10">challenge</span>, J. Environ. Eng. Div., 106, 1–17 (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '543',
    'McConnell et&nbsp;al. (1975)',
    'McConnell, G., Ferguson, D. M., &#x0026; Pearson, C. R.: <span class="cmti-10">Chlorinated hydrocarbons and the</span> <span class="cmti-10">environment</span>, Endeavour, 34, 13–18, <a href="https://doi.org/10.1016/0160-9327(75)90062-9" >doi:10.1016/0160-9327(75)90062-9</a> (1975). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3232',
    'McCoy et&nbsp;al. (1990)',
    'McCoy, W. F., Blatchley, III, E. R., &#x0026; Johnson, R. W.: <span class="cmti-10">Hypohalous acid and</span> <span class="cmti-10">haloamine flashoff in industrial evaporative cooling systems</span>, Tech. rep., Cooling Tower Institute, paper no.: TP90-09 (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3723',
    'McFall et&nbsp;al. (2020)',
    'McFall, A. S., Johnson, A. W., &#x0026; Anastasio, C.: <span class="cmti-10">Air–water partitioning</span> <span class="cmti-10">of biomass-burning phenols and the effects of temperature and salinity</span>, Environ. Sci. Technol., 54, 3823–3830, <a href="https://doi.org/10.1021/ACS.EST.9B06443" >doi:10.1021/ACS.EST.9B06443</a> (2020). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3672',
    'McIntosh and Heffron (2000)',
    'McIntosh, J. M. &#x0026; Heffron, J. J. A.: <span class="cmti-10">Modelling</span> <span class="cmti-10">alterations in the partition coefficient in in vitro biological systems</span> <span class="cmti-10">using headspace gas chromatography</span>, J. Chromatogr. B, 738, 207–216, <a href="https://doi.org/10.1016/S0378-4347(99)00506-X" >doi:10.1016/S0378-4347(99)00506-X</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3641',
    'McKeown and Stowell (1927)',
    'McKeown, A. &#x0026; Stowell, F. P.: <span class="cmti-10">XVI.—The</span> <span class="cmti-10">vapour pressures of mixtures of (a) methyl acetate and water ; (b)</span> <span class="cmti-10">methyl acetate, sucrose, and water</span>, J. Chem. Soc., pp. 97–103, <a href="https://doi.org/10.1039/JR9270000097" >doi:10.1039/JR9270000097</a> (1927). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2975',
    'McLachlan et&nbsp;al. (1990)',
    'McLachlan, M., Mackay, D., &#x0026; Jones, P. H.: <span class="cmti-10">A conceptual model of</span> <span class="cmti-10">organic chemical volatilization at waterfalls</span>, Environ. Sci. Technol., 24, 252–257, <a href="https://doi.org/10.1021/ES00072A015" >doi:10.1021/ES00072A015</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1014',
    'McLinden (1989)',
    'McLinden, M. O.: <span class="cmti-10">Physical properties of alternatives to</span> <span class="cmti-10">the fully halogenated chlorofluorocarbons</span>, in: WMO Report 20, Scientific Assessment of Stratospheric Ozone: 1989, Volume II, pp. 11–38, World Meteorol. Organ., Geneva, ISBN 9280712551 (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3098',
    'McNeill et&nbsp;al. (2012)',
    'McNeill, V. F., Woo, J. L., Kim, D. D., Schwier, A. N., Wannell, N. J., Sumner, A. J., &#x0026; Barakat, J. M.: <span class="cmti-10">Aqueous-phase secondary organic</span> <span class="cmti-10">aerosol and organosulfate formation in atmospheric aerosols: a modeling</span> <span class="cmti-10">study</span>, Environ. Sci. Technol., 46, 8075–8081, <a href="https://doi.org/10.1021/ES3002986" >doi:10.1021/ES3002986</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3009',
    'McPhedran et&nbsp;al. (2013)',
    'McPhedran, K. N., Seth, R., &#x0026; Drouillard, K. G.: <span class="cmti-10">Evaluation of the gas</span> <span class="cmti-10">stripping technique for calculation of Henry&#x2019;s law constants using the</span> <span class="cmti-10">initial slope method for 1,2,4,5-tetrachlorobenzene, pentachlorobenzene,</span> <span class="cmti-10">and hexachlorobenzene</span>, Chemosphere, 91, 1648–1652, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2012.12.017" >doi:10.1016/J.CHEMOSPHERE.2012.12.017</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '505',
    'Meadows and Spedding (1974)',
    'Meadows, R. W. &#x0026; Spedding, D. J.: <span class="cmti-10">The</span> <span class="cmti-10">solubility of very low concentrations of carbon monoxide in aqueous</span> <span class="cmti-10">solution</span>, Tellus, 26, 143–149, <a href="https://doi.org/10.3402/TELLUSA.V26I1-2.9745" >doi:10.3402/TELLUSA.V26I1-2.9745</a> (1974). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3436',
    'Mentel et&nbsp;al. (1999)',
    'Mentel, T. F., Sohn, M., &#x0026; Wahner, A.: <span class="cmti-10">Nitrate effect in the heterogeneous hydrolysis of dinitrogen</span> <span class="cmti-10">pentoxide on aqueous aerosols</span>, Phys. Chem. Chem. Phys., 1, 5451–5457, <a href="https://doi.org/10.1039/A905338G" >doi:10.1039/A905338G</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1658',
    'Mentel et&nbsp;al. (2004)',
    'Mentel, T. F., Folkers, M., Tillmann, R., Henk, H., Wahner, A., Otjes, R., Blom, M., &#x0026; ten Brink, H. M.: <span class="cmti-10">Determination of the Henry</span> <span class="cmti-10">coefficients for organic aerosol components</span>, Geophys. Res. Abstr., 6, 1525 (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3657',
    'Merk and Riederer (1997)',
    'Merk, S. &#x0026; Riederer, M.: <span class="cmti-10">Sorption of volatile</span> <span class="cmti-10">C1 to C6 alkanols in plant cuticles</span>, J. Exp. Bot., 48, 1095–1104, <a href="https://doi.org/10.1093/JXB/48.5.1095" >doi:10.1093/JXB/48.5.1095</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2454',
    'Metcalfe et&nbsp;al. (1980)',
    'Metcalfe, C. D., McLeese, D. W., &#x0026; Zitko, V.: <span class="cmti-10">Rate of volatilization of fenitrothion from</span> <span class="cmti-10">fresh water</span>, Chemosphere, 9, 151–155, <a href="https://doi.org/10.1016/0045-6535(80)90086-7" >doi:10.1016/0045-6535(80)90086-7</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1145',
    'Meylan and Howard (1991)',
    'Meylan, W. M. &#x0026; Howard, P. H.: <span class="cmti-10">Bond</span> <span class="cmti-10">contribution method for estimating Henry&#x2019;s law constants</span>, Environ. Toxicol. Chem., 10, 1283–1293, <a href="https://doi.org/10.1002/ETC.5620101007" >doi:10.1002/ETC.5620101007</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3650',
    'Meynier et&nbsp;al. (2003)',
    'Meynier, A., Garillon, A., Lethuaut, L., &#x0026; Genot, C.: <span class="cmti-10">Partition of five aroma</span> <span class="cmti-10">compounds between air and skim milk, anhydrous milk fat or full-fat</span> <span class="cmti-10">cream</span>, Lait, 83, 223–235, <a href="https://doi.org/10.1051/LAIT:2003012" >doi:10.1051/LAIT:2003012</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3552',
    'Mick et&nbsp;al. (2016)',
    'Mick, J. R., Barhaghi, M. S., &#x0026; Potoff, J. J.: <span class="cmti-10">Prediction of radon-222 phase behavior by Monte Carlo</span> <span class="cmti-10">simulation</span>, J. Chem. Eng. Data, 61, 1625–1631, <a href="https://doi.org/10.1021/ACS.JCED.5B01002" >doi:10.1021/ACS.JCED.5B01002</a> (2016). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3261',
    'Miguel et&nbsp;al. (2000)',
    'Miguel, A. A. F., Ferreira, A. G. M., &#x0026; Fonseca, I. M. A.: <span class="cmti-10">Solubilities</span> <span class="cmti-10">of some new refrigerants in water</span>, Fluid Phase Equilib., 173, 97–107, <a href="https://doi.org/10.1016/S0378-3812(00)00390-3" >doi:10.1016/S0378-3812(00)00390-3</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1928',
    'Miller and Stuart (2000)',
    'Miller, M. E. &#x0026; Stuart, J. D.: <span class="cmti-10">Measurement of</span> <span class="cmti-10">aqueous Henry&#x2019;s law constants for oxygenates and aromatics found in</span> <span class="cmti-10">gasolines by the static headspace method</span>, Anal. Chem., 72, 622–625, <a href="https://doi.org/10.1021/AC990757C" >doi:10.1021/AC990757C</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1927',
    'Miller and Stuart (2003)',
    'Miller, M. E. &#x0026; Stuart, J. D.: <span class="cmti-10">Correction: Measurement of aqueous Henry&#x2019;s law constants</span> <span class="cmti-10">for oxygenates and aromatics found in gasolines by the static headspace</span> <span class="cmti-10">method</span>, Anal. Chem., 75, 1037, <a href="https://doi.org/10.1021/AC034002O" >doi:10.1021/AC034002O</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3605',
    'Millero et&nbsp;al. (2002a)',
    'Millero, F. J., Huang, F., &#x0026; Laferiere, A. L.: <span class="cmti-10">Solubility of oxygen in the major sea salts as a</span> <span class="cmti-10">function of concentration and temperature</span>, Mar. Chem., 78, 217–230, <a href="https://doi.org/10.1016/S0304-4203(02)00034-8" >doi:10.1016/S0304-4203(02)00034-8</a> (2002a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3606',
    'Millero et&nbsp;al. (2002b)',
    'Millero, F. J., Huang, F., &#x0026; Laferiere, A. L.: <span class="cmti-10">The solubility of oxygen in the major sea salts</span> <span class="cmti-10">and their mixtures at 25</span><span class="cmsy-7">∘</span>C, Geochim. Cosmochim. Acta, 66, 2349–2359, <a href="https://doi.org/10.1016/S0016-7037(02)00838-4" >doi:10.1016/S0016-7037(02)00838-4</a> (2002b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '671',
    'Mirabel et&nbsp;al. (1996)',
    'Mirabel, P., George, C., Magi, L., &#x0026; Ponche, J. L.: <span class="cmti-10">Chapter</span> <span class="cmti-10">6.3: Gas-liquid interactions</span>, in: Heterogeneous and Liquid-Phase Processes, edited by Warneck, P., pp. 175–181, Springer Verlag, Berlin, <a href="https://doi.org/10.1007/978-3-642-61445-3_6" >doi:10.1007/978-3-642-61445-3_6</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3082',
    'Mirvish et&nbsp;al. (1976)',
    'Mirvish, S. S., Issenberg, P., &#x0026; Sornson, H. C.: <span class="cmti-10">Air–water and ether-water distribution</span> <span class="cmti-10">of N-nitroso compounds: implications for laboratory safety, analytic</span> <span class="cmti-10">methodology, and carcinogenicity for the rat esophagus, nose, and liver</span>, J. Natl. Cancer Inst., 56, 1125–1129, <a href="https://doi.org/10.1093/JNCI/56.6.1125" >doi:10.1093/JNCI/56.6.1125</a> (1976). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3512',
    'Modarresi et&nbsp;al. (2005)',
    'Modarresi, H., Modarress, H., &#x0026; Dearden, J. C.: <span class="cmti-10">Henry&#x2019;s law constant of hydrocarbons</span> <span class="cmti-10">in air–water system: The cavity ovality effect on the non-electrostatic</span> <span class="cmti-10">contribution term of solvation free energy</span>, SAR QSAR Environ. Res., 16, 461–482, <a href="https://doi.org/10.1080/10659360500319869" >doi:10.1080/10659360500319869</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2530',
    'Modarresi et&nbsp;al. (2007)',
    'Modarresi, H., Modarress, H., &#x0026; Dearden, J. C.: <span class="cmti-10">QSPR model of Henry&#x2019;s law</span> <span class="cmti-10">constant for a diverse set of organic chemicals based on genetic</span> <span class="cmti-10">algorithm-radial basis function network approach</span>, Chemosphere, 66, 2067–2076, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2006.09.049" >doi:10.1016/J.CHEMOSPHERE.2006.09.049</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3117',
    'Mohebbi et&nbsp;al. (2012)',
    'Mohebbi, V., Naderifar, A., Behbahani, R. M., &#x0026; Moshfeghian, M.: <span class="cmti-10">Determination of Henry&#x2019;s law constant of light hydrocarbon</span> <span class="cmti-10">gases at low temperatures</span>, J. Chem. Thermodyn., 51, 8–11, <a href="https://doi.org/10.1016/J.JCT.2012.02.014" >doi:10.1016/J.JCT.2012.02.014</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '507',
    'Möller and Mauersberger (1992)',
    'Möller, D. &#x0026; Mauersberger, G.: <span class="cmti-10">Aqueous</span> <span class="cmti-10">phase chemical reaction system used in cloud chemistry modelling</span>, in: EUROTRAC Special Publication: Clouds: Models and Mechanisms, edited by Flossmann, A., Cvitaš, T., Möller, D., &#x0026; Mauersberger, G., pp. 77–93 (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1026',
    'Moore (2000)',
    'Moore, R. M.: <span class="cmti-10">The</span> <span class="cmti-10">solubility of a suite of low molecular weight organochlorine compounds</span> <span class="cmti-10">in seawater and implications for estimating the marine source of methyl</span> <span class="cmti-10">chloride to the atmosphere</span>, Chemosphere Global Change Sci., 2, 95–99, <a href="https://doi.org/10.1016/S1465-9972(99)00045-8" >doi:10.1016/S1465-9972(99)00045-8</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '583',
    'Moore et&nbsp;al. (1995)',
    'Moore, R. M., Geen, C. E., &#x0026; Tait, V. K.: <span class="cmti-10">Determination of Henry&#x2019;s law constants for a suite of</span> <span class="cmti-10">naturally occuring halogenated methanes in seawater</span>, Chemosphere, 30, 1183–1191, <a href="https://doi.org/10.1016/0045-6535(95)00009-W" >doi:10.1016/0045-6535(95)00009-W</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3783',
    'Morrison and Billett (1952)',
    'Morrison, T. J. &#x0026; Billett, F.: <span class="cmti-10">730. The salting-out of non-electrolytes.</span> <span class="cmti-10">Part II. The effect of variation in non-electrolyte</span>, J. Chem. Soc., pp. 3819–3822, <a href="https://doi.org/10.1039/JR9520003819" >doi:10.1039/JR9520003819</a> (1952). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '764',
    'Morrison and Johnstone (1954)',
    'Morrison, T. J. &#x0026; Johnstone, N. B.: <span class="cmti-10">Solubilities of the inert gases in water</span>, J. Chem. Soc., pp. 3441–3446, <a href="https://doi.org/10.1039/JR9540003441" >doi:10.1039/JR9540003441</a> (1954). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3950',
    'Morrow et&nbsp;al. (2006)',
    'Morrow, M., McMahon, T., Leighton, T., Shamim, N., Angle, G., Chen, J., Carlisle, S., Isbell, D., Slotnick, J., Mitchell, E., Henson, W., Koch, E., &#x0026; Knorr, M.: <span class="cmti-10">Reregistration eligibility decision (RED)</span> <span class="cmti-10">for chlorine dioxide and sodium chlorite (case 4023)</span>, Tech. Rep. EPA 738-R-06-007, United States Environmental Protection Agency, URL https://www3.epa.gov/pesticides/chem_search/reg_actions/reregistration/red_PC-020503_3-Aug-06.pdf (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '189',
    'Mozurkewich (1986)',
    'Mozurkewich, M.: <span class="cmti-10">Comment on “Possible role of</span> NO<sub><span class="cmr-7">3</span></sub> <span class="cmti-10">in the nighttime chemistry of a cloud” by William L.</span><span class="cmti-10"> Chameides</span>, J. Geophys. Res., 91, 14 569–14 570, <a href="https://doi.org/10.1029/JD091ID13P14569" >doi:10.1029/JD091ID13P14569</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '276',
    'Mozurkewich (1995)',
    'Mozurkewich, M.: <span class="cmti-10">Mechanisms for the release of</span> <span class="cmti-10">halogens from sea-salt particles by free radical reactions</span>, J. Geophys. Res., 100, 14 199–14 207, <a href="https://doi.org/10.1029/94JD00358" >doi:10.1029/94JD00358</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3926',
    'Muccitelli and Wen (1978)',
    'Muccitelli, J. A. &#x0026; Wen, W.-Y.: <span class="cmti-10">Solubilities of</span> <span class="cmti-10">hydrogen and deuterium gases in water and their isotope fractionation</span> <span class="cmti-10">factor</span>, J. Solution Chem., 7, 257–267, <a href="https://doi.org/10.1007/BF00644273" >doi:10.1007/BF00644273</a> (1978). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2965',
    'Muir et&nbsp;al. (2004)',
    'Muir, D. C. G., Teixeira, C., &#x0026; Wania, F.: <span class="cmti-10">Empirical and modeling evidence of regional atmospheric</span> <span class="cmti-10">transport of current-use pesticides</span>, Environ. Toxicol. Chem., 23, 2421–2432, <a href="https://doi.org/10.1897/03-457" >doi:10.1897/03-457</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1514',
    'Müller and Heal (2001)',
    'Müller, B. &#x0026; Heal, M. R.: <span class="cmti-10">The Henry&#x2019;s law</span> <span class="cmti-10">coefficient of 2-nitrophenol over the temperature range 278–303 </span>K, Chemosphere, 45, 309–314, <a href="https://doi.org/10.1016/S0045-6535(00)00592-0" >doi:10.1016/S0045-6535(00)00592-0</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3534',
    'Munder et&nbsp;al. (2000)',
    'Munder, B., Lidal, H., &#x0026; Sandall, O. C.: <span class="cmti-10">Physical solubility of hydrogen sulfide in aqueous</span> <span class="cmti-10">solutions of 2-(tert-butylamino)ethanol</span>, J. Chem. Eng. Data, 45, 1201–1204, <a href="https://doi.org/10.1021/JE000166F" >doi:10.1021/JE000166F</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2446',
    'Munson et&nbsp;al. (1964)',
    'Munson, E. S., Saidman, L. J., &#x0026; Eger, E. I.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">fluroxene in blood and tissue homogenates</span>, Anesthesiology, 25, 638–640, <a href="https://doi.org/10.1097/00000542-196409000-00010" >doi:10.1097/00000542-196409000-00010</a> (1964). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2903',
    'Munz and Roberts (1986)',
    'Munz, C. &#x0026; Roberts, P. V.: <span class="cmti-10">Effects of solute</span> <span class="cmti-10">concentration and cosolvents on the aqueous activity coefficient</span> <span class="cmti-10">of halogenated hydrocarbons</span>, Environ. Sci. Technol., 20, 830–836, <a href="https://doi.org/10.1021/ES00150A013" >doi:10.1021/ES00150A013</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2243',
    'Munz and Roberts (1987)',
    'Munz, C. &#x0026; Roberts, P. V.: <span class="cmti-10">Air–water phase</span> <span class="cmti-10">equilibria of volatile organic solutes</span>, J. Am. Water Works Assoc., 79, 62–69, <a href="https://doi.org/10.1002/J.1551-8833.1987.TB02844.X" >doi:10.1002/J.1551-8833.1987.TB02844.X</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3797',
    'Murphy et&nbsp;al. (1983a)',
    'Murphy, T. J., Pokojowczyk, J. C., &#x0026; Mullin, M. D.: <span class="cmti-10">Vapor exchange</span> <span class="cmti-10">of PCBs with Lake Michigan: The atmosphere as a sink for</span> <span class="cmti-10">PCBs</span>, Tech. Rep. EPA/600/D-83/097 (NTIS PB83250316), United States Environmental Protection Agency, URL https://ntrl.ntis.gov/NTRL/dashboard/searchResults.xhtml?searchQuery=PB83250316 (1983a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'murphy83',
    'Murphy et&nbsp;al. (1983b)',
    'Murphy, T. J., Pokojowczyk, J. C., &#x0026; Mullin, M. D.: <span class="cmti-10">Vapor exchange of PCBs</span> <span class="cmti-10">with Lake Michigan: The atmosphere as a sink for PCBs</span>, in: Physical Behavior of PCBs in the Great Lakes, edited by Mackay, D., Patterson, S., Eisenreich, S. J., &#x0026; Simmons, M. S., pp. 49–58, Ann Arbor Science, Ann Arbor, Mich. (1983b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '853',
    'Murphy et&nbsp;al. (1987)',
    'Murphy, T. J., Mullin, M. D., &#x0026; Meyer, J. A.: <span class="cmti-10">Equilibration of polychlorinated</span> <span class="cmti-10">biphenyls and toxaphene with air and water</span>, Environ. Sci. Technol., 21, 155–162, <a href="https://doi.org/10.1021/ES00156A005" >doi:10.1021/ES00156A005</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3566',
    'Murray and Riley (1969)',
    'Murray, C. N. &#x0026; Riley, J. P.: <span class="cmti-10">The solubility of</span> <span class="cmti-10">gases in distilled water and sea water — II. Oxygen</span>, Deep-Sea Res. Oceanogr. Abstr., 16, 311–320, <a href="https://doi.org/10.1016/0011-7471(69)90021-7" >doi:10.1016/0011-7471(69)90021-7</a> (1969). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3567',
    'Murray and Riley (1970)',
    'Murray, C. N. &#x0026; Riley, J. P.: <span class="cmti-10">The solubility of</span> <span class="cmti-10">gases in distilled water and sea water — III. Argon</span>, Deep-Sea Res. Oceanogr. Abstr., 17, 203–209, <a href="https://doi.org/10.1016/0011-7471(70)90100-2" >doi:10.1016/0011-7471(70)90100-2</a> (1970). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3568',
    'Murray and Riley (1971)',
    'Murray, C. N. &#x0026; Riley, J. P.: <span class="cmti-10">The solubility of</span> <span class="cmti-10">gases in distilled water and sea water — IV. Carbon dioxide</span>, Deep-Sea Res. Oceanogr. Abstr., 18, 533–541, <a href="https://doi.org/10.1016/0011-7471(71)90077-5" >doi:10.1016/0011-7471(71)90077-5</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3565',
    'Murray et&nbsp;al. (1969)',
    'Murray, C. N., Riley, J. P., &#x0026; Wilson, T. R. S.: <span class="cmti-10">The solubility of gases in distilled water and sea</span> <span class="cmti-10">water — I. Nitrogen</span>, Deep-Sea Res. Oceanogr. Abstr., 16, 297–310, <a href="https://doi.org/10.1016/0011-7471(69)90020-5" >doi:10.1016/0011-7471(69)90020-5</a> (1969). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2542',
    'Myrdal and Yalkowsky (1994)',
    'Myrdal, P. &#x0026; Yalkowsky, S. H.: <span class="cmti-10">A simple</span> <span class="cmti-10">scheme for calculating aqueous solubility, vapor pressure and Henry&#x2019;s law</span> <span class="cmti-10">constant: application to the chlorobenzenes</span>, SAR QSAR Environ. Res., 2, 17–28, <a href="https://doi.org/10.1080/10629369408028837" >doi:10.1080/10629369408028837</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3644',
    'Nahon et&nbsp;al. (2000)',
    'Nahon, D. F., Harrison, M., &#x0026; Roozen, J. P.: <span class="cmti-10">Modeling flavor release from aqueous sucrose</span> <span class="cmti-10">solutions, using mass transfer and partition coefficients</span>, J. Agric. Food Chem., 48, 1278–1284, <a href="https://doi.org/10.1021/JF990464K" >doi:10.1021/JF990464K</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2387',
    'Nelson and Hoff (1968)',
    'Nelson, P. E. &#x0026; Hoff, J. E.: <span class="cmti-10">Food volatiles: Gas chromatographic determination</span> <span class="cmti-10">of partition coefficients in water-lipid systems</span>, Int. J. Mass Spectrom., 228, 479–482, <a href="https://doi.org/10.1111/J.1365-2621.1968.TB03659.X" >doi:10.1111/J.1365-2621.1968.TB03659.X</a> (1968). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3584',
    'Nguyen (2013)',
    'Nguyen, T.: Amine Volatility in CO<sub><span class="cmr-7">2</span></sub> Capture, Ph.D. thesis, The University of Texas at Austin, USA, URL https://rochelle.che.utexas.edu/files/2015/02/Nguyen-2013-Amine-Volatility-in-CO2-Capture.pdf (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '531',
    'Nicholson et&nbsp;al. (1984)',
    'Nicholson, B. C., Maguire, B. P., &#x0026; Bursill, D. B.: <span class="cmti-10">Henry&#x2019;s law constants for the</span> <span class="cmti-10">trihalomethanes: Effects of water composition and temperature</span>, Environ. Sci. Technol., 18, 518–521, <a href="https://doi.org/10.1021/ES00125A006" >doi:10.1021/ES00125A006</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '622',
    'Nielsen et&nbsp;al. (1994)',
    'Nielsen, F., Olsen, E., &#x0026; Fredenslund, A.: <span class="cmti-10">Henry&#x2019;s law constants and infinite dilution activity</span> <span class="cmti-10">coefficients for volatile organic compounds in water by a validated</span> <span class="cmti-10">batch air stripping method</span>, Environ. Sci. Technol., 28, 2133–2138, <a href="https://doi.org/10.1021/ES00061A022" >doi:10.1021/ES00061A022</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2487',
    'Niinemets and Reichstein (2002)',
    'Niinemets, U. &#x0026; Reichstein, M.: <span class="cmti-10">A model</span> <span class="cmti-10">analysis of the effects of nonspecific monoterpenoid storage in</span> <span class="cmti-10">leaf tissues on emission kinetics and composition in Mediterranean</span> <span class="cmti-10">sclerophyllous Quercus species</span>, Global Biogeochem. Cycles, 16, 1110, <a href="https://doi.org/10.1029/2002GB001927" >doi:10.1029/2002GB001927</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '984',
    'Nirmalakhandan et&nbsp;al. (1997)',
    'Nirmalakhandan, N., Brennan, R. A., &#x0026; Speece, R. E.: <span class="cmti-10">Predicting</span> <span class="cmti-10">Henry&#x2019;s law constant and the effect of temperature on Henry&#x2019;s law</span> <span class="cmti-10">constant</span>, Wat. Res., 31, 1471–1481, <a href="https://doi.org/10.1016/S0043-1354(96)00395-8" >doi:10.1016/S0043-1354(96)00395-8</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2482',
    'Nirmalakhandan and Speece (1988)',
    'Nirmalakhandan, N. N. &#x0026; Speece, R. E.: <span class="cmti-10">QSAR model for predicting Henry&#x2019;s constant</span>, Environ. Sci. Technol., 22, 1349–1357, <a href="https://doi.org/10.1021/ES00176A016" >doi:10.1021/ES00176A016</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3532',
    'Nozière and Riemer (2003)',
    'Nozière, B. &#x0026; Riemer, D. D.: <span class="cmti-10">The chemical processing of gas-phase</span> <span class="cmti-10">carbonyl compounds by sulfuric acid aerosols: 2,4-pentanedione</span>, Atmos. Environ., 37, 841–851, <a href="https://doi.org/10.1016/S1352-2310(02)00934-2" >doi:10.1016/S1352-2310(02)00934-2</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3705',
    'Nunn (1958)',
    'Nunn, J. F.: <span class="cmti-10">Respiratory measurements in the presence of</span> <span class="cmti-10">nitrous oxide: storage of gas samples and chemical methods of analysis</span>, Br. J. Anaesth., 30, 254–263, <a href="https://doi.org/10.1093/BJA/30.6.254" >doi:10.1093/BJA/30.6.254</a> (1958). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3732',
    'Odabasi and Adali (2016)',
    'Odabasi, M. &#x0026; Adali, M.: <span class="cmti-10">Determination of</span> <span class="cmti-10">temperature dependent Henry&#x2019;s law constants of polychlorinated</span> <span class="cmti-10">naphthalenes: Application to air-sea exchange in Izmir Bay, Turkey</span>, Atmos. Environ., 147, 200–208, <a href="https://doi.org/10.1016/J.ATMOSENV.2016.10.009" >doi:10.1016/J.ATMOSENV.2016.10.009</a> (2016). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2819',
    'Odabasi et&nbsp;al. (2006)',
    'Odabasi, M., Cetin, B., &#x0026; Sofuoglu, A.: <span class="cmti-10">Henry&#x2019;s law constant, octanol-air partition coefficient</span> <span class="cmti-10">and supercooled liquid vapor pressure of carbazole as a function of</span> <span class="cmti-10">temperature: Application to gas/particle partitioning in the atmosphere</span>, Chemosphere, 62, 1087–1096, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2005.05.035" >doi:10.1016/J.CHEMOSPHERE.2005.05.035</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3526',
    'O&#x2019;Farrell and Waghorne (2010)',
    'O&#x2019;Farrell, C. E. &#x0026; Waghorne, W. E.: <span class="cmti-10">Henry&#x2019;s law constants of organic compounds in water and </span><span class="cmmi-10">n</span><span class="cmti-10">-octane at T</span> <span class="cmti-10">= 293.2K</span>, J. Chem. Eng. Data, 55, 1655–1658, <a href="https://doi.org/10.1021/JE900711H" >doi:10.1021/JE900711H</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1139',
    'Oliver (1985)',
    'Oliver, B. G.: <span class="cmti-10">Desorption of chlorinated hydrocarbons from</span> <span class="cmti-10">spiked and anthropogenically contaminated sediments</span>, Chemosphere, 14, 1087–1106, <a href="https://doi.org/10.1016/0045-6535(85)90029-3" >doi:10.1016/0045-6535(85)90029-3</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3246',
    'Olsen et&nbsp;al. (2016)',
    'Olsen, R., Kvamme, B., &#x0026; Kuznetsova, T.: <span class="cmti-10">Free energy of solvation and Henry&#x2019;s law solubility</span> <span class="cmti-10">constants for mono-, di- and tri-ethylene glycol in water and methane</span>, Fluid Phase Equilib., 418, 152–159, <a href="https://doi.org/10.1016/J.FLUID.2015.10.019" >doi:10.1016/J.FLUID.2015.10.019</a> (2016). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1110',
    'Olson (1998)',
    'Olson, J. D.: <span class="cmti-10">The</span> <span class="cmti-10">vapor pressure of pure and aqueous glutaraldehyde</span>, Fluid Phase Equilib., 150–151, 713–720, <a href="https://doi.org/10.1016/S0378-3812(98)00351-3" >doi:10.1016/S0378-3812(98)00351-3</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3303',
    'Ömür-Özbek and Dietrich (2005)',
    'Ömür-Özbek, P. &#x0026; Dietrich, A. M.: <span class="cmti-10">Determination of temperature-dependent Henry&#x2019;s law constants</span> <span class="cmti-10">of odorous contaminants and their application to human perception</span>, Environ. Sci. Technol., 39, 3957–3963, <a href="https://doi.org/10.1021/ES0480264" >doi:10.1021/ES0480264</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3305',
    'Ondo and Dohnal (2007)',
    'Ondo, D. &#x0026; Dohnal, V.: <span class="cmti-10">Temperature dependence</span> <span class="cmti-10">of limiting activity coefficients and Henry&#x2019;s law constants of cyclic</span> <span class="cmti-10">and open-chain ethers in water</span>, Fluid Phase Equilib., 262, 121–136, <a href="https://doi.org/10.1016/J.FLUID.2007.08.013" >doi:10.1016/J.FLUID.2007.08.013</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3564',
    'Ooki and Yokouchi (2011)',
    'Ooki, A. &#x0026; Yokouchi, Y.: <span class="cmti-10">Determination of</span> <span class="cmti-10">Henry&#x2019;s law constant of halocarbons in seawater and analysis of</span> <span class="cmti-10">sea-to-air flux of iodoethane (</span>C<sub><span class="cmr-7">2</span></sub>H<sub><span class="cmr-7">5</span></sub>I<span class="cmti-10">) in the Indian and Southern</span> <span class="cmti-10">Oceans based on partial pressure measurements</span>, Geochem. J., 45, e1–e7, <a href="https://doi.org/10.2343/GEOCHEMJ.1.0122" >doi:10.2343/GEOCHEMJ.1.0122</a> (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3121',
    'Opresko et&nbsp;al. (1998)',
    'Opresko, D. M., Young, R. A., Faust, R. A., Talmage, S. S., Watson, A. P., Ross, R. H., Davidson, K. A., &#x0026; King, J.: <span class="cmti-10">Chemical warfare</span> <span class="cmti-10">agents: estimating oral reference doses</span>, Rev. Environ. Contam. Toxicol., 156, 1–183, <a href="https://doi.org/10.1007/978-1-4612-1722-0_1" >doi:10.1007/978-1-4612-1722-0_1</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3702',
    'Orcutt and Seevers (1937a)',
    'Orcutt, F. S. &#x0026; Seevers, M. H.: <span class="cmti-10">A method for</span> <span class="cmti-10">determining the solubility of gases in pure liquids or solutions by the</span> <span class="cmti-10">Van Slyke-Neill manometric apparatus</span>, J. Biol. Chem., 117, 501–507, <a href="https://doi.org/10.1016/S0021-9258(18)74550-X" >doi:10.1016/S0021-9258(18)74550-X</a> (1937a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'orcutt37',
    'Orcutt and Seevers (1937b)',
    'Orcutt, F. S. &#x0026; Seevers, M. H.: <span class="cmti-10">The solubility</span> <span class="cmti-10">coefficients of cyclopropane for water, oils and human blood</span>, J. Pharmacol. Exp. Ther., 59, 206–210 (1937b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3770',
    'Ordóñez et&nbsp;al. (2012)',
    'Ordóñez, C., Lamarque, J.-F., Tilmes, S., Kinnison, D. E., Atlas, E. L., Blake, D. R., Sousa Santos, G., Brasseur, G., &#x0026; Saiz-Lopez, A.: <span class="cmti-10">Bromine</span> <span class="cmti-10">and iodine chemistry in a global chemistry-climate model: description</span> <span class="cmti-10">and evaluation of very short-lived oceanic sources</span>, Atmos. Chem. Phys., 12, 1423–1447, <a href="https://doi.org/10.5194/ACP-12-1423-2012" >doi:10.5194/ACP-12-1423-2012</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '516',
    'O&#x2019;Sullivan et&nbsp;al. (1996)',
    'O&#x2019;Sullivan, D. W., Lee, M., Noone, B. C., &#x0026; Heikes, B. G.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">law constant determinations for hydrogen peroxide, methyl hydroperoxide,</span> <span class="cmti-10">hydroxymethyl hydroperoxide, ethyl hydroperoxide, and peroxyacetic acid</span>, J. Phys. Chem., 100, 3241–3247, <a href="https://doi.org/10.1021/JP951168N" >doi:10.1021/JP951168N</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2836',
    'Otto et&nbsp;al. (1997)',
    'Otto, S., Riello, L., Düring, R.-A., Hummel, H. E., &#x0026; Zanin, G.: <span class="cmti-10">Herbicide dissipation</span> <span class="cmti-10">and dynamics modelling in three different tillage systems</span>, Chemosphere, 34, 163–178, <a href="https://doi.org/10.1016/S0045-6535(96)00356-6" >doi:10.1016/S0045-6535(96)00356-6</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3809',
    'Ourisson and Kastner (1939)',
    'Ourisson, J. &#x0026; Kastner, M.: <span class="cmti-10">Determination des</span> <span class="cmti-10">tensions de vapeurs des solutions d&#x2019;acide hypochloreux </span><span class="cmti-10">à 10</span><span class="cmsy-7">∘</span><span class="cmti-10"> et 20</span><span class="cmsy-7">∘</span><span class="cmti-10"> C</span>, Bull. Soc. Chim. Memoirs, 6, 1307–1311 (1939). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3626',
    'Owens et&nbsp;al. (1986)',
    'Owens, J. W., Wasik, S. P., &#x0026; DeVoe, H.: <span class="cmti-10">Aqueous</span> <span class="cmti-10">solubilities and enthalpies of solution of </span><span class="cmmi-10">n</span><span class="cmti-10">-alkylbenzenes</span>, J. Chem. Eng. Data, 31, 47–51, <a href="https://doi.org/10.1021/JE00043A016" >doi:10.1021/JE00043A016</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2563',
    'Paasivirta and Sinkkonen (2009)',
    'Paasivirta, J. &#x0026; Sinkkonen, S. I.: <span class="cmti-10">Environmentally relevant properties of all 209</span> <span class="cmti-10">polychlorinated biphenyl congeners for modeling their fate in different</span> <span class="cmti-10">natural and climatic conditions</span>, J. Chem. Eng. Data, 54, 1189–1213, <a href="https://doi.org/10.1021/JE800501H" >doi:10.1021/JE800501H</a> (2009). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1143',
    'Paasivirta et&nbsp;al. (1999)',
    'Paasivirta, J., Sinkkonen, S., Mikkelson, P., Rantio, T., &#x0026; Wania, F.: <span class="cmti-10">Estimation of vapor pressures, solubilities and Henry&#x2019;s law constants</span> <span class="cmti-10">of selected persistent organic pollutants as functions of temperature</span>, Chemosphere, 39, 811–832, <a href="https://doi.org/10.1016/S0045-6535(99)00016-8" >doi:10.1016/S0045-6535(99)00016-8</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '582',
    'Palmer et&nbsp;al. (1985)',
    'Palmer, D. A., Ramette, R. W., &#x0026; Mesmer, R. E.: <span class="cmti-10">The hydrolysis of</span> <span class="cmti-10">iodine: Equilibria at high temperatures</span>, J. Nucl. Mater., 130, 280–286, <a href="https://doi.org/10.1016/0022-3115(85)90317-4" >doi:10.1016/0022-3115(85)90317-4</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '34',
    'Pandis and Seinfeld (1989)',
    'Pandis, S. N. &#x0026; Seinfeld, J. H.: <span class="cmti-10">Sensitivity analysis of a chemical mechanism for</span> <span class="cmti-10">aqueous-phase atmospheric chemistry</span>, J. Geophys. Res., 94, 1105–1126, <a href="https://doi.org/10.1029/JD094ID01P01105" >doi:10.1029/JD094ID01P01105</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2431',
    'Pankow et&nbsp;al. (1996)',
    'Pankow, J. F., Rathbun, R. E., &#x0026; Zogorski, J. S.: <span class="cmti-10">Calculated volatilization rates</span> <span class="cmti-10">of fuel oxygenate compounds and other gasoline-related compounds from</span> <span class="cmti-10">rivers and streams</span>, Chemosphere, 33, 921–937, <a href="https://doi.org/10.1016/0045-6535(96)00227-5" >doi:10.1016/0045-6535(96)00227-5</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2449',
    'Park et&nbsp;al. (1987)',
    'Park, J. H., Hussam, A., Couasnon, P., Fritz, D., &#x0026; Carr, P. W.: <span class="cmti-10">Experimental</span> <span class="cmti-10">reexamination of selected partition coefficients from Rohrschneider&#x2019;s data</span> <span class="cmti-10">set</span>, Anal. Chem., 59, 1970–1976, <a href="https://doi.org/10.1021/AC00142A016" >doi:10.1021/AC00142A016</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '592',
    'Park and Lee (1988)',
    'Park, J.-Y. &#x0026; Lee, Y.-N.: <span class="cmti-10">Solubility and decomposition</span> <span class="cmti-10">kinetics of nitrous acid in aqueous solution</span>, J. Phys. Chem., 92, 6294–6302, <a href="https://doi.org/10.1021/J100333A025" >doi:10.1021/J100333A025</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2905',
    'Park et&nbsp;al. (1997)',
    'Park, S.-J., Han, S.-D., &#x0026; Ryu, S.-A.: <span class="cmti-10">Measurement of air/water partition coefficient (Henry&#x2019;s law constant)</span> <span class="cmti-10">by using EPICS method and their relationship with vapor pressure and</span> <span class="cmti-10">water solubility</span>, J. Korean Inst. Chem. Eng., 35, 915–920 (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2486',
    'Park et&nbsp;al. (1982)',
    'Park, T., Rettich, T. R., Battino, R., Peterson, D., &#x0026; Wilhelm, E.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">gases in liquids. 14. Bunsen coefficients for several fluorine-containing</span> <span class="cmti-10">gases (Freons) dissolved in water at 298.15K</span>, J. Chem. Eng. Data, 27, 324–326, <a href="https://doi.org/10.1021/JE00029A027" >doi:10.1021/JE00029A027</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3544',
    'Parker (1992)',
    'Parker, V. D.: <span class="cmti-10">The reversible reduction potential of the</span> <span class="cmti-10">proton in water and in non-aqueous solvents</span>, Acta Chem. Scand., 46, 692–694, <a href="https://doi.org/10.3891/ACTA.CHEM.SCAND.46-0692" >doi:10.3891/ACTA.CHEM.SCAND.46-0692</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3400',
    'Parnis et&nbsp;al. (2015)',
    'Parnis, J. M., Mackay, D., &#x0026; Harner, T.: <span class="cmti-10">Temperature dependence of Henry&#x2019;s law constants and</span> <span class="cmmi-10">K</span><sub><span class="cmr-7">OA</span></sub> <span class="cmti-10">for simple and heteroatom-substituted PAHs by COSMO-RS</span>, Atmos. Environ., 110, 27–35, <a href="https://doi.org/10.1016/J.ATMOSENV.2015.03.032" >doi:10.1016/J.ATMOSENV.2015.03.032</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '724',
    'Parsons et&nbsp;al. (1971)',
    'Parsons, G. H., Rochester, C. H., &#x0026; Wood, C. E. C.: <span class="cmti-10">Effect of 4-substitution on</span> <span class="cmti-10">the thermodynamics of hydration of phenol and the phenoxide anion</span>, J. Chem. Soc. B, pp. 533–536, <a href="https://doi.org/10.1039/J29710000533" >doi:10.1039/J29710000533</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '725',
    'Parsons et&nbsp;al. (1972)',
    'Parsons, G. H., Rochester, C. H., Rostron, A., &#x0026; Sykes, P. C.: <span class="cmti-10">The</span> <span class="cmti-10">thermodynamics of hydration of phenols</span>, J. Chem. Soc. Perkin Trans. 2, pp. 136–138, <a href="https://doi.org/10.1039/P29720000136" >doi:10.1039/P29720000136</a> (1972). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '544',
    'Pearson and McConnell (1975)',
    'Pearson, C. R. &#x0026; McConnell, G.: <span class="cmti-10">Chlorinated</span> C<sub><span class="cmr-7">1</span></sub> <span class="cmti-10">and</span> C<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">hydrocarbons in the marine environment</span>, Proc. R. Soc. Lond. B, 189, 305–332, <a href="https://doi.org/10.1098/RSPB.1975.0059" >doi:10.1098/RSPB.1975.0059</a> (1975). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '720',
    'Pecsar and Martin (1966)',
    'Pecsar, R. E. &#x0026; Martin, J. J.: <span class="cmti-10">Solution thermodynamics from gas-liquid chromatography</span>, Anal. Chem., 38, 1661–1669, <a href="https://doi.org/10.1021/AC60244A009" >doi:10.1021/AC60244A009</a> (1966). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '837',
    'Peng and Wan (1997)',
    'Peng, J. &#x0026; Wan, A.: <span class="cmti-10">Measurement of Henry&#x2019;s constants</span> <span class="cmti-10">of high-volatility organic compounds using a headspace autosampler</span>, Environ. Sci. Technol., 31, 2998–3003, <a href="https://doi.org/10.1021/ES970240N" >doi:10.1021/ES970240N</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1114',
    'Peng and Wan (1998)',
    'Peng, J. &#x0026; Wan, A.: <span class="cmti-10">Effect of ionic strength on Henry&#x2019;s</span> <span class="cmti-10">constants of volatile organic compounds</span>, Chemosphere, 36, 2731–2740, <a href="https://doi.org/10.1016/S0045-6535(97)10232-6" >doi:10.1016/S0045-6535(97)10232-6</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1229',
    'Perlinger et&nbsp;al. (1993)',
    'Perlinger, J. A., Eisenreich, S. J., &#x0026; Capel, P. D.: <span class="cmti-10">Application of headspace analysis to the</span> <span class="cmti-10">study of sorption of hydrophobic organic chemicals to</span> <span class="cmmi-10">α</span><span class="cmsy-10">−</span>Al<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">3</span></sub>, Environ. Sci. Technol., 27, 928–937, <a href="https://doi.org/10.1021/ES00042A016" >doi:10.1021/ES00042A016</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'perry73',
    'Perry and Chilton (1973)',
    'Perry, R. H. &#x0026; Chilton, C. H.: Chemical Engineers&#x2019; Handbook, 5th edition, McGraw-Hill, Inc., ISBN 0070855471 (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2286',
    'Petersen et&nbsp;al. (1998)',
    'Petersen, G., Pleijel, J. M. K., Bloxam, R., &#x0026; Vinod Kumar, A.: <span class="cmti-10">A comprehensive Eulerian modeling</span> <span class="cmti-10">framework for airborne mercury species: Development and testing of the</span> <span class="cmti-10">tropospheric chemistry module (TCM)</span>, Atmos. Environ., 32, 829–843, <a href="https://doi.org/10.1016/S1352-2310(97)00049-6" >doi:10.1016/S1352-2310(97)00049-6</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2556',
    'Petrasek et&nbsp;al. (1983)',
    'Petrasek, A. C., Kugelman, I. J., Austern, B. M., Pressley, T. A., Winslow, L. A., &#x0026; Wise, R. H.: <span class="cmti-10">Fate of toxic organic compounds</span> <span class="cmti-10">in wastewater treatment plants</span>, J. Water Pollut. Control Fed., 55, 1286–1296 (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1524',
    'Pfeifer et&nbsp;al. (2001)',
    'Pfeifer, O., Lohmann, U., &#x0026; Ballschmiter, K.: <span class="cmti-10">Halogenated methyl-phenyl</span> <span class="cmti-10">ethers (anisoles) in the environment: Determination of vapor pressures,</span> <span class="cmti-10">aqueous solubilities, Henry&#x2019;s law constants, and gas/water- (</span><span class="cmmi-10">K</span><sub><span class="cmr-7">gw</span></sub><span class="cmti-10">),</span> <span class="cmmi-10">n</span><span class="cmti-10">-octanol/water- (</span><span class="cmmi-10">K</span><sub><span class="cmr-7">ow</span></sub><span class="cmti-10">) and gas/n-octanol (</span><span class="cmmi-10">K</span><sub><span class="cmr-7">go</span></sub><span class="cmti-10">) partition coefficients</span>, Fresenius J. Anal. Chem., 371, 598–606, <a href="https://doi.org/10.1007/S002160101077" >doi:10.1007/S002160101077</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3642',
    'Philippe et&nbsp;al. (2003)',
    'Philippe, E., Seuvre, A.-M., Colas, B., Langendorff, V., Schippa, C., &#x0026; Voilley, A.: <span class="cmti-10">Behavior of flavor compounds in model food systems:</span> <span class="cmti-10">a thermodynamic study</span>, J. Agric. Food Chem., 51, 1393–1398, <a href="https://doi.org/10.1021/JF020862E" >doi:10.1021/JF020862E</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2223',
    'Pierotti et&nbsp;al. (1959)',
    'Pierotti, G. J., Deal, C. H., &#x0026; Derr, E. L.: <span class="cmti-10">Activity coefficients and molecular structure</span>, Ind. Eng. Chem., 51, 95–102, <a href="https://doi.org/10.1021/IE50589A048" >doi:10.1021/IE50589A048</a>, (data available in supplement, document no. 5782, American Documentation Institute, Library of Congress, Washington, D.C.) (1959). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '766',
    'Pierotti (1965)',
    'Pierotti, R. A.: <span class="cmti-10">Aqueous solutions of nonpolar gases</span>, J. Phys. Chem., 69, 281–288, <a href="https://doi.org/10.1021/J100885A043" >doi:10.1021/J100885A043</a> (1965). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3648',
    'Pividal et&nbsp;al. (1992)',
    'Pividal, K. A., Birtigh, A., &#x0026; Sandler, S. I.: <span class="cmti-10">Infinite dilution activity coefficients for oxygenate</span> <span class="cmti-10">systems determined using a differential static cell</span>, J. Chem. Eng. Data, 37, 484–487, <a href="https://doi.org/10.1021/JE00008A025" >doi:10.1021/JE00008A025</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2994',
    'Plassmann et&nbsp;al. (2010)',
    'Plassmann, M. M., Meyer, T., Lei, Y. D., Wania, F., McLachlan, M. S., &#x0026; Berger, U.: <span class="cmti-10">Theoretical and experimental simulation of the fate</span> <span class="cmti-10">of semifluorinated </span><span class="cmmi-10">n</span><span class="cmti-10">-alkanes during snowmelt</span>, Environ. Sci. Technol., 44, 6692–6697, <a href="https://doi.org/10.1021/ES101562W" >doi:10.1021/ES101562W</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2991',
    'Plassmann et&nbsp;al. (2011)',
    'Plassmann, M. M., Meyer, T., Lei, Y. D., Wania, F., McLachlan, M. S., &#x0026; Berger, U.: <span class="cmti-10">Laboratory studies on the fate of perfluoroalkyl</span> <span class="cmti-10">carboxylates and sulfonates during snowmelt</span>, Environ. Sci. Technol., 45, 6872–6878, <a href="https://doi.org/10.1021/ES201249D" >doi:10.1021/ES201249D</a> (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3015',
    'Plyasunov (2012)',
    'Plyasunov, A. V.: <span class="cmti-10">Thermodynamics of</span> Si(OH)<sub><span class="cmr-7">4</span></sub> <span class="cmti-10">in the</span> <span class="cmti-10">vapor phase of water: Henry&#x2019;s and vapor-liquid distribution constants,</span> <span class="cmti-10">fugacity and cross virial coefficients</span>, Geochim. Cosmochim. Acta, 77, 215–231, <a href="https://doi.org/10.1016/J.GCA.2011.11.019" >doi:10.1016/J.GCA.2011.11.019</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3673',
    'Plyasunov and Shock (2000)',
    'Plyasunov, A. V. &#x0026; Shock, E. L.: <span class="cmti-10">Thermodynamic functions of hydration of hydrocarbons</span> <span class="cmti-10">at 298.15K and 0.1MPa</span>, Geochim. Cosmochim. Acta, 64, 439–468, <a href="https://doi.org/10.1016/S0016-7037(99)00330-0" >doi:10.1016/S0016-7037(99)00330-0</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3617',
    'Plyasunov and Shock (2001)',
    'Plyasunov, A. V. &#x0026; Shock, E. L.: <span class="cmti-10">Group</span> <span class="cmti-10">contribution values of the infinite dilution thermodynamic functions of</span> <span class="cmti-10">hydration for aliphatic noncyclic hydrocarbons, alcohols, and ketones</span> <span class="cmti-10">at 298.15 </span>K <span class="cmti-10">and 0.1</span> MPa, J. Chem. Eng. Data, 46, 1016–1019, <a href="https://doi.org/10.1021/JE0002282" >doi:10.1021/JE0002282</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3674',
    'Plyasunov et&nbsp;al. (2001)',
    'Plyasunov, A. V., O&#x2019;Connell, J. P., Wood, R. H., &#x0026; Shock, E. L.: <span class="cmti-10">Semiempirical equation of state for the infinite dilution thermodynamic</span> <span class="cmti-10">functions of hydration of nonelectrolytes over wide ranges of</span> <span class="cmti-10">temperature and pressure</span>, Fluid Phase Equilib., 183–184, 133–142, <a href="https://doi.org/10.1016/S0378-3812(01)00427-7" >doi:10.1016/S0378-3812(01)00427-7</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3616',
    'Plyasunov et&nbsp;al. (2004)',
    'Plyasunov, A. V., Plyasunova, N. V., &#x0026; Shock, E. L.: <span class="cmti-10">Group contribution values for</span> <span class="cmti-10">the thermodynamic functions of hydration of aliphatic esters at 298.15 </span>K<span class="cmti-10">,</span> <span class="cmti-10">0.1</span> MPa, J. Chem. Eng. Data, 49, 1152–1167, <a href="https://doi.org/10.1021/JE049850A" >doi:10.1021/JE049850A</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3621',
    'Plyasunov et&nbsp;al. (2006)',
    'Plyasunov, A. V., Plyasunova, N. V., &#x0026; Shock, E. L.: <span class="cmti-10">Group contribution values</span> <span class="cmti-10">for the thermodynamic functions of hydration at 298.15 </span>K<span class="cmti-10">, 0.1</span> MPa<span class="cmti-10">. 4.</span> <span class="cmti-10">aliphatic nitriles and dinitriles</span>, J. Chem. Eng. Data, 51, 1481–1490, <a href="https://doi.org/10.1021/JE060129+" >doi:10.1021/JE060129+</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3618',
    'Plyasunova et&nbsp;al. (2004)',
    'Plyasunova, N. V., Plyasunov, A. V., &#x0026; Shock, E. L.: <span class="cmti-10">Group contribution values</span> <span class="cmti-10">for the thermodynamic functions of hydration at 298.15 </span>K<span class="cmti-10">, 0.1</span> MPa<span class="cmti-10">. 2.</span> <span class="cmti-10">aliphatic thiols, alkyl sulfides, and polysulfides</span>, J. Chem. Eng. Data, 50, 246–253, <a href="https://doi.org/10.1021/JE0497045" >doi:10.1021/JE0497045</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2946',
    'Podoll et&nbsp;al. (1986)',
    'Podoll, R. T., Jaber, H. M., &#x0026; Mill, T.: <span class="cmti-10">Tetrachlorodibenzodioxin: rates of volatilization and</span> <span class="cmti-10">photolysis in the environment</span>, Environ. Sci. Technol., 20, 490–492, <a href="https://doi.org/10.1021/ES00147A008" >doi:10.1021/ES00147A008</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1923',
    'Pollien et&nbsp;al. (2003)',
    'Pollien, P., Jordan, A., Lindinger, W., &#x0026; Yeretzian, C.: <span class="cmti-10">Liquid-air</span> <span class="cmti-10">partitioning of volatile compounds in coffee: dynamic measurements using</span> <span class="cmti-10">proton-transfer-reaction mass spectrometry</span>, Int. J. Mass Spectrom., 228, 69–80, <a href="https://doi.org/10.1016/S1387-3806(03)00197-0" >doi:10.1016/S1387-3806(03)00197-0</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3547',
    'Potter II and Clynne (1978)',
    'Potter II, R. W. &#x0026; Clynne, M. A.: <span class="cmti-10">The</span> <span class="cmti-10">solubility of the noble gases He, Ne, Ar, Kr, and Xe in water up to the</span> <span class="cmti-10">critical point</span>, J. Solution Chem., 7, 837–844, <a href="https://doi.org/10.1007/BF00650811" >doi:10.1007/BF00650811</a> (1978). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2822',
    'Poulain et&nbsp;al. (2010)',
    'Poulain, L., Katrib, Y., Isikli, E., Liu, Y., Wortham, H., Mirabel, P., Le Calvé, S., &#x0026; Monod, A.: <span class="cmti-10">In-cloud multiphase behaviour of</span> <span class="cmti-10">acetone in the troposphere: Gas uptake, Henry&#x2019;s law equilibrium</span> <span class="cmti-10">and aqueous phase photooxidation</span>, Chemosphere, 81, 312–320, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2010.07.032" >doi:10.1016/J.CHEMOSPHERE.2010.07.032</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3787',
    'Power and Stegall (1970)',
    'Power, G. G. &#x0026; Stegall, H.: <span class="cmti-10">Solubility of gases</span> <span class="cmti-10">in human red blood cell ghosts</span>, J. Appl. Physiol., 29, 145–149, <a href="https://doi.org/10.1152/JAPPL.1970.29.2.145" >doi:10.1152/JAPPL.1970.29.2.145</a> (1970). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '522',
    'Przyjazny et&nbsp;al. (1983)',
    'Przyjazny, A., Janicki, W., Chrzanowski, W., &#x0026; Staszewski, R.: <span class="cmti-10">Headspace gas</span> <span class="cmti-10">chromatographic determination of distribution coefficients of selected</span> <span class="cmti-10">organosulphur compounds and their dependence on some parameters</span>, J. Chromatogr., 280, 249–260, <a href="https://doi.org/10.1016/S0021-9673(00)91567-X" >doi:10.1016/S0021-9673(00)91567-X</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3583',
    'Pye et&nbsp;al. (2013)',
    'Pye, H. O. T., Pinder, R. W., Piletic, I. R., Xie, Y., Capps, S. L., Lin, Y.-H., Surratt, J. D., Zhang, Z., Gold, A., Luecken, D. J., Hutzell, W. T., Jaoui, M., Offenberg, J. H., Kleindienst, T. E., Lewandowski, M., &#x0026; Edney, E. O.: <span class="cmti-10">Epoxide pathways improve model predictions of</span> <span class="cmti-10">isoprene markers and reveal key role of acidity in aerosol formation</span>, Environ. Sci. Technol., 47, 11 056–11 064, <a href="https://doi.org/10.1021/ES402106H" >doi:10.1021/ES402106H</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3575',
    'Qin et&nbsp;al. (2021)',
    'Qin, C., Gou, Y., Wang, Y., Mao, Y., Liao, H., Wang, Q., &#x0026; Xie, M.: <span class="cmti-10">Gas–particle</span> <span class="cmti-10">partitioning of polyol tracers at a suburban site in Nanjing, east China:</span> <span class="cmti-10">increased partitioning to the particle phase</span>, Atmos. Chem. Phys., 21, 12 141–12 153, <a href="https://doi.org/10.5194/ACP-21-12141-2021" >doi:10.5194/ACP-21-12141-2021</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2930',
    'Ramachandran et&nbsp;al. (1996)',
    'Ramachandran, B. R., Allen, J. M., &#x0026; Halpern, A. M.: <span class="cmti-10">Air–water</span> <span class="cmti-10">partitioning of environmentally important organic compounds</span>, J. Chem. Educ., 73, 1058–1061, <a href="https://doi.org/10.1021/ED073P1058" >doi:10.1021/ED073P1058</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3546',
    'Ramstedt (1911)',
    'Ramstedt, E.: <span class="cmti-10">Sur la solubilit</span><span class="cmti-10">é de l&#x2019;</span><span class="cmti-10">émanation du</span> <span class="cmti-10">radium dans les liquides organiques</span>, Radium (Paris), 8, 253–256, <a href="https://doi.org/10.1051/RADIUM:0191100807025301" >doi:10.1051/RADIUM:0191100807025301</a> (1911). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2234',
    'Rathbun and Tai (1982)',
    'Rathbun, R. E. &#x0026; Tai, D. Y.: <span class="cmti-10">Volatilization</span> <span class="cmti-10">of ketones from water</span>, Water Air Soil Pollut., 17, 281–293, <a href="https://doi.org/10.1007/BF00283158" >doi:10.1007/BF00283158</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2546',
    'Raventos-Duran et&nbsp;al. (2010)',
    'Raventos-Duran, T., Camredon, M., Valorso, R., Mouchel-Vallon, C., &#x0026; Aumont, B.: <span class="cmti-10">Structure-activity relationships to estimate the effective</span> <span class="cmti-10">Henry&#x2019;s law constants of organics of atmospheric interest</span>, Atmos. Chem. Phys., 10, 7643–7654, <a href="https://doi.org/10.5194/ACP-10-7643-2010" >doi:10.5194/ACP-10-7643-2010</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3608',
    'Razumovskii and Zaikov (1971)',
    'Razumovskii, S. D. &#x0026; Zaikov, G. E.: <span class="cmti-10">The</span> <span class="cmti-10">solubility of ozone in various solvents</span>, Russ. Chem. Bull., 20, 616–620, <a href="https://doi.org/10.1007/BF00853885" >doi:10.1007/BF00853885</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '797',
    'Régimbal and Mozurkewich (1997)',
    'Régimbal, J.-M. &#x0026; Mozurkewich, M.: <span class="cmti-10">Peroxynitric acid decay mechanisms and kinetics at low pH</span>, J. Phys. Chem. A, 101, 8822–8829, <a href="https://doi.org/10.1021/JP971908N" >doi:10.1021/JP971908N</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2013',
    'Reichl (1995)',
    'Reichl, A.: Messung und Korrelierung von Gaslöslichkeiten halogenierter Kohlenwasserstoffe, Ph.D. thesis, Technische Universität Berlin, Germany (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '491',
    'Rettich et&nbsp;al. (1981)',
    'Rettich, T. R., Handa, Y. P., Battino, R., &#x0026; Wilhelm, E.: <span class="cmti-10">Solubility of gases in liquids.</span> <span class="cmti-10">13.</span><span class="cmti-10"> High-precision determination of Henry&#x2019;s constants for methane and</span> <span class="cmti-10">ethane in liquid water at 275 to 328 </span>K, J. Phys. Chem., 85, 3230–3237, <a href="https://doi.org/10.1021/J150622A006" >doi:10.1021/J150622A006</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3529',
    'Rettich et&nbsp;al. (1982)',
    'Rettich, T. R., Battino, R., &#x0026; Wilhelm, E.: <span class="cmti-10">Solubility of gases in liquids. 15. High-precision</span> <span class="cmti-10">determination of Henry coefficients for carbon monoxide in liquid</span> <span class="cmti-10">water at 278 to 323K</span>, Ber. Bunsenges. Phys. Chem., 86, 1128–1132, <a href="https://doi.org/10.1002/BBPC.198200051" >doi:10.1002/BBPC.198200051</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2434',
    'Rettich et&nbsp;al. (1984)',
    'Rettich, T. R., Battino, R., &#x0026; Wilhelm, E.: <span class="cmti-10">Solubility of gases in liquids. XVI. Henry&#x2019;s law</span> <span class="cmti-10">coefficients for nitrogen in water at 5 to 50</span><span class="cmsy-7">∘</span>C, J. Solution Chem., 13, 335–348, <a href="https://doi.org/10.1007/BF00645706" >doi:10.1007/BF00645706</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2435',
    'Rettich et&nbsp;al. (1992)',
    'Rettich, T. R., Battino, R., &#x0026; Wilhelm, E.: <span class="cmti-10">Solubility of gases in liquids. 18. High-precision</span> <span class="cmti-10">determination of Henry fugacities for argon in liquid water at 2 to 40</span><span class="cmsy-7">∘</span>C, J. Solution Chem., 21, 987–1004, <a href="https://doi.org/10.1007/BF00650874" >doi:10.1007/BF00650874</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3412',
    'Rettich et&nbsp;al. (2000)',
    'Rettich, T. R., Battino, R., &#x0026; Wilhelm, E.: <span class="cmti-10">Solubility of gases in liquids. 22. High-precision</span> <span class="cmti-10">determination of Henry&#x2019;s law constants of oxygen in liquid water from</span> <span class="cmti-10">T = 274K to T = 328K</span>, J. Chem. Thermodyn., 32, 1145–1156, <a href="https://doi.org/10.1006/JCHT.1999.0581" >doi:10.1006/JCHT.1999.0581</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '863',
    'Rex (1906)',
    'Rex, A.: <span class="cmti-10">Über die L</span><span class="cmti-10">öslichkeit der Halogenderivate der</span> <span class="cmti-10">Kohlenwasserstoffe in Wasser</span>, Z. Phys. Chem., 55, 355–370, <a href="https://doi.org/10.1515/ZPCH-1906-5519" >doi:10.1515/ZPCH-1906-5519</a> (1906). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2815',
    'Reyes-Pérez et&nbsp;al. (2008)',
    'Reyes-Pérez, E., Le Calvé, S., &#x0026; Mirabel, P.: <span class="cmti-10">UV absorption spectrum</span> <span class="cmti-10">and Henry&#x2019;s law constant of EPTC</span>, Atmos. Environ., 42, 7940–7946, <a href="https://doi.org/10.1016/J.ATMOSENV.2008.07.017" >doi:10.1016/J.ATMOSENV.2008.07.017</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2489',
    'Reza and Trejo (2004)',
    'Reza, J. &#x0026; Trejo, A.: <span class="cmti-10">Temperature dependence of</span> <span class="cmti-10">the infinite dilution activity coefficient and Henry&#x2019;s law constant of</span> <span class="cmti-10">polycyclic aromatic hydrocarbons in water</span>, Chemosphere, 56, 537–547, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2004.04.020" >doi:10.1016/J.CHEMOSPHERE.2004.04.020</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3791',
    'Rice et&nbsp;al. (1997a)',
    'Rice, C. P., Chernyak, S. M., Hapeman, C. J., &#x0026; Biboulian, S.: <span class="cmti-10">Air–water</span> <span class="cmti-10">distribution of the endosulfan isomers</span>, J. Environ. Qual., 26, 1101–1106, <a href="https://doi.org/10.2134/JEQ1997.00472425002600040022X" >doi:10.2134/JEQ1997.00472425002600040022X</a> (1997a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '893',
    'Rice et&nbsp;al. (1997b)',
    'Rice, C. P., Chernyak, S. M., &#x0026; McConnell, L. L.: <span class="cmti-10">Henry&#x2019;s law constants for pesticides</span> <span class="cmti-10">measured as a function of temperature and salinity</span>, J. Agric. Food Chem., 45, 2291–2298, <a href="https://doi.org/10.1021/JF960834U" >doi:10.1021/JF960834U</a> (1997b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3635',
    'Richon et&nbsp;al. (1985)',
    'Richon, D., Sorrentino, F., &#x0026; Voilley, A.: <span class="cmti-10">Infinite dilution activity coefficients by the inert</span> <span class="cmti-10">gas stripping method: extension to the study of viscous and foaming</span> <span class="cmti-10">mixtures</span>, Ind. Eng. Chem. Process Des. Dev., 24, 1160–1165, <a href="https://doi.org/10.1021/I200031A044" >doi:10.1021/I200031A044</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2966',
    'Riederer (1990)',
    'Riederer, M.: <span class="cmti-10">Estimating partitioning and transport</span> <span class="cmti-10">of organic chemicals in the foliage/atmosphere system: discussion</span> <span class="cmti-10">of a fugacity-based model</span>, Environ. Sci. Technol., 24, 829–837, <a href="https://doi.org/10.1021/ES00076A006" >doi:10.1021/ES00076A006</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2929',
    'Rinker and Sandall (2000)',
    'Rinker, E. B. &#x0026; Sandall, O. C.: <span class="cmti-10">Physical</span> <span class="cmti-10">solubility of hydrogen sulfide in several aqueous solvents</span>, Can. J. Chem. Eng., 78, 232–236, <a href="https://doi.org/10.1002/CJCE.5450780130" >doi:10.1002/CJCE.5450780130</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3414',
    'Rischbieter et&nbsp;al. (2000)',
    'Rischbieter, E., Stein, H., &#x0026; Schumpe, A.: <span class="cmti-10">Ozone solubilities in water and aqueous</span> <span class="cmti-10">salt solutions</span>, J. Chem. Eng. Data, 45, 338–340, <a href="https://doi.org/10.1021/JE990263C" >doi:10.1021/JE990263C</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3578',
    'Rivera-Rios (2018)',
    'Rivera-Rios, J. C.: Atmospheric chemistry of isoprene hydroxyhydroperoxides, Ph.D. thesis, Graduate School of Arts &#x0026; Sciences, Harvard University, USA, URL https://dash.harvard.edu/handle/1/41127811 (2018). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1481',
    'Riveros et&nbsp;al. (1998)',
    'Riveros, P. A., Koren, D., McNamara, V. M., &#x0026; Binvignat, J.: <span class="cmti-10">Cyanide recovery</span> <span class="cmti-10">from a gold mill barren solution containing high levels of copper</span>, CIM Bull., 91, 73–81 (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '599',
    'Robbins et&nbsp;al. (1993)',
    'Robbins, G. A., Wang, S., &#x0026; Stuart, J. D.: <span class="cmti-10">Using the headspace method to determine Henry&#x2019;s</span> <span class="cmti-10">law constants</span>, Anal. Chem., 65, 3113–3118, <a href="https://doi.org/10.1021/AC00069A026" >doi:10.1021/AC00069A026</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2389',
    'Roberts and Pollien (1997)',
    'Roberts, D. D. &#x0026; Pollien, P.: <span class="cmti-10">Analysis of aroma</span> <span class="cmti-10">release during microwave heating</span>, J. Agric. Food Chem., 45, 4388–4392, <a href="https://doi.org/10.1021/JF9702508" >doi:10.1021/JF9702508</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3525',
    'Roberts and Liu (2019)',
    'Roberts, J. M. &#x0026; Liu, Y.: <span class="cmti-10">Solubility and solution-phase chemistry of isocyanic acid, methyl</span> <span class="cmti-10">isocyanate, and cyanogen halides</span>, Atmos. Chem. Phys., 19, 4419–4437, <a href="https://doi.org/10.5194/ACP-19-4419-2019" >doi:10.5194/ACP-19-4419-2019</a> (2019). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2360',
    'Roberts et&nbsp;al. (2008)',
    'Roberts, J. M., Osthoff, H. D., Brown, S. S., &#x0026; Ravishankara, A. R.: N<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">5</span></sub> <span class="cmti-10">oxidizes chloride to</span> Cl<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">in acidic atmospheric aerosol</span>, Science, 321, 1059, <a href="https://doi.org/10.1126/SCIENCE.1158777" >doi:10.1126/SCIENCE.1158777</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2805',
    'Roberts et&nbsp;al. (2011)',
    'Roberts, J. M., Veres, P. R., Cochran, A. K., Warneke, C., Burling, I. R., Yokelson, R. J., Lerner, B., Gilman, J. B., Kuster, W. C., Fall, R., &#x0026; de Gouw, J.: <span class="cmti-10">Isocyanic acid in the atmosphere and its possible</span> <span class="cmti-10">link to smoke-related health effects</span>, Proc. Natl. Acad. Sci. USA, 108, 8966–8971, <a href="https://doi.org/10.1073/PNAS.1103352108" >doi:10.1073/PNAS.1103352108</a> (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3726',
    'Roberts et&nbsp;al. (2022)',
    'Roberts, J. M., Neuman, J. A., Brown, S. S., Veres, P. R., Coggon, M. M., Stockwell, C. E., Warneke, C., Peischl, J., &#x0026; Robinson, M. A.: <span class="cmti-10">Furoyl peroxynitrate (fur-PAN), a product of VOC-NOx photochemistry</span> <span class="cmti-10">from biomass burning emissions: photochemical synthesis, calibration,</span> <span class="cmti-10">chemical characterization, and first atmospheric observations</span>, Environ. Sci. Atmos., 2, 1087–1100, <a href="https://doi.org/10.1039/D2EA00068G" >doi:10.1039/D2EA00068G</a> (2022). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3961',
    'Robinson et&nbsp;al. (1997)',
    'Robinson, G. N., Worsnop, D. R., Jayne, J. T., , Kolb, C. E., &#x0026; Davidovits, P.: <span class="cmti-10">Heterogeneous uptake of</span> ClONO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">and</span> N<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">5</span></sub> <span class="cmti-10">by sulfuric</span> <span class="cmti-10">acid solutions</span>, J. Geophys. Res., 102, 3583–3601, <a href="https://doi.org/10.1029/96JD03457" >doi:10.1029/96JD03457</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '727',
    'Rochester and Symonds (1973)',
    'Rochester, H. &#x0026; Symonds, J. R.: <span class="cmti-10">Thermodynamic studies of fluoroalcohols. Part 3. – The thermodynamics</span> <span class="cmti-10">of transfer of five fluoroalcohols from the gas-phase to aqueous solution</span>, J. Chem. Soc. Faraday Trans. 1, 69, 1577–1585, <a href="https://doi.org/10.1039/F19736901577" >doi:10.1039/F19736901577</a> (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'rodriguez2001',
    'Rodríguez-Sevilla et&nbsp;al. (2001)',
    'Rodríguez-Sevilla, J., Álvarez Diaz, M., Diaz Garcia, C., &#x0026; Limiñana de la Fe, G.: <span class="cmti-10">Thermodynamic equilibrium of</span> SO<sub><span class="cmr-7">2</span></sub><span class="cmti-10">-</span>H<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">system at low partial</span> <span class="cmti-10">pressures</span>, Afinidad, 492, 141–146 (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3536',
    'Rodríguez-Sevilla et&nbsp;al. (2002)',
    'Rodríguez-Sevilla, J., Álvarez, M., Limiñana, G., &#x0026; Díaz, M. C.: <span class="cmti-10">Dilute</span> SO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">absorption equilibria in aqueous</span> HCl <span class="cmti-10">and</span> NaCl <span class="cmti-10">solutions at</span> <span class="cmti-10">298.15K</span>, J. Chem. Eng. Data, 47, 1339–1345, <a href="https://doi.org/10.1021/JE015538E" >doi:10.1021/JE015538E</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3530',
    'Roduner and Bartels (1992)',
    'Roduner, E. &#x0026; Bartels, D. M.: <span class="cmti-10">Solvent and</span> <span class="cmti-10">isotope effects on addition of atomic hydrogen to benzene in</span> <span class="cmti-10">aqueous solution</span>, Ber. Bunsenges. Phys. Chem., 96, 1037–1042, <a href="https://doi.org/10.1002/BBPC.19920960813" >doi:10.1002/BBPC.19920960813</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '792',
    'Rohrschneider (1973)',
    'Rohrschneider, L.: <span class="cmti-10">Solvent characterization by</span> <span class="cmti-10">gas-liquid partition coefficients of selected solutes</span>, Anal. Chem., 45, 1241–1247, <a href="https://doi.org/10.1021/AC60329A023" >doi:10.1021/AC60329A023</a> (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3066',
    'Ross and Hudson (1957)',
    'Ross, S. &#x0026; Hudson, J. B.: <span class="cmti-10">Henry&#x2019;s law constants of</span> <span class="cmti-10">butadiene in aqueous solutions of a cationic surfactant</span>, J. Colloid Sci., 12, 523–525, <a href="https://doi.org/10.1016/0095-8522(57)90054-5" >doi:10.1016/0095-8522(57)90054-5</a> (1957). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2549',
    'Roth and Sullivan (1981)',
    'Roth, J. A. &#x0026; Sullivan, D. E.: <span class="cmti-10">Solubility of ozone</span> <span class="cmti-10">in water</span>, Ind. Eng. Chem. Fund., 20, 137–140 (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3944',
    'Roth (1897)',
    'Roth, W.: <span class="cmti-10">Ueber die Absorption des Stickoxyduls in w</span><span class="cmti-10">ässrigen</span> <span class="cmti-10">L</span><span class="cmti-10">ösungen verschieden dissociierter Stoffe</span>, Z. Phys. Chem., 24, 114–151, <a href="https://doi.org/10.1515/ZPCH-1897-2408" >doi:10.1515/ZPCH-1897-2408</a> (1897). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3810',
    'Roth (1929)',
    'Roth, W. A.: <span class="cmti-10">Zur Thermochemie</span> <span class="cmti-10">des Chlors und der unterchlorigen S</span><span class="cmti-10">äure</span>, Z. Phys. Chem., 145, 289–297, <a href="https://doi.org/10.1515/ZPCH-1929-14525" >doi:10.1515/ZPCH-1929-14525</a> (1929). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3949',
    'Roth (1942)',
    'Roth, W. A.: <span class="cmti-10">Notiz zur Thermochemie des Chlormonoxydes</span>, Z. Phys. Chem., A191, 248–250, <a href="https://doi.org/10.1515/ZPCH-1942-19117" >doi:10.1515/ZPCH-1942-19117</a> (1942). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3089',
    'Rubbiani (2013)',
    'Rubbiani, M.: <span class="cmti-10">CLH Report for Brodifacoum</span>, Tech. rep., European Chemicals Agency (ECHA), URL https://echa.europa.eu/documents/10162/71fd5c43-4db1-d6ac-bb74-a6359304c54a (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '585',
    'Rudich et&nbsp;al. (1996)',
    'Rudich, Y., Talukdar, R. K., Ravishankara, A. R., &#x0026; Fox, R. W.: <span class="cmti-10">Reactive</span> <span class="cmti-10">uptake of</span> NO<sub><span class="cmr-7">3</span></sub> <span class="cmti-10">on pure water and ionic solutions</span>, J. Geophys. Res., 101, 21 023–21 031, <a href="https://doi.org/10.1029/96JD01844" >doi:10.1029/96JD01844</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3784',
    'Ruetschi and Amlie (1966)',
    'Ruetschi, P. &#x0026; Amlie, R. F.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">hydrogen in potassium hydroxide and sulfuric acid. Salting-out and</span> <span class="cmti-10">hydration</span>, J. Phys. Chem., 70, 718–723, <a href="https://doi.org/10.1021/J100875A018" >doi:10.1021/J100875A018</a> (1966). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3572',
    'Ruiz-Bevia and Fernandez-Torres (2010)',
    'Ruiz-Bevia, F. &#x0026; Fernandez-Torres, M. J.: <span class="cmti-10">Determining the Henry&#x2019;s law constants</span> <span class="cmti-10">of THMs in seawater by means of purge-and-trap gas chromatography</span> <span class="cmti-10">(PT-GC): The influence of seawater as sample matrix</span>, Anal. Sci., 26, 723–726, <a href="https://doi.org/10.2116/ANALSCI.26.723" >doi:10.2116/ANALSCI.26.723</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3539',
    'Rumble (2021)',
    'Rumble, J. R.: CRC Handbook of Chemistry and Physics, 102nd Edition, CRC Press, Boca Raton, FL, URL https://hbcp.chemnetbase.com (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '731',
    'Russell et&nbsp;al. (1992)',
    'Russell, C. J., Dixon, S. L., &#x0026; Jurs, P. C.: <span class="cmti-10">Computer-assisted study of the relationship between</span> <span class="cmti-10">molecular structure and Henry&#x2019;s law constant</span>, Anal. Chem., 64, 1350–1355, <a href="https://doi.org/10.1021/AC00037A009" >doi:10.1021/AC00037A009</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2235',
    'Ryan et&nbsp;al. (1988)',
    'Ryan, J. A., Bell, R. M., Davidson, J. M., &#x0026; O&#x2019;Connor, G. A.: <span class="cmti-10">Plant uptake of</span> <span class="cmti-10">non-ionic organic chemicals from soils</span>, Chemosphere, 17, 2299–2323, <a href="https://doi.org/10.1016/0045-6535(88)90142-7" >doi:10.1016/0045-6535(88)90142-7</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2218',
    'Rytting et&nbsp;al. (1978)',
    'Rytting, J. H., Huston, L. P., &#x0026; Higuchi, T.: <span class="cmti-10">Thermodynamic group contributions for</span> <span class="cmti-10">hydroxyl, amino, and methylene groups</span>, J. Pharm. Sci., 69, 615–618, <a href="https://doi.org/10.1002/JPS.2600670510" >doi:10.1002/JPS.2600670510</a> (1978). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2935',
    'Ryu and Park (1999)',
    'Ryu, S.-A. &#x0026; Park, S.-J.: <span class="cmti-10">A rapid determination method</span> <span class="cmti-10">of the air/water partition coefficient and its application</span>, Fluid Phase Equilib., 161, 295–304, <a href="https://doi.org/10.1016/S0378-3812(99)00193-4" >doi:10.1016/S0378-3812(99)00193-4</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1141',
    'Sabljić and Güsten (1989)',
    'Sabljić, A. &#x0026; Güsten, H.: <span class="cmti-10">Predicting Henry&#x2019;s law</span> <span class="cmti-10">constants for polychlorinated biphenyls</span>, Chemosphere, 19, 1503–1511, <a href="https://doi.org/10.1016/0045-6535(89)90495-5" >doi:10.1016/0045-6535(89)90495-5</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2915',
    'Saçan et&nbsp;al. (2005)',
    'Saçan, M. T., Özkul, M., &#x0026; Erdem, S. S.: <span class="cmti-10">Physico-chemical properties</span> <span class="cmti-10">of PCDD/PCDFs and phthalate esters</span>, SAR QSAR Environ. Res., 16, 443–459, <a href="https://doi.org/10.1080/10659360500320602" >doi:10.1080/10659360500320602</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1142',
    'Sagebiel et&nbsp;al. (1992)',
    'Sagebiel, J. C., Seiber, J. N., &#x0026; Woodrow, J. E.: <span class="cmti-10">Comparison of headspace and gas-stripping</span> <span class="cmti-10">methods for determining the Henry&#x2019;s law constant (</span><span class="cmmi-10">H</span><span class="cmti-10">) for organic</span> <span class="cmti-10">compounds of low to intermediate </span><span class="cmmi-10">H</span>, Chemosphere, 25, 1763–1768, <a href="https://doi.org/10.1016/0045-6535(92)90017-L" >doi:10.1016/0045-6535(92)90017-L</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2216',
    'Sahsuvar et&nbsp;al. (2003)',
    'Sahsuvar, L., Helm, P. A., Jantunen, L. M., &#x0026; Bidleman, T. F.: <span class="cmti-10">Henry&#x2019;s law</span> <span class="cmti-10">constants for </span><span class="cmmi-10">α</span><span class="cmti-10">-, </span><span class="cmmi-10">β</span><span class="cmti-10">-, and </span><span class="cmmi-10">γ</span><span class="cmti-10">-hexachlorocyclohexanes (HCHs) as a function</span> <span class="cmti-10">of temperature and revised estimates of gas exchange in Arctic regions</span>, Atmos. Environ., 37, 983–992, <a href="https://doi.org/10.1016/S1352-2310(02)00936-6" >doi:10.1016/S1352-2310(02)00936-6</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3710',
    'Saidman et&nbsp;al. (1966)',
    'Saidman, L. J., Eger, E. I., Munson, E. S., &#x0026; Severinghaus, J. W.: <span class="cmti-10">A method for</span> <span class="cmti-10">determining solubility of anesthetics utilizing the Scholander apparatus</span>, Anesthesiology, 27, 180–184, <a href="https://doi.org/10.1097/00000542-196603000-00011" >doi:10.1097/00000542-196603000-00011</a> (1966). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2312',
    'Saiz-Lopez et&nbsp;al. (2008)',
    'Saiz-Lopez, A., Plane, J. M. C., Mahajan, A. S., Anderson, P. S., Bauguitte, S. J.-B., Jones, A. E., Roscoe, H. K., Salmon, R. A., Bloss, W. J., Lee, J. D., &#x0026; Heard, D. E.: <span class="cmti-10">On the vertical distribution of</span> <span class="cmti-10">boundary layer halogens over coastal Antarctica: implications for</span> O<sub><span class="cmr-7">3</span></sub><span class="cmti-10">,</span> HO<sub><span class="cmr-7">x</span></sub><span class="cmti-10">,</span> NO<sub><span class="cmr-7">x</span></sub> <span class="cmti-10">and the</span> Hg <span class="cmti-10">lifetime</span>, Atmos. Chem. Phys., 8, 887–900, <a href="https://doi.org/10.5194/ACP-8-887-2008" >doi:10.5194/ACP-8-887-2008</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3760',
    'Saiz-Lopez et&nbsp;al. (2014)',
    'Saiz-Lopez, A., Fernandez, R. P., Ordóñez, C., Kinnison, D. E., Gómez Martín, J. C., Lamarque, J.-F., &#x0026; Tilmes, S.: <span class="cmti-10">Iodine chemistry in</span> <span class="cmti-10">the troposphere and its effect on ozone</span>, Atmos. Chem. Phys., 14, 13 119–13 143, <a href="https://doi.org/10.5194/ACP-14-13119-2014" >doi:10.5194/ACP-14-13119-2014</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '958',
    'Sander (1999)',
    'Sander, R.: <span class="cmti-10">Modeling atmospheric chemistry: Interactions</span> <span class="cmti-10">between gas-phase species and liquid cloud/aerosol particles</span>, Surv. Geophys., 20, 1–31, <a href="https://doi.org/10.1023/A:1006501706704" >doi:10.1023/A:1006501706704</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2000',
    'Sander (2015)',
    'Sander, R.: <span class="cmti-10">Compilation of Henry&#x2019;s law constants (version</span> <span class="cmti-10">4.0) for water as solvent</span>, Atmos. Chem. Phys., 15, 4399–4981, <a href="https://doi.org/10.5194/ACP-15-4399-2015" >doi:10.5194/ACP-15-4399-2015</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '271',
    'Sander and Crutzen (1996)',
    'Sander, R. &#x0026; Crutzen, P. J.: <span class="cmti-10">Model study</span> <span class="cmti-10">indicating halogen activation and ozone destruction in polluted air</span> <span class="cmti-10">masses transported to the sea</span>, J. Geophys. Res., 101, 9121–9138, <a href="https://doi.org/10.1029/95JD03793" >doi:10.1029/95JD03793</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3509',
    'Sander et&nbsp;al. (2022)',
    'Sander, R., Acree Jr., W. E., De Visscher, A., Schwartz, S. E., &#x0026; Wallington, T. J.: <span class="cmti-10">Henry&#x2019;s law constants (IUPAC Recommendations</span> <span class="cmti-10">2021)</span>, Pure Appl. Chem., 94, 71–85, <a href="https://doi.org/10.1515/PAC-2020-0302" >doi:10.1515/PAC-2020-0302</a> (2022). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1945',
    'Sander et&nbsp;al. (2006)',
    'Sander, S. P., Friedl, R. R., Golden, D. M., Kurylo, M. J., Moortgat, G. K., Keller-Rudek, H., Wine, P. H., Ravishankara, A. R., Kolb, C. E., Molina, M. J., Finlayson-Pitts, B. J., Huie, R. E., &#x0026; Orkin, V. L.: <span class="cmti-10">Chemical Kinetics and Photochemical</span> <span class="cmti-10">Data for Use in Atmospheric Studies, Evaluation Number 15</span>, JPL Publication 06-2, Jet Propulsion Laboratory, Pasadena, CA, URL https://jpldataeval.jpl.nasa.gov (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2626',
    'Sander et&nbsp;al. (2011)',
    'Sander, S. P., Abbatt, J., Barker, J. R., Burkholder, J. B., Friedl, R. R., Golden, D. M., Huie, R. E., Kolb, C. E., Kurylo, M. J., Moortgat, G. K., Orkin, V. L., &#x0026; Wine, P. H.: <span class="cmti-10">Chemical Kinetics and</span> <span class="cmti-10">Photochemical Data for Use in Atmospheric Studies, Evaluation No.</span> <span class="cmti-10">17</span>, JPL Publication 10-6, Jet Propulsion Laboratory, Pasadena, URL https://jpldataeval.jpl.nasa.gov (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2455',
    'Sanders and Seiber (1983)',
    'Sanders, P. F. &#x0026; Seiber, J. N.: <span class="cmti-10">A chamber for</span> <span class="cmti-10">measuring volatilization of pesticides from model soil and water disposal</span> <span class="cmti-10">systems</span>, Chemosphere, 12, 999–1012, <a href="https://doi.org/10.1016/0045-6535(83)90252-7" >doi:10.1016/0045-6535(83)90252-7</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2280',
    'Sanemasa (1975)',
    'Sanemasa, I.: <span class="cmti-10">The solubility of elemental mercury vapor in</span> <span class="cmti-10">water</span>, Bull. Chem. Soc. Jpn., 48, 1795–1798, <a href="https://doi.org/10.1246/BCSJ.48.1795" >doi:10.1246/BCSJ.48.1795</a> (1975). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2984',
    'Sanemasa et&nbsp;al. (1981)',
    'Sanemasa, I., Akari, M., Deguchi, T., &#x0026; Nagai, H.: <span class="cmti-10">Solubilities of benzene and the</span> <span class="cmti-10">alkylbenzenes in water – method for obtaining aqueous solutions saturated</span> <span class="cmti-10">with vapours in equilibrium with organic liquids</span>, Chem. Lett., 10, 225–228, <a href="https://doi.org/10.1246/CL.1981.225" >doi:10.1246/CL.1981.225</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2453',
    'Sanemasa et&nbsp;al. (1982)',
    'Sanemasa, I., Araki, M., Deguchi, T., &#x0026; Nagai, H.: <span class="cmti-10">Solubility measurements of benzene</span> <span class="cmti-10">and the alkylbenzenes in water by making use of solute vapor</span>, Bull. Chem. Soc. Jpn., 55, 1054–1062, <a href="https://doi.org/10.1246/BCSJ.55.1054" >doi:10.1246/BCSJ.55.1054</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '644',
    'Santl et&nbsp;al. (1994)',
    'Santl, H., Brandsch, R., &#x0026; Gruber, L.: <span class="cmti-10">Experimental determination of Henry&#x2019;s law constant (HLC)</span> <span class="cmti-10">for some lower chlorinated dibenzodioxins</span>, Chemosphere, 29, 2209–2214, <a href="https://doi.org/10.1016/0045-6535(94)90388-3" >doi:10.1016/0045-6535(94)90388-3</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3214',
    'Sanz and Gmehling (2005)',
    'Sanz, M. T. &#x0026; Gmehling, J.: <span class="cmti-10">Isothermal</span> <span class="cmti-10">vapor-liquid equilibrium, excess enthalpy data, and activity coefficients at</span> <span class="cmti-10">infinite dilution for the binary system water + methyl lactate</span>, J. Chem. Eng. Data, 50, 85–88, <a href="https://doi.org/10.1021/JE049824C" >doi:10.1021/JE049824C</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2554',
    'Sarraute et&nbsp;al. (2004)',
    'Sarraute, S., Delepine, H., Costa Gomes, M. F., &#x0026; Majer, V.: <span class="cmti-10">Aqueous</span> <span class="cmti-10">solubility, Henry&#x2019;s law constants and air/water partition coefficients of</span> <span class="cmmi-10">n</span><span class="cmti-10">-octane and two halogenated octanes</span>, Chemosphere, 57, 1543–1551, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2004.07.046" >doi:10.1016/J.CHEMOSPHERE.2004.07.046</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2122',
    'Sarraute et&nbsp;al. (2006)',
    'Sarraute, S., Mokbel, I., Costa Gomes, M. F., Majer, V., Delepine, H., &#x0026; Jose, J.: <span class="cmti-10">Vapour pressures, aqueous solubility,</span> <span class="cmti-10">Henry&#x2019;s law constants and air/water partition coefficients of</span> <span class="cmti-10">1,8-dichlorooctane and 1,8-dibromooctane</span>, Chemosphere, 64, 1829–1836, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2006.01.057" >doi:10.1016/J.CHEMOSPHERE.2006.01.057</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2917',
    'Sato and Nakajima (1979a)',
    'Sato, A. &#x0026; Nakajima, T.: <span class="cmti-10">Partition coefficients</span> <span class="cmti-10">of some aromatic hydrocarbons and ketones in water, blood and oil</span>, Br. J. Ind. Med., 36, 231–234, <a href="https://doi.org/10.1136/OEM.36.3.231" >doi:10.1136/OEM.36.3.231</a> (1979a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2918',
    'Sato and Nakajima (1979b)',
    'Sato, A. &#x0026; Nakajima, T.: <span class="cmti-10">A structure-activity</span> <span class="cmti-10">relationship of some chlorinated hydrocarbons</span>, Arch. Environ. Health, 34, 69–75, <a href="https://doi.org/10.1080/00039896.1979.10667371" >doi:10.1080/00039896.1979.10667371</a> (1979b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1200',
    'Sauer (1997)',
    'Sauer, F.: Bestimmung von H<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">2</span></sub> und organischen Peroxiden in Labor- und Feldmessungen mittels Umkehrphasen-Hochdruck-Flüssigkeitschromatographie und enzymatischer Nachsäulenderivatisierung, Ph.D. thesis, Johannes Gutenberg-Universität, Mainz, Germany (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3025',
    'Savary et&nbsp;al. (2014)',
    'Savary, G., Hucher, N., Petibon, O., &#x0026; Grisel, M.: <span class="cmti-10">Study of interactions between aroma</span> <span class="cmti-10">compounds and acacia gum using headspace measurements</span>, Food Hydrocolloids, 37, 1–6, <a href="https://doi.org/10.1016/J.FOODHYD.2013.10.026" >doi:10.1016/J.FOODHYD.2013.10.026</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '588',
    'Saxena and Hildemann (1996)',
    'Saxena, P. &#x0026; Hildemann, L. M.: <span class="cmti-10">Water-soluble organics in atmospheric particles: A critical review</span> <span class="cmti-10">of the literature and application of thermodynamics to identify candidate</span> <span class="cmti-10">compounds</span>, J. Atmos. Chem., 24, 57–109, <a href="https://doi.org/10.1007/BF00053823" >doi:10.1007/BF00053823</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2491',
    'Saylor et&nbsp;al. (1938)',
    'Saylor, J. H., Stuckey, J. M., &#x0026; Gross, P. M.: <span class="cmti-10">Solubility studies. V. the validity of Henry&#x2019;s law for</span> <span class="cmti-10">the calculation of vapor solubilities</span>, J. Am. Chem. Soc., 60, 373–376, <a href="https://doi.org/10.1021/JA01269A041" >doi:10.1021/JA01269A041</a> (1938). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '735',
    'Schäfer and Lax (1962)',
    'Schäfer, K. &#x0026; Lax, E., eds.: Landolt-Börnstein. Zahlenwerte und Funktionen aus Physik, Chemie, Astronomie, Geophysik und Technik. II. Band. Eigenschaften der Materie in ihren Aggregatzuständen. 2. Teil. Gleichgewichte ausser Schmelzgleichgewichten. Bandteil b. Lösungsgleichgewichte I, Springer Verlag, Berlin-Göttingen-Heidelberg, ISBN 9783540028680 (1962). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '859',
    'Schaffer and Daubert (1969)',
    'Schaffer, D. L. &#x0026; Daubert, T. E.: <span class="cmti-10">Gas-liquid chromatographic determination of</span> <span class="cmti-10">solution properties of oxygenated compounds in water</span>, Anal. Chem., 41, 1585–1589, <a href="https://doi.org/10.1021/AC60281A016" >doi:10.1021/AC60281A016</a> (1969). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3662',
    'Scharlin (1996)',
    'Scharlin, P.: IUPAC Solubility Data Series, Volume 62, Carbon Dioxide in Water and Aqueous Electrolyte Solutions, Oxford University Press (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2485',
    'Scharlin and Battino (1994)',
    'Scharlin, P. &#x0026; Battino, R.: <span class="cmti-10">Solubility of</span> CCl<sub><span class="cmr-7">2</span></sub>F<sub><span class="cmr-7">2</span></sub><span class="cmti-10">,</span> CClF<sub><span class="cmr-7">3</span></sub><span class="cmti-10">,</span> CF<sub><span class="cmr-7">4</span></sub> <span class="cmti-10">and c-</span>C<sub><span class="cmr-7">4</span></sub>F<sub><span class="cmr-7">8</span></sub> <span class="cmti-10">in</span> H<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">and</span> D<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">at 288 to 318 </span>K <span class="cmti-10">and</span> <span class="cmti-10">101.325kPa. Thermodynamics of transfer of gases from</span> H<sub><span class="cmr-7">2</span></sub>O <span class="cmti-10">to</span> D<sub><span class="cmr-7">2</span></sub>O, Fluid Phase Equilib., 95, 137–147, <a href="https://doi.org/10.1016/0378-3812(94)80066-9" >doi:10.1016/0378-3812(94)80066-9</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3569',
    'Scharlin and Battino (1995)',
    'Scharlin, P. &#x0026; Battino, R.: <span class="cmti-10">Solubility of</span> CCl<sub><span class="cmr-7">2</span></sub>F<sub><span class="cmr-7">2</span></sub><span class="cmti-10">,</span> CClF<sub><span class="cmr-7">3</span></sub><span class="cmti-10">,</span> CF<sub><span class="cmr-7">4</span></sub><span class="cmti-10">, and</span> CH<sub><span class="cmr-7">4</span></sub> <span class="cmti-10">in water</span> <span class="cmti-10">and seawater at 288.15-303.15K and 101.325kPa</span>, J. Chem. Eng. Data, 40, 167–169, <a href="https://doi.org/10.1021/JE00017A036" >doi:10.1021/JE00017A036</a> (1995). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1060',
    'Scheer et&nbsp;al. (1997)',
    'Scheer, V., Frenzel, A., Behnke, W., Zetzsch, C., Magi, L., George, C., &#x0026; Mirabel, P.: <span class="cmti-10">Uptake of nitrosyl chloride (</span>NOCl<span class="cmti-10">) by aqueous solutions</span>, J. Phys. Chem. A, 101, 9359–9366, <a href="https://doi.org/10.1021/JP972143M" >doi:10.1021/JP972143M</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3777',
    'Schmidt (1979)',
    'Schmidt, U.: <span class="cmti-10">The solubility of carbon monoxide and hydrogen in water</span> <span class="cmti-10">and sea-water at partial pressures of about </span>10<sup><span class="cmsy-7">−</span><span class="cmr-7">5</span></sup> <span class="cmti-10">atmospheres</span>, Tellus, 31, 68–74, <a href="https://doi.org/10.1111/J.2153-3490.1979.TB00883.X" >doi:10.1111/J.2153-3490.1979.TB00883.X</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'schoen23',
    'Schoen (1923)',
    'Schoen, R.: <span class="cmti-10">Zur Kenntnis der Acetylenwirkung. II.</span> <span class="cmti-10">Mitteilung. Die L</span><span class="cmti-10">öslichkeit von Acetylen in Wasser und Blut</span>, Z. Physiol. Chem., 127, 243–259, <a href="https://doi.org/10.1515/BCHM2.1923.127.4-6.243" >doi:10.1515/BCHM2.1923.127.4-6.243</a> (1923). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1331',
    'Schoene and Steinhanses (1985)',
    'Schoene, K. &#x0026; Steinhanses, J.: <span class="cmti-10">Determination of Henry&#x2019;s law constant by automated</span> <span class="cmti-10">head space-gas chromatography</span>, Fresenius J. Anal. Chem., 321, 538–543, <a href="https://doi.org/10.1007/BF00464360" >doi:10.1007/BF00464360</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3933',
    'Schoenfeld (1855)',
    'Schoenfeld, F.: <span class="cmti-10">Ueber den Absorptionsco</span><span class="cmti-10">ëfficienten der</span> <span class="cmti-10">schwefligen S</span><span class="cmti-10">äure, des Chlors und des Schwefelwasserstoffs</span>, Liebigs Ann. Chem., 95, 1–23, <a href="https://doi.org/10.1002/JLAC.18550950102" >doi:10.1002/JLAC.18550950102</a> (1855). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3597',
    'Schöne (1873)',
    'Schöne, E.: <span class="cmti-10">Ueber das Verhalten von Ozon und Wasser zu einander</span>, Ber. Dtsch. Chem. Ges., 6, 1224–1229, <a href="https://doi.org/10.1002/CBER.187300602111" >doi:10.1002/CBER.187300602111</a> (1873). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3665',
    'Schröder et&nbsp;al. (2010)',
    'Schröder, B., Santos, L. M. N. B. F., Rocha, M. A. A., Oliveira, M. B., Marrucho, I. M., &#x0026; Coutinho, J. A. P.: <span class="cmti-10">Prediction of environmental</span> <span class="cmti-10">parameters of polycyclic aromatic hydrocarbons with COSMO-RS</span>, Chemosphere, 79, 821–829, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2010.02.059" >doi:10.1016/J.CHEMOSPHERE.2010.02.059</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3666',
    'Schröder et&nbsp;al. (2011)',
    'Schröder, B., Freire, M. G., Varanda, F. R., Marrucho, I. M., Santos, L. M. N. B. F., &#x0026; Coutinho, J. A. P.: <span class="cmti-10">Aqueous solubility, effects of salts</span> <span class="cmti-10">on aqueous solubility, and partitioning behavior of hexafluorobenzene:</span> <span class="cmti-10">Experimental results and COSMO-RS predictions</span>, Chemosphere, 84, 415–422, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2011.03.055" >doi:10.1016/J.CHEMOSPHERE.2011.03.055</a> (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3655',
    'Schröder et&nbsp;al. (2013)',
    'Schröder, B., Coutinho, J., &#x0026; Santos, L. M. N. B. F.: <span class="cmti-10">Predicting</span> <span class="cmti-10">physico-chemical properties of alkylated naphthalenes with COSMO-RS</span>, Polycyclic Aromat. Compd., 33, 1–19, <a href="https://doi.org/10.1080/10406638.2012.683231" >doi:10.1080/10406638.2012.683231</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2171',
    'Schroeder and Munthe (1998)',
    'Schroeder, W. H. &#x0026; Munthe, J.: <span class="cmti-10">Atmospheric mercury – An overview</span>, Atmos. Environ., 32, 809–822, <a href="https://doi.org/10.1016/S1352-2310(97)00293-8" >doi:10.1016/S1352-2310(97)00293-8</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2947',
    'Schroy et&nbsp;al. (1985)',
    'Schroy, J. M., Hileman, F. D., &#x0026; Cheng, S. C.: <span class="cmti-10">Physical/chemical properties of 2,3,7,8-TCDD</span>, Chemosphere, 14, 877–880, <a href="https://doi.org/10.1016/0045-6535(85)90207-3" >doi:10.1016/0045-6535(85)90207-3</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2823',
    'Schuhfried et&nbsp;al. (2011)',
    'Schuhfried, E., Biasioli, F., Aprea, E., Cappellin, L., Soukoulis, C., Ferrigno, A., Märk, T. D., &#x0026; Gasperi, F.: <span class="cmti-10">PTR-MS measurements</span> <span class="cmti-10">and analysis of models for the calculation of Henry&#x2019;s law constants</span> <span class="cmti-10">of monosulfides and disulfides</span>, Chemosphere, 83, 311–317, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2010.12.051" >doi:10.1016/J.CHEMOSPHERE.2010.12.051</a> (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3423',
    'Schuhfried et&nbsp;al. (2015)',
    'Schuhfried, E., Aprea, E., Märk, T. D., &#x0026; Biasioli, F.: <span class="cmti-10">Refined</span> <span class="cmti-10">measurements of Henry&#x2019;s law constant of terpenes with inert gas</span> <span class="cmti-10">stripping coupled with PTR-MS</span>, Water Air Soil Pollut., 226, 120, <a href="https://doi.org/10.1007/S11270-015-2337-2" >doi:10.1007/S11270-015-2337-2</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3767',
    'Schumpe (1993)',
    'Schumpe, A.: <span class="cmti-10">The estimation of gas solubilities in salt solutions</span>, Chem. Eng. Sci., 48, 153–158, <a href="https://doi.org/10.1016/0009-2509(93)80291-W" >doi:10.1016/0009-2509(93)80291-W</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '672',
    'Schurath et&nbsp;al. (1996)',
    'Schurath, U., Bongartz, A., Kames, J., Wunderlich, C., &#x0026; Carstens, T.: <span class="cmti-10">Chapter 6.4: Laboratory determination of physico-chemical rate</span> <span class="cmti-10">parameters pertinent to mass transfer into cloud and fog droplets</span>, in: Heterogeneous and Liquid-Phase Processes, edited by Warneck, P., pp. 182–189, Springer Verlag, Berlin, <a href="https://doi.org/10.1007/978-3-642-61445-3_6" >doi:10.1007/978-3-642-61445-3_6</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2165',
    'Schüürmann (2000)',
    'Schüürmann, G.: <span class="cmti-10">Prediction of Henry&#x2019;s law constant of</span> <span class="cmti-10">benzene derivatives using quantum chemical continuum-solvation models</span>, J. Comput. Chem., 21, 17–34, <a href="https://doi.org/10.1002/(SICI)1096-987X(20000115)21:1<17::AID-JCC3>3.0.CO;2-5" >doi:10.1002/(SICI)1096-987X(20000115)21:1<17::AID-JCC3>3.0.CO;2-5</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3681',
    'Schwardt et&nbsp;al. (2021)',
    'Schwardt, A., Dahmke, A., &#x0026; Köber, R.: <span class="cmti-10">Henry&#x2019;s law constants of volatile organic</span> <span class="cmti-10">compounds between 0 and 95</span><span class="cmsy-7">∘</span>C <span class="cmti-10">– Data compilation and complementation</span> <span class="cmti-10">in</span> <span class="cmti-10">context of urban temperature increases of the subsurface</span>, Chemosphere, 272, 129 858, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2021.129858" >doi:10.1016/J.CHEMOSPHERE.2021.129858</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3722',
    'Schwardt et&nbsp;al. (2022)',
    'Schwardt, A., Dahmke, A., &#x0026; Köber, R.: <span class="cmti-10">Corrigendum to “Henry&#x2019;s law constants</span> <span class="cmti-10">of volatile organic compounds between 0 and 95</span><span class="cmsy-7">∘</span>C <span class="cmti-10">– Data compilation</span> <span class="cmti-10">and complementation in context of urban temperature increases of the</span> <span class="cmti-10">subsurface” [Chem. 272 (2021) 129858]</span>, Chemosphere, 298, 134 683, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2022.134683" >doi:10.1016/J.CHEMOSPHERE.2022.134683</a> (2022). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '209',
    'Schwartz (1984)',
    'Schwartz, S. E.: <span class="cmti-10">Gas- and aqueous-phase chemistry of</span> HO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">in liquid water clouds</span>, J. Geophys. Res., 89, 11 589–11 598, <a href="https://doi.org/10.1029/JD089ID07P11589" >doi:10.1029/JD089ID07P11589</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '19',
    'Schwartz (1986)',
    'Schwartz, S. E.: <span class="cmti-10">Mass-transport considerations pertinent</span> <span class="cmti-10">to aqueous phase reactions of gases in liquid-water clouds</span>, in: Chemistry of Multiphase Atmospheric Systems, NATO ASI Series, Vol. G6, edited by Jaeschke, W., pp. 415–471, Springer Verlag, Berlin, <a href="https://doi.org/10.1007/978-3-642-70627-1_16" >doi:10.1007/978-3-642-70627-1_16</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '449',
    'Schwartz and White (1981)',
    'Schwartz, S. E. &#x0026; White, W. H.: <span class="cmti-10">Solubility</span> <span class="cmti-10">equilibria of the nitrogen oxides and oxyacids in dilute aqueous solution</span>, in: Advances in Environmental Science and Engineering, edited by Pfafflin, J. R. &#x0026; Ziegler, E. N., vol. 4, pp. 1–45, Gordon and Breach Science Publishers, NY (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2458',
    'Schwarz and Wasik (1977)',
    'Schwarz, F. P. &#x0026; Wasik, S. P.: <span class="cmti-10">A fluorescence</span> <span class="cmti-10">method for the measurement of the partition coefficients of naphthalene,</span> <span class="cmti-10">1-methylnaphthalene, and 1-ethylnaphthalene in water</span>, J. Chem. Eng. Data, 22, 270–273, <a href="https://doi.org/10.1021/JE60074A009" >doi:10.1021/JE60074A009</a> (1977). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '762',
    'Schwarz and Bielski (1986)',
    'Schwarz, H. A. &#x0026; Bielski, B. H. J.: <span class="cmti-10">Reactions of</span> HO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">and</span> O<sub><span class="cmr-7">2</span></sub><sup><span class="cmsy-7">−</span></sup> <span class="cmti-10">with iodine and bromine and the</span> I<sub><span class="cmr-7">2</span></sub><sup><span class="cmsy-7">−</span></sup> <span class="cmti-10">and </span>I <span class="cmti-10">atom reduction</span> <span class="cmti-10">potentials</span>, J. Phys. Chem., 90, 1445–1448, <a href="https://doi.org/10.1021/J100398A045" >doi:10.1021/J100398A045</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '761',
    'Schwarz and Dodson (1984)',
    'Schwarz, H. A. &#x0026; Dodson, R. W.: <span class="cmti-10">Equilibrium</span> <span class="cmti-10">between hydroxyl radicals and thallium(II) and the oxidation potential</span> <span class="cmti-10">of</span> OH(aq), J. Phys. Chem., 88, 3643–3647, <a href="https://doi.org/10.1021/J150660A053" >doi:10.1021/J150660A053</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '941',
    'Schwarzenbach et&nbsp;al. (1988)',
    'Schwarzenbach, R. P., Stierli, R., Folsom, B. R., &#x0026; Zeyer, J.: <span class="cmti-10">Compound properties relevant for</span> <span class="cmti-10">assessing the environmental partitioning of nitrophenols</span>, Environ. Sci. Technol., 22, 83–92, <a href="https://doi.org/10.1021/ES00166A009" >doi:10.1021/ES00166A009</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3807',
    'Secoy and Cady (1941)',
    'Secoy, C. H. &#x0026; Cady, G. H.: <span class="cmti-10">The effect of</span> <span class="cmti-10">temperature and pressure on the solubility of chlorine monoxide in water</span>, J. Am. Chem. Soc., 63, 2504–2508, <a href="https://doi.org/10.1021/JA01854A055" >doi:10.1021/JA01854A055</a> (1941). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '523',
    'Seinfeld (1986)',
    'Seinfeld, J. H.: Atmospheric Chemistry and Physics of Air Pollution, Wiley-Interscience Publication, NY, ISBN 0471828572 (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '935',
    'Seinfeld and Pandis (1997)',
    'Seinfeld, J. H. &#x0026; Pandis, S. N.: Atmospheric Chemistry and Physics, John Wiley &#x0026; Sons, Inc., ISBN 0471178160 (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3612',
    'Serra and Palavra (2003)',
    'Serra, M. C. C. &#x0026; Palavra, A. M. F.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">1-butene in water and in a medium for cultivation of a bacterial strain</span>, J. Solution Chem., 32, 527–534, <a href="https://doi.org/10.1023/A:1025313916280" >doi:10.1023/A:1025313916280</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '520',
    'Servant et&nbsp;al. (1991)',
    'Servant, J., Kouadio, G., Cros, B., &#x0026; Delmas, R.: <span class="cmti-10">Carboxylic monoacids</span> <span class="cmti-10">in the air of Mayombe forest (Congo): Role of the forest as a source or</span> <span class="cmti-10">sink</span>, J. Atmos. Chem., 12, 367–380, <a href="https://doi.org/10.1007/BF00114774" >doi:10.1007/BF00114774</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3102',
    'Setschenow (1889)',
    'Setschenow, J.: <span class="cmti-10">Über die Konstitution der Salzl</span><span class="cmti-10">ösungen</span> <span class="cmti-10">auf Grund ihres Verhaltens zu Kohlens</span><span class="cmti-10">äure</span>, Z. Phys. Chem., 4, 117–125, <a href="https://doi.org/10.1515/ZPCH-1889-0409" >doi:10.1515/ZPCH-1889-0409</a> (1889). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2572',
    'Severit (1997)',
    'Severit, P.: <span class="cmti-10">Experimentelle Untersuchung der Desorption</span> <span class="cmti-10">von Quecksilber und Quecksilberverbindungen aus w</span><span class="cmti-10">ässrigen L</span><span class="cmti-10">ösungen</span>, Diplomarbeit, Universität Köln, Germany (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2829',
    'Seyfioglu and Odabasi (2007)',
    'Seyfioglu, R. &#x0026; Odabasi, M.: <span class="cmti-10">Determination</span> <span class="cmti-10">of Henry&#x2019;s law constant of formaldehyde as a function of temperature:</span> <span class="cmti-10">Application to air–water exchange in Tahtali lake in Izmir, Turkey</span>, Environ. Monit. Assess., 128, 343–349, <a href="https://doi.org/10.1007/S10661-006-9317-3" >doi:10.1007/S10661-006-9317-3</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1930',
    'Sheikheldin et&nbsp;al. (2001)',
    'Sheikheldin, S. Y., Cardwell, T. J., Cattrall, R. W., Luque de Castro, M. D., &#x0026; Kolev, S. D.: <span class="cmti-10">Determination of Henry&#x2019;s law constants of</span> <span class="cmti-10">phenols by pervaporation-flow injection analysis</span>, Environ. Sci. Technol., 35, 178–181, <a href="https://doi.org/10.1021/ES001406E" >doi:10.1021/ES001406E</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2948',
    'Shen and Wania (2005)',
    'Shen, L. &#x0026; Wania, F.: <span class="cmti-10">Compilation, evaluation, and</span> <span class="cmti-10">selection of physical-chemical property data for organochlorine pesticides</span>, J. Chem. Eng. Data, 50, 742–768, <a href="https://doi.org/10.1021/JE049693F" >doi:10.1021/JE049693F</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3083',
    'Shen (1982)',
    'Shen, T. T.: <span class="cmti-10">Estimation of organic compound emissions</span> <span class="cmti-10">from waste lagoons</span>, J. Air Pollut. Control Assoc., 32, 79–82, <a href="https://doi.org/10.1080/00022470.1982.10465374" >doi:10.1080/00022470.1982.10465374</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '548',
    'Shepson et&nbsp;al. (1996)',
    'Shepson, P. B., Mackay, E., &#x0026; Muthuramu, K.: <span class="cmti-10">Henry&#x2019;s law constants</span> <span class="cmti-10">and removal processes for several atmospheric </span><span class="cmmi-10">β</span><span class="cmti-10">-hydroxy alkyl nitrates</span>, Environ. Sci. Technol., 30, 3618–3623, <a href="https://doi.org/10.1021/ES960538Y" >doi:10.1021/ES960538Y</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3679',
    'Sherblom et&nbsp;al. (1992)',
    'Sherblom, P. M., Gschwend, P. M., &#x0026; Eganhouse, R. P.: <span class="cmti-10">Aqueous</span> <span class="cmti-10">solubilities, vapor pressures, and 1-octanol-water partition coefficients</span> <span class="cmti-10">for</span> C<sub><span class="cmr-7">9</span></sub><span class="cmti-10">–</span>C<sub><span class="cmr-7">14</span></sub> <span class="cmti-10">linear alkylbenzenes</span>, J. Chem. Eng. Data, 37, 394–399, <a href="https://doi.org/10.1021/JE00008A005" >doi:10.1021/JE00008A005</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3603',
    'Sherwood et&nbsp;al. (1991)',
    'Sherwood, J. E., Stagnitti, F., Kokkinn, M. J., &#x0026; Williams, W. D.: <span class="cmti-10">Dissolved oxygen concentrations in hypersaline waters</span>, Limnol. Oceanogr., 36, 235–250, <a href="https://doi.org/10.4319/LO.1991.36.2.0235" >doi:10.4319/LO.1991.36.2.0235</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3003',
    'Shi et&nbsp;al. (1999)',
    'Shi, Q., Davidovits, P., Jayne, J. T., Worsnop, D. R., &#x0026; Kolb, C. E.: <span class="cmti-10">Uptake of</span> <span class="cmti-10">gas-phase ammonia. 1. Uptake by aqueous surfaces as a function of pH</span>, J. Phys. Chem. A, 103, 8812–8823, <a href="https://doi.org/10.1021/JP991696P" >doi:10.1021/JP991696P</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2897',
    'Shimotori and Arnold (2003)',
    'Shimotori, T. &#x0026; Arnold, W. A.: <span class="cmti-10">Measurement</span> <span class="cmti-10">and estimation of Henry&#x2019;s law constants of chlorinated ethylenes</span> <span class="cmti-10">in aqueous surfactant solutions</span>, J. Chem. Eng. Data, 48, 253–261, <a href="https://doi.org/10.1021/JE025553Z" >doi:10.1021/JE025553Z</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2338',
    'Shiu and Ma (2000)',
    'Shiu, W. Y. &#x0026; Ma, K.-C.: <span class="cmti-10">Temperature dependence</span> <span class="cmti-10">of physical-chemical properties of selected chemicals of environmental</span> <span class="cmti-10">interest. I. mononuclear and polynuclear aromatic hydrocarbons</span>, J. Phys. Chem. Ref. Data, 29, 41–130, <a href="https://doi.org/10.1063/1.556055" >doi:10.1063/1.556055</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2302',
    'Shiu and Mackay (1986)',
    'Shiu, W. Y. &#x0026; Mackay, D.: <span class="cmti-10">A critical review</span> <span class="cmti-10">of aqueous solubilities, vapor pressures, Henry&#x2019;s law constants, and</span> <span class="cmti-10">octanol-water partition coefficients of the polychlorinated biphenyls</span>, J. Phys. Chem. Ref. Data, 15, 911–929, <a href="https://doi.org/10.1063/1.555755" >doi:10.1063/1.555755</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2339',
    'Shiu and Mackay (1997)',
    'Shiu, W.-Y. &#x0026; Mackay, D.: <span class="cmti-10">Henry&#x2019;s law constants</span> <span class="cmti-10">of selected aromatic hydrocarbons, alcohols, and ketones</span>, J. Chem. Eng. Data, 42, 27–30, <a href="https://doi.org/10.1021/JE960218U" >doi:10.1021/JE960218U</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '854',
    'Shiu et&nbsp;al. (1988)',
    'Shiu, W. Y., Doucette, W., Gobas, F. A. P. C., Andren, A., &#x0026; Mackay, D.: <span class="cmti-10">Physical-chemical properties of chlorinated dibenzo-p-dioxins</span>, Environ. Sci. Technol., 22, 651–658, <a href="https://doi.org/10.1021/ES00171A006" >doi:10.1021/ES00171A006</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '638',
    'Shiu et&nbsp;al. (1994)',
    'Shiu, W.-Y., Ma, K.-C., Varhaníčková, D., &#x0026; Mackay, D.: <span class="cmti-10">Chlorophenols and alkylphenols:</span> <span class="cmti-10">A review and correlation of environmentally relevant properties and</span> <span class="cmti-10">fate in an evaluative environment</span>, Chemosphere, 29, 1155–1224, <a href="https://doi.org/10.1016/0045-6535(94)90252-6" >doi:10.1016/0045-6535(94)90252-6</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2285',
    'Shon et&nbsp;al. (2005)',
    'Shon, Z.-H., Kim, K.-H., Kim, M.-Y., &#x0026; Lee, M.: <span class="cmti-10">Modeling</span> <span class="cmti-10">study of reactive gaseous mercury in the urban air</span>, Atmos. Environ., 39, 749–761, <a href="https://doi.org/10.1016/J.ATMOSENV.2004.09.071" >doi:10.1016/J.ATMOSENV.2004.09.071</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3785',
    'Shoor et&nbsp;al. (1969)',
    'Shoor, S. K., Walker, Jr., R. D., &#x0026; Gubbins, K. E.: <span class="cmti-10">Salting out of nonpolar gases in</span> <span class="cmti-10">aqueous potassium hydroxide solutions</span>, J. Phys. Chem., 73, 312–317, <a href="https://doi.org/10.1021/J100722A006" >doi:10.1021/J100722A006</a> (1969). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2999',
    'Shunthirasingham et&nbsp;al. (2013)',
    'Shunthirasingham, C., Cao, X., Lei, Y. D., &#x0026; Wania, F.: <span class="cmti-10">Large bubbles</span> <span class="cmti-10">reduce the surface sorption artifact during inert gas stripping</span>, J. Chem. Eng. Data, 58, 792–797, <a href="https://doi.org/10.1021/JE301326T" >doi:10.1021/JE301326T</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3704',
    'Siebeck (1909)',
    'Siebeck, R.: <span class="cmti-10">Über die</span> <span class="cmti-10">Aufnahme von Stickoxydul im Blut</span>, Skand. Arch. Physiol., 21, 368–382, <a href="https://doi.org/10.1111/J.1748-1716.1909.TB00063.X" >doi:10.1111/J.1748-1716.1909.TB00063.X</a> (1909). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2973',
    'Siebers and Mattusch (1996)',
    'Siebers, J. &#x0026; Mattusch, P.: <span class="cmti-10">Determination</span> <span class="cmti-10">of airborne residues in greenhouses after application of pesticides</span>, Chemosphere, 33, 1597–1607, <a href="https://doi.org/10.1016/0045-6535(96)00279-2" >doi:10.1016/0045-6535(96)00279-2</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2967',
    'Siebers et&nbsp;al. (1994)',
    'Siebers, J., Gottschild, D., &#x0026; Nolting, H.-G.: <span class="cmti-10">Pesticides in precipitation in northern Germany</span>, Chemosphere, 28, 1559–1570, <a href="https://doi.org/10.1016/0045-6535(94)90249-6" >doi:10.1016/0045-6535(94)90249-6</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3135',
    'Sieg et&nbsp;al. (2008)',
    'Sieg, K., Fries, E., &#x0026; Püttmann, W.: <span class="cmti-10">Analysis of benzene, toluene, ethylbenzene, xylenes and </span><span class="cmmi-10">n</span><span class="cmti-10">-aldehydes</span> <span class="cmti-10">in melted snow water via solid-phase dynamic extraction combined</span> <span class="cmti-10">with gas chromatography/mass spectrometry</span>, J. Chromatogr. A, 1178, 178–186, <a href="https://doi.org/10.1016/J.CHROMA.2007.11.025" >doi:10.1016/J.CHROMA.2007.11.025</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2645',
    'Sieg et&nbsp;al. (2009)',
    'Sieg, K., Starokozheva, E., Schmidt, M. U., &#x0026; Püttmann, W.: <span class="cmti-10">Inverse temperature dependence of Henry&#x2019;s law coefficients for volatile</span> <span class="cmti-10">organic compounds in supercooled water</span>, Chemosphere, 77, 8–14, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2009.06.028" >doi:10.1016/J.CHEMOSPHERE.2009.06.028</a> (2009). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '795',
    'Signer et&nbsp;al. (1969)',
    'Signer, R., Arm, H., &#x0026; Daenicker, H.: <span class="cmti-10">Dampfdr</span><span class="cmti-10">ücke, Dichten,</span> <span class="cmti-10">thermodynamische Mischfunktionen und Brechungsindices der bin</span><span class="cmti-10">ären</span> <span class="cmti-10">Systeme Wasser-Tetrahydrofuran und Wasser-Di</span><span class="cmti-10">äthyl</span><span class="cmti-10">äther bei 25</span><span class="cmsy-7">∘</span>, Helv. Chim. Acta, 52, 2347–2351, <a href="https://doi.org/10.1002/HLCA.19690520816" >doi:10.1002/HLCA.19690520816</a> (1969). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2551',
    'Simpson and Lovell (1962)',
    'Simpson, L. B. &#x0026; Lovell, F. P.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">methyl, ethyl, and vinyl acetylene in several solvents</span>, J. Chem. Eng. Data, 7, 498–552, <a href="https://doi.org/10.1021/JE60015A017" >doi:10.1021/JE60015A017</a> (1962). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3808',
    'Sisi et&nbsp;al. (1971)',
    'Sisi, J. C., Dubeau, C., &#x0026; Ozanne, N.: <span class="cmti-10">Solubility of hydrogen selenide gas in water</span>, J. Chem. Eng. Data, 16, 78–79, <a href="https://doi.org/10.1021/JE60048A023" >doi:10.1021/JE60048A023</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2465',
    'Slater and Spedding (1981)',
    'Slater, R. M. &#x0026; Spedding, D. J.: <span class="cmti-10">Transport of</span> <span class="cmti-10">dieldrin between air and water</span>, Arch. Environ. Contam. Toxicol., 10, 25–33, <a href="https://doi.org/10.1007/BF01057572" >doi:10.1007/BF01057572</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2104',
    'Smith and Harvey (2007)',
    'Smith, F. L. &#x0026; Harvey, A. H.: <span class="cmti-10">Avoid common</span> <span class="cmti-10">pitfalls when using Henry&#x2019;s law</span>, Chem. Eng. Prog., pp. 33–39 (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3945',
    'Smith and Bomberger (1980)',
    'Smith, J. H. &#x0026; Bomberger, D. C.: <span class="cmti-10">Prediction</span> <span class="cmti-10">of volatilization rate of chemicals in water</span>, in: Hydrocarbons and Halogenated Hydrocarbons in the Aquatic Environment, edited by Afghan, B. K., Mackay, D., Braun, H. E., Chau, A. S. Y., Lawrence, J., Lean, D. R. S., Meresz, O., Miles, J. R. W., Pierce, R. C., Rees, G. A. V., White, R. E., Whittle, D. M., &#x0026; Williams, D. T., pp. 445–451, Plenum Press New York (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2236',
    'Smith et&nbsp;al. (1981a)',
    'Smith, J. H., Bomberger, D. C., &#x0026; Haynes, D. L.: <span class="cmti-10">Volatilization rates</span> <span class="cmti-10">of intermediate and low volatility chemicals from water</span>, Chemosphere, 10, 281–289, <a href="https://doi.org/10.1016/0045-6535(81)90028-X" >doi:10.1016/0045-6535(81)90028-X</a> (1981a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3071',
    'Smith et&nbsp;al. (1993)',
    'Smith, J. R., Neuhauser, E. F., Middleton, A. C., Cunningham, J. J., Weightman, R. L., &#x0026; Linz, D. G.: <span class="cmti-10">Treatment of organically</span> <span class="cmti-10">contaminated groundwaters in municipal activated sludge systems</span>, Water Environ. Res., 65, 804–818, <a href="https://doi.org/10.2175/WER.65.7.2" >doi:10.2175/WER.65.7.2</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2445',
    'Smith et&nbsp;al. (1981b)',
    'Smith, R. A., Porter, E. G., &#x0026; Miller, K. W.: <span class="cmti-10">The solubility of anesthetic gases in lipid bilayers</span>, Biochim. Biophys. Acta - Biomembranes, 645, 327–338 (1981b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '483',
    'Snider and Dawson (1985)',
    'Snider, J. R. &#x0026; Dawson, G. A.: <span class="cmti-10">Tropospheric</span> <span class="cmti-10">light alcohols, carbonyls, and acetonitrile: Concentrations in the</span> <span class="cmti-10">southwestern United States and Henry&#x2019;s law data</span>, J. Geophys. Res., 90, 3797–3805, <a href="https://doi.org/10.1029/JD090ID02P03797" >doi:10.1029/JD090ID02P03797</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3005',
    'Sotelo et&nbsp;al. (1989)',
    'Sotelo, J. L., Beltrán, F. J., Benitez, F. J., &#x0026; Beltrán-Heredia, J.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">law constant for the ozone-water system</span>, Wat. Res., 23, 1239–1246, <a href="https://doi.org/10.1016/0043-1354(89)90186-3" >doi:10.1016/0043-1354(89)90186-3</a> (1989). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3671',
    'Souchon et&nbsp;al. (2004)',
    'Souchon, I., Athès, V., Pierre, F.-X., &#x0026; Marin, M.: <span class="cmti-10">Liquid-liquid extraction and air stripping</span> <span class="cmti-10">in membrane contactor: application to aroma compounds recovery</span>, Desalination, 163, 39–46, <a href="https://doi.org/10.1016/S0011-9164(04)90174-9" >doi:10.1016/S0011-9164(04)90174-9</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2459',
    'Southworth (1979)',
    'Southworth, G. R.: <span class="cmti-10">The role of volatilization in</span> <span class="cmti-10">removing polycyclic aromatic hydrocarbons from aquatic environments</span>, Bull. Environ. Contam. Toxicol., 21, 507–514, <a href="https://doi.org/10.1007/BF01685462" >doi:10.1007/BF01685462</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3064',
    'St-Pierre et&nbsp;al. (2014)',
    'St-Pierre, J., Wetton, B., Zhai, Y., &#x0026; Gea, J.: <span class="cmti-10">Liquid water scavenging</span> <span class="cmti-10">of PEMFC contaminants</span>, J. Electrochem. Soc., 161, E3357–E3364, <a href="https://doi.org/10.1149/2.0291409JES" >doi:10.1149/2.0291409JES</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '799',
    'Staffelbach and Kok (1993)',
    'Staffelbach, T. A. &#x0026; Kok, G. L.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">law constants for aqueous solutions of hydrogen peroxide and</span> <span class="cmti-10">hydroxymethyl hydroperoxide</span>, J. Geophys. Res., 98, 12 713–12 717, <a href="https://doi.org/10.1029/93JD01022" >doi:10.1029/93JD01022</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2958',
    'Staples et&nbsp;al. (1997)',
    'Staples, C. A., Peterson, D. R., Parkerton, T. F., &#x0026; Adams, W. J.: <span class="cmti-10">The</span> <span class="cmti-10">environmental fate of phthalate esters: A literature review</span>, Chemosphere, 35, 667–749, <a href="https://doi.org/10.1016/S0045-6535(97)00195-1" >doi:10.1016/S0045-6535(97)00195-1</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '747',
    'Staudinger and Roberts (1996)',
    'Staudinger, J. &#x0026; Roberts, P. V.: <span class="cmti-10">A critical</span> <span class="cmti-10">review of Henry&#x2019;s</span> <span class="cmti-10">law constants for environmental applications</span>, Crit. Rev. Environ. Sci. Technol., 26, 205–297, <a href="https://doi.org/10.1080/10643389609388492" >doi:10.1080/10643389609388492</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1525',
    'Staudinger and Roberts (2001)',
    'Staudinger, J. &#x0026; Roberts, P. V.: <span class="cmti-10">A critical</span> <span class="cmti-10">compilation of Henry&#x2019;s law constant temperature dependence relations</span> <span class="cmti-10">for organic compounds in dilute aqueous solutions</span>, Chemosphere, 44, 561–576, <a href="https://doi.org/10.1016/S0045-6535(00)00505-1" >doi:10.1016/S0045-6535(00)00505-1</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2441',
    'Steward et&nbsp;al. (1973)',
    'Steward, A., Allott, P. R., Cowles, A. L., &#x0026; Mapleson, W. W.: <span class="cmti-10">Solubility</span> <span class="cmti-10">coefficients for inhaled anaesthetics for water, oil and biological media</span>, Br. J. Anaesth., 45, 282–293, <a href="https://doi.org/10.1093/BJA/45.3.282" >doi:10.1093/BJA/45.3.282</a> (1973). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2933',
    'Stock and Kuß (1917)',
    'Stock, A. &#x0026; Kuß, E.: <span class="cmti-10">Zur Kenntnis des Kohlenoxysulfides COS</span>, Ber. Dtsch. Chem. Ges., 50, 159–164, <a href="https://doi.org/10.1002/CBER.19170500125" >doi:10.1002/CBER.19170500125</a> (1917). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2467',
    'Stoelting and Longshore (1972)',
    'Stoelting, R. K. &#x0026; Longshore, R. E.: <span class="cmti-10">The</span> <span class="cmti-10">effects of</span> <span class="cmti-10">temperature on fluroxene, halothane, and methoxyflurane blood-gas and</span> <span class="cmti-10">cerebrospinal fluid-gas partition coefficients</span>, Anesthesiology, 36, 503–505, <a href="https://doi.org/10.1097/00000542-197205000-00018" >doi:10.1097/00000542-197205000-00018</a> (1972). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1937',
    'Straver and de Loos (2005)',
    'Straver, E. J. M. &#x0026; de Loos, T. W.: <span class="cmti-10">Determination of Henry&#x2019;s law constants and activity</span> <span class="cmti-10">coefficients at infinite dilution of flavor compounds in water at 298 </span>K <span class="cmti-10">with a gas-chromatographic method</span>, J. Chem. Eng. Data, 50, 1171–1176, <a href="https://doi.org/10.1021/JE0495942" >doi:10.1021/JE0495942</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1571',
    'Strekowski and George (2005)',
    'Strekowski, R. S. &#x0026; George, C.: <span class="cmti-10">Measurement</span> <span class="cmti-10">of Henry&#x2019;s law constants for acetone, 2-butanone, 2,3-butanedione and</span> <span class="cmti-10">isobutyraldehyde using a horizontal flow reactor</span>, J. Chem. Eng. Data, 50, 804–810, <a href="https://doi.org/10.1021/JE034137R" >doi:10.1021/JE034137R</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3592',
    'Stumm (1958)',
    'Stumm, W.: <span class="cmti-10">Ozone as a disinfectant for water and sewage</span>, J. Boston Soc. Civil Eng., 45, 68–79 (1958). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '997',
    'Suleimenov and Krupp (1994)',
    'Suleimenov, O. M. &#x0026; Krupp, R. E.: <span class="cmti-10">Solubility</span> <span class="cmti-10">of hydrogen sulfide in pure water and in</span> NaCl <span class="cmti-10">solutions, from 20 to 320</span><span class="cmsy-7">∘</span>C <span class="cmti-10">and at saturation pressures</span>, Geochim. Cosmochim. Acta, 58, 2433–2444, <a href="https://doi.org/10.1016/0016-7037(94)90022-1" >doi:10.1016/0016-7037(94)90022-1</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1157',
    'Suntio et&nbsp;al. (1988)',
    'Suntio, L. R., Shiu, W. Y., Mackay, D., Seiber, J. N., &#x0026; Glotfelty, D.: <span class="cmti-10">Critical</span> <span class="cmti-10">review of Henry&#x2019;s law constants for pesticides</span>, Rev. Environ. Contam. Toxicol., 103, 1–59, <a href="https://doi.org/10.1007/978-1-4612-3850-8_1" >doi:10.1007/978-1-4612-3850-8_1</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1162',
    'Suzuki et&nbsp;al. (1992)',
    'Suzuki, T., Ohtaguchi, K., &#x0026; Koide, K.: <span class="cmti-10">Application of principal components analysis to calculate</span> <span class="cmti-10">Henry&#x2019;s constant from molecular structure</span>, Comput. Chem., 16, 41–52, <a href="https://doi.org/10.1016/0097-8485(92)85007-L" >doi:10.1016/0097-8485(92)85007-L</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1340',
    'Swain and Thornton (1962)',
    'Swain, C. G. &#x0026; Thornton, E. R.: <span class="cmti-10">Initial-state</span> <span class="cmti-10">and transition-state isotope effects of methyl halides in light and heavy</span> <span class="cmti-10">water</span>, J. Am. Chem. Soc., 84, 822–826, <a href="https://doi.org/10.1021/JA00864A029" >doi:10.1021/JA00864A029</a> (1962). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3703',
    'Sy and Hasbrouck (1964)',
    'Sy, W. P. &#x0026; Hasbrouck, J. D.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">nitrous oxide in water and in canine blood</span>, Anesthesiology, 25, 59–63, <a href="https://doi.org/10.1097/00000542-196401000-00010" >doi:10.1097/00000542-196401000-00010</a> (1964). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2451',
    'Tabai et&nbsp;al. (1997)',
    'Tabai, S., Rogalski, M., Solimando, R., &#x0026; Malanowski, S. K.: <span class="cmti-10">Activity coefficients</span> <span class="cmti-10">of chlorophenols in water at infinite dilution</span>, J. Chem. Eng. Data, 42, 1147–1150, <a href="https://doi.org/10.1021/JE960336H" >doi:10.1021/JE960336H</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2531',
    'Taft et&nbsp;al. (1985)',
    'Taft, R. W., Abraham, M. H., Doherty, R. M., &#x0026; Kamlet, M. J.: <span class="cmti-10">The</span> <span class="cmti-10">molecular properties governing solubilities of organic nonelectrolytes in</span> <span class="cmti-10">water</span>, Nature, 313, 384–386, <a href="https://doi.org/10.1038/313384A0" >doi:10.1038/313384A0</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2289',
    'Talmi and Mesmer (1975)',
    'Talmi, Y. &#x0026; Mesmer, R. E.: <span class="cmti-10">Studies on vaporization and halogen decomposition</span> <span class="cmti-10">of methyl mercury compounds using gc with a microwave detector</span>, Wat. Res., 9, 547–552, <a href="https://doi.org/10.1016/0043-1354(75)90080-9" >doi:10.1016/0043-1354(75)90080-9</a> (1975). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '598',
    'Tancrède and Yanagisawa (1990)',
    'Tancrède, M. V. &#x0026; Yanagisawa, Y.: <span class="cmti-10">An</span> <span class="cmti-10">analytical method to determine Henry&#x2019;s law constant for selected volatile</span> <span class="cmti-10">organic compounds at concentrations and temperatures corresponding</span> <span class="cmti-10">to tap water use</span>, J. Air Waste Manage. Assoc., 40, 1658–1663, <a href="https://doi.org/10.1080/10473289.1990.10466813" >doi:10.1080/10473289.1990.10466813</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3966',
    'Taube and Dodgen (1949)',
    'Taube, H. &#x0026; Dodgen, H.: <span class="cmti-10">Applications of</span> <span class="cmti-10">radioactive chlorine to the study of the mechanisms of reactions involving</span> <span class="cmti-10">changes in the oxidation state of chlorine</span>, J. Am. Chem. Soc., 71, 3330–3336, <a href="https://doi.org/10.1021/JA01178A016" >doi:10.1021/JA01178A016</a> (1949). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3132',
    'Teja et&nbsp;al. (2001)',
    'Teja, A. S., Gupta, A. K., Bullock, K., Chai, X.-S., &#x0026; Zhu, J.: <span class="cmti-10">Henry&#x2019;s constants of methanol</span> <span class="cmti-10">in aqueous systems containing salts</span>, Fluid Phase Equilib., 185, 265–274, <a href="https://doi.org/10.1016/S0378-3812(01)00476-9" >doi:10.1016/S0378-3812(01)00476-9</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2807',
    'Templeton and King (1971)',
    'Templeton, J. C. &#x0026; King, E. L.: <span class="cmti-10">Kinetic and</span> <span class="cmti-10">equilibrium studies on azidochromium(III) ion in concentrated perchloric</span> <span class="cmti-10">acid</span>, J. Am. Chem. Soc., 93, 7160–7166, <a href="https://doi.org/10.1021/JA00755A009" >doi:10.1021/JA00755A009</a> (1971). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1146',
    'ten Hulscher et&nbsp;al. (1992)',
    'ten Hulscher, T. E. M., van der Velde, L. E., &#x0026; Bruggeman, W. A.: <span class="cmti-10">Temperature dependence</span> <span class="cmti-10">of Henry&#x2019;s law constants for selected chlorobenzenes, polychlorinated</span> <span class="cmti-10">biphenyls and polycyclic aromatic hydrocarbons</span>, Environ. Toxicol. Chem., 11, 1595–1603, <a href="https://doi.org/10.1002/ETC.5620111109" >doi:10.1002/ETC.5620111109</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3122',
    'Terraglio and Manganelli (1967)',
    'Terraglio, F. P. &#x0026; Manganelli, R. M.: <span class="cmti-10">The</span> <span class="cmti-10">absorption of atmospheric sulfur dioxide by water solutions</span>, J. Air Pollut. Control Assoc., 17, 403–406, <a href="https://doi.org/10.1080/00022470.1967.10468999" >doi:10.1080/00022470.1967.10468999</a> (1967). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3948',
    'Thomas et&nbsp;al. (2006)',
    'Thomas, J. E., Ou, L.-T., Allen Jr., L. H., Vu, J. C., &#x0026; Dickson, D. W.: <span class="cmti-10">Henry&#x2019;s law constants and mass transfer coefficients for methyl bromide</span> <span class="cmti-10">and 1,3-dichloropropene applied to Florida sandy field soil</span>, Chemosphere, 62, 980–988, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2005.06.017" >doi:10.1016/J.CHEMOSPHERE.2005.06.017</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '895',
    'Thomas et&nbsp;al. (1998)',
    'Thomas, K., Volz-Thomas, A., Mihelcic, D., Smit, H. G. J., &#x0026; Kley, D.: <span class="cmti-10">On the exchange of</span> NO<sub><span class="cmr-7">3</span></sub> <span class="cmti-10">radicals with aqueous solutions:</span> <span class="cmti-10">Solubility and sticking coefficient</span>, J. Atmos. Chem., 29, 17–43, <a href="https://doi.org/10.1023/A:1005860312363" >doi:10.1023/A:1005860312363</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '203',
    'Thompson and Zafiriou (1983)',
    'Thompson, A. M. &#x0026; Zafiriou, O. C.: <span class="cmti-10">Air-sea</span> <span class="cmti-10">fluxes of transient atmospheric species</span>, J. Geophys. Res., 88, 6696–6708, <a href="https://doi.org/10.1029/JC088IC11P06696" >doi:10.1029/JC088IC11P06696</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3680',
    'Thompson et&nbsp;al. (2018)',
    'Thompson, J. G., Matin, N. S., Abad, K., Onneweer, F., Bhatnagar, S., &#x0026; Liu, K.: <span class="cmti-10">Determining the Henry&#x2019;s volatility coefficient of nitrosamines</span> <span class="cmti-10">in</span> CO<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">capture solvents</span>, Int. J. Greenhouse Gas Control, 73, 104–110, <a href="https://doi.org/10.1016/J.IJGGC.2018.04.004" >doi:10.1016/J.IJGGC.2018.04.004</a> (2018). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'timmermans60',
    'Timmermans (1960)',
    'Timmermans, J.: The Physico-Chemical Constants of Binary Systems in Concentrated Solutions, Vol. 4, Interscience Publisher, Inc., New York, NY (1960). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3712',
    'Timofejew (1890)',
    'Timofejew, W.: <span class="cmti-10">Über die Absorption von Wasserstoff</span> <span class="cmti-10">und Sauerstoff in Wasser und Alkohol</span>, Z. Phys. Chem., 6, 141–152, <a href="https://doi.org/10.1515/ZPCH-1890-0614" >doi:10.1515/ZPCH-1890-0614</a> (1890). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2494',
    'Tittlemier et&nbsp;al. (2002)',
    'Tittlemier, S. A., Halldorson, T., Stern, G. A., &#x0026; Tomy, G. T.: <span class="cmti-10">Vapor</span> <span class="cmti-10">pressures, aqueous solubilities, and Henry&#x2019;s law constants of some</span> <span class="cmti-10">brominated flame retardants</span>, Environ. Toxicol. Chem., 21, 1804–1810, <a href="https://doi.org/10.1002/ETC.5620210907" >doi:10.1002/ETC.5620210907</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2469',
    'Tittlemier et&nbsp;al. (2004)',
    'Tittlemier, S. A., Braekevelt, E., Halldorson, T., Reddy, C. M., &#x0026; Norstrom, R. J.: <span class="cmti-10">Vapour pressures, aqueous solubilities, Henry&#x2019;s</span> <span class="cmti-10">Law constants, and octanol/water partition coefficients of a series of</span> <span class="cmti-10">mixed halogenated dimethyl bipyrroles</span>, Chemosphere, 57, 1373–1381, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2004.08.061" >doi:10.1016/J.CHEMOSPHERE.2004.08.061</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'tomlin98',
    'Tomlin (1998)',
    'Tomlin, C. D. S.: The Pesticide Manual, 11th Edition, British Crop Production Council, ISBN 1901396118 (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2443',
    'Trampe and Eckert (1993)',
    'Trampe, D. B. &#x0026; Eckert, C. A.: <span class="cmti-10">A dew point</span> <span class="cmti-10">technique for limiting activity coefficients in nonionic solutions</span>, AIChE J., 39, 1045–1050, <a href="https://doi.org/10.1002/AIC.690390613" >doi:10.1002/AIC.690390613</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1155',
    'Tremp et&nbsp;al. (1993)',
    'Tremp, J., Mattrel, P., Fingler, S., &#x0026; Giger, W.: <span class="cmti-10">Phenols</span> <span class="cmti-10">and nitrophenols as tropospheric pollutants: Emissions from automobile</span> <span class="cmti-10">exhausts and phase transfer in the atmosphere</span>, Water Air Soil Pollut., 68, 113–123, <a href="https://doi.org/10.1007/BF00479396" >doi:10.1007/BF00479396</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1217',
    'Treves et&nbsp;al. (2000)',
    'Treves, K., Shragina, L., &#x0026; Rudich, Y.: <span class="cmti-10">Henry&#x2019;s law constants of some </span><span class="cmmi-10">β</span><span class="cmti-10">-, </span><span class="cmmi-10">γ</span><span class="cmti-10">-, and </span><span class="cmmi-10">δ</span><span class="cmti-10">-hydroxy</span> <span class="cmti-10">nitrates of atmospheric interest</span>, Environ. Sci. Technol., 34, 1197–1203, <a href="https://doi.org/10.1021/ES990558A" >doi:10.1021/ES990558A</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '856',
    'Tse et&nbsp;al. (1992)',
    'Tse, G., Orbey, H., &#x0026; Sandler, S. I.: <span class="cmti-10">Infinite dilution activity coefficients and Henry&#x2019;s law coefficients</span> <span class="cmti-10">of some priority water pollutants determined by a relative gas</span> <span class="cmti-10">chromatographic method</span>, Environ. Sci. Technol., 26, 2017–2022, <a href="https://doi.org/10.1021/ES00034A021" >doi:10.1021/ES00034A021</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2922',
    'Tsibul&#x2019;skii et&nbsp;al. (1979)',
    'Tsibul&#x2019;skii, V. V., Tsibul&#x2019;skaya, I. A., &#x0026; Yaglitskaya, N. N.: <span class="cmti-10">Sampling</span> <span class="cmti-10">and storage of samples for the gas-chromatographic Determination of</span> <span class="cmti-10">aromatic-hydrocarbons as microimpurities in gases</span>, J. Anal. Chem. USSR, 34, 1052–1055 (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2911',
    'Tsonopoulos and Wilson (1983)',
    'Tsonopoulos, C. &#x0026; Wilson, G. M.: <span class="cmti-10">High-temperature mutual solubilities</span> <span class="cmti-10">of hydrocarbons and water. Part I: Benzene, cyclohexane and </span><span class="cmmi-10">n</span><span class="cmti-10">-hexane</span>, AIChE J., 29, 990–999, <a href="https://doi.org/10.1002/AIC.690290618" >doi:10.1002/AIC.690290618</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3668',
    'Tsuji et&nbsp;al. (1990)',
    'Tsuji, M., Nakano, T., &#x0026; T.Okuno: <span class="cmti-10">Desorption of odor substances</span> <span class="cmti-10">from water bodies to the atmosphere</span>, Atmos. Environ., 24, 2019–2021, <a href="https://doi.org/10.1016/0960-1686(90)90236-G" >doi:10.1016/0960-1686(90)90236-G</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2980',
    'Tucker et&nbsp;al. (1981)',
    'Tucker, E. E., Lane, E. H., &#x0026; Christian, S. D.: <span class="cmti-10">Vapor pressure studies of hydrophobic interactions. formation of</span> <span class="cmti-10">benzene-benzene and cyclohexane-cyclohexanol dimers in dilute aqueous</span> <span class="cmti-10">solution</span>, J. Solution Chem., 10, 1–20, <a href="https://doi.org/10.1007/BF00652776" >doi:10.1007/BF00652776</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2463',
    'Turner et&nbsp;al. (1996)',
    'Turner, L. H., Chiew, Y. C., Ahlert, R. C., &#x0026; Kosson, D. S.: <span class="cmti-10">Measuring vapor-liquid</span> <span class="cmti-10">equilibrium for aqueous-organic systems: Review and a new technique</span>, AIChE J., 42, 1772–1788, <a href="https://doi.org/10.1002/AIC.690420629" >doi:10.1002/AIC.690420629</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1516',
    'Ueberfeld et&nbsp;al. (2001)',
    'Ueberfeld, J., Zbinden, H., Gisin, N., &#x0026; Pellaux, J. P.: <span class="cmti-10">Determination of Henry&#x2019;s</span> <span class="cmti-10">constant using a photoacoustic sensor</span>, J. Chem. Thermodyn., 33, 755–764, <a href="https://doi.org/10.1006/JCHT.2000.0776" >doi:10.1006/JCHT.2000.0776</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'vankrevelen49',
    'Van Krevelen et&nbsp;al. (1949)',
    'Van Krevelen, D. W., Hoftijzer, P. J., &#x0026; Huntjens, F. J.: <span class="cmti-10">Composition</span> <span class="cmti-10">and vapor pressures of aqueous solutions of ammonia, carbon dioxide and</span> <span class="cmti-10">hydrogen sulfide</span>, Recl. Trav. Chim. Pays-Bas, 68, 191–216 (1949). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2125',
    'van Roon et&nbsp;al. (2005)',
    'van Roon, A., Parsons, J. R., Kloeze, A. M. T., &#x0026; Govers, H. A. J.: <span class="cmti-10">Fate and transport of monoterpenes through soils. Part I. Prediction of</span> <span class="cmti-10">temperature dependent soil fate model input-parameters</span>, Chemosphere, 61, 599–609, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2005.02.081" >doi:10.1016/J.CHEMOSPHERE.2005.02.081</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3670',
    'van Ruth and Villeneuve (2002)',
    'van Ruth, S. M. &#x0026; Villeneuve, E.: <span class="cmti-10">Influence</span> <span class="cmti-10">of </span><span class="cmmi-10">β</span><span class="cmti-10">-lactoglobulin, pH and presence of other aroma compounds on</span> <span class="cmti-10">the air/liquid partition coefficients of 20 aroma compounds varying</span> <span class="cmti-10">in functional group and chain length</span>, Food Chem., 79, 157–164, <a href="https://doi.org/10.1016/S0308-8146(02)00124-3" >doi:10.1016/S0308-8146(02)00124-3</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3611',
    'van Ruth et&nbsp;al. (2001)',
    'van Ruth, S. M., Grossmann, I., Geary, M., &#x0026; Delahunty, C. M.: <span class="cmti-10">Interactions</span> <span class="cmti-10">between artificial saliva and 20 aroma compounds in water and oil model</span> <span class="cmti-10">systems</span>, J. Agric. Food Chem., 49, 2409–2413, <a href="https://doi.org/10.1021/JF001510F" >doi:10.1021/JF001510F</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3643',
    'van Ruth et&nbsp;al. (2002)',
    'van Ruth, S. M., de Vries, G., Geary, M., &#x0026; Giannouli, P.: <span class="cmti-10">Influence of composition</span> <span class="cmti-10">and structure of oil-in-water emulsions on retention of aroma</span> <span class="cmti-10">compounds</span>, J. Sci. Food Agric., 82, 1028–1035, <a href="https://doi.org/10.1002/JSFA.1137" >doi:10.1002/JSFA.1137</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3707',
    'Van Slyke et&nbsp;al. (1934)',
    'Van Slyke, D. D., Dillon, R. T., &#x0026; Margaria, R.: <span class="cmti-10">Studies of gas and electrolyte</span> <span class="cmti-10">equilibria in blood: XVIII. Solubility and physical state of atmospheric</span> <span class="cmti-10">nitrogen in blood cells and plasma</span>, J. Biol. Chem., 105, 571–596, <a href="https://doi.org/10.1016/S0021-9258(18)75528-2" >doi:10.1016/S0021-9258(18)75528-2</a> (1934). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2896',
    'Vane and Giroux (2000)',
    'Vane, L. M. &#x0026; Giroux, E. L.: <span class="cmti-10">Henry&#x2019;s law constants</span> <span class="cmti-10">and micellar partitioning of volatile organic compounds in surfactant</span> <span class="cmti-10">solutions</span>, J. Chem. Eng. Data, 45, 38–47, <a href="https://doi.org/10.1021/JE990195U" >doi:10.1021/JE990195U</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3582',
    'Vasilakos et&nbsp;al. (2021)',
    'Vasilakos, P., Hu, Y., Russell, A., &#x0026; Nenes, A.: <span class="cmti-10">Determining the role of acidity, fate</span> <span class="cmti-10">and formation of IEPOX-derived SOA in CMAQ</span>, Atmos., 12, 707, <a href="https://doi.org/10.3390/ATMOS12060707" >doi:10.3390/ATMOS12060707</a> (2021). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '995',
    'Villalta et&nbsp;al. (1996)',
    'Villalta, P. W., Lovejoy, E. R., &#x0026; Hanson, D. R.: <span class="cmti-10">Reaction probability of peroxyacetyl</span> <span class="cmti-10">radical on aqueous surfaces</span>, Geophys. Res. Lett., 23, 1765–1768, <a href="https://doi.org/10.1029/96GL01286" >doi:10.1029/96GL01286</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2925',
    'Vitenberg and Dobryakov (2008)',
    'Vitenberg, A. G. &#x0026; Dobryakov, Y. G.: <span class="cmti-10">Gas-chromatographic determination of the distribution ratios of volatile</span> <span class="cmti-10">substances in gas-liquid systems</span>, Russ. J. Appl. Chem., 81, 339–359, <a href="https://doi.org/10.1134/S1070427208030014" >doi:10.1134/S1070427208030014</a> (2008). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1492',
    'Vitenberg et&nbsp;al. (1974)',
    'Vitenberg, A. G., Ioffe, B. V., &#x0026; Borisov, V. N.: <span class="cmti-10">Application of phase equilibria to gas chromatographic</span> <span class="cmti-10">trace analysis</span>, Chromatographia, 7, 610–619, <a href="https://doi.org/10.1007/BF02269053" >doi:10.1007/BF02269053</a> (1974). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '728',
    'Vitenberg et&nbsp;al. (1975)',
    'Vitenberg, A. G., Ioffe, B. V., Dimitrova, Z. S., &#x0026; Butaeva, I. L.: <span class="cmti-10">Determination of</span> <span class="cmti-10">gas-liquid partition coefficients by means of gas chromatographic analysis</span>, J. Chromatogr., 112, 319–327, <a href="https://doi.org/10.1016/S0021-9673(00)99964-3" >doi:10.1016/S0021-9673(00)99964-3</a> (1975). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '446',
    'Vogt et&nbsp;al. (1996)',
    'Vogt, R., Crutzen, P. J., &#x0026; Sander, R.: <span class="cmti-10">A mechanism for halogen release from sea-salt aerosol</span> <span class="cmti-10">in the remote marine boundary layer</span>, Nature, 383, 327–330, <a href="https://doi.org/10.1038/383327A0" >doi:10.1038/383327A0</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '897',
    'Vogt et&nbsp;al. (1999)',
    'Vogt, R., Sander, R., von Glasow, R., &#x0026; Crutzen, P. J.: <span class="cmti-10">Iodine chemistry and its role in</span> <span class="cmti-10">halogen activation and ozone loss in the marine boundary layer: A model</span> <span class="cmti-10">study</span>, J. Atmos. Chem., 32, 375–395, <a href="https://doi.org/10.1023/A:1006179901037" >doi:10.1023/A:1006179901037</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3134',
    'Volkamer et&nbsp;al. (2009)',
    'Volkamer, R., Ziemann, P. J., &#x0026; Molina, M. J.: <span class="cmti-10">Secondary organic</span> <span class="cmti-10">aerosol formation from acetylene (</span>C<sub><span class="cmr-7">2</span></sub>H<sub><span class="cmr-7">2</span></sub><span class="cmti-10">): seed effect on SOA yields due</span> <span class="cmti-10">to organic photochemistry in the aerosol aqueous phase</span>, Atmos. Chem. Phys., 9, 1907–1928, <a href="https://doi.org/10.5194/ACP-9-1907-2009" >doi:10.5194/ACP-9-1907-2009</a> (2009). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3714',
    'von Antropoff (1910)',
    'von Antropoff, A.: <span class="cmti-10">The solubility of xenon, krypton,</span> <span class="cmti-10">argon, neon, and helium in water</span>, Proc. R. Soc. Lond. A, 83, 474–482, <a href="https://doi.org/10.1098/RSPA.1910.0036" >doi:10.1098/RSPA.1910.0036</a> (1910). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1721',
    'von Hartungen et&nbsp;al. (2004)',
    'von Hartungen, E., Wisthaler, A., Mikoviny, T., Jaksch, D., Boscaini, E., Dunphy, P. J., &#x0026; Märk, T. D.: <span class="cmti-10">Proton-transfer-reaction mass</span> <span class="cmti-10">spectrometry (PTR-MS) of carboxylic acids. Determination of Henry&#x2019;s</span> <span class="cmti-10">law constants and axillary odour investigations</span>, Int. J. Mass Spectrom., 239, 243–248, <a href="https://doi.org/10.1016/J.IJMS.2004.09.009" >doi:10.1016/J.IJMS.2004.09.009</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3551',
    'Vrabec et&nbsp;al. (2001)',
    'Vrabec, J., Stoll, J., &#x0026; Hasse, H.: <span class="cmti-10">A set of molecular models for symmetric quadrupolar fluids</span>, J. Phys. Chem. B, 105, 12 126–12 133, <a href="https://doi.org/10.1021/JP012542O" >doi:10.1021/JP012542O</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'vitovec68',
    'Vtovec (1968)',
    'Vtovec, J.: <span class="cmti-10">Absorption of acetylene and carbon dioxide in</span> <span class="cmti-10">water, xylene and methanol in a packed column</span>, Collect. Czech. Chem. Commun., 33, 1203–1210, <a href="https://doi.org/10.1135/CCCC19681203" >doi:10.1135/CCCC19681203</a> (1968). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '489',
    'Wagman et&nbsp;al. (1982)',
    'Wagman, D. D., Evans, W. H., Parker, V. B., Schumm, R. H., Halow, I., Bailey, S. M., Churney, K. L., &#x0026; Nuttall, R. L.: <span class="cmti-10">The NBS tables of</span> <span class="cmti-10">chemical thermodynamic properties; Selected values for inorganic and</span> C<sub><span class="cmr-7">1</span></sub> <span class="cmti-10">and</span> C<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">organic substances in SI units</span>, J. Phys. Chem. Ref. Data, 11, suppl. 2, URL https://srd.nist.gov/JPCRD/jpcrdS2Vol11.pdf (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2555',
    'Wagner and Pruss (1993)',
    'Wagner, W. &#x0026; Pruss, A.: <span class="cmti-10">International equations</span> <span class="cmti-10">for the saturation properties of ordinary water substance. Revised</span> <span class="cmti-10">according to the international temperature scale of 1990. Addendum to</span> <span class="cmti-10">J.</span><span class="cmti-10"> Phys.</span><span class="cmti-10"> Chem.</span><span class="cmti-10"> Ref.</span><span class="cmti-10"> Data 16, 893 (1987)</span>, J. Phys. Chem. Ref. Data, 22, 783–787, <a href="https://doi.org/10.1063/1.555926" >doi:10.1063/1.555926</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3409',
    'Wang et&nbsp;al. (2017)',
    'Wang, C., Yuan, T., Wood, S. A., Goss, K.-U., Li, J., Ying, Q., &#x0026; Wania, F.: <span class="cmti-10">Uncertain Henry&#x2019;s law constants compromise equilibrium partitioning</span> <span class="cmti-10">calculations of atmospheric oxidation products</span>, Atmos. Chem. Phys., 17, 7529–7540, <a href="https://doi.org/10.5194/ACP-17-7529-2017" >doi:10.5194/ACP-17-7529-2017</a> (2017). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '318',
    'Wang et&nbsp;al. (1994)',
    'Wang, T. X., Kelley, M. D., Cooper, J. N., Beckwith, R. C., &#x0026; Margerum, D. W.: <span class="cmti-10">Equilibrium, kinetic, and UV-spectral characteristics of aqueous</span> <span class="cmti-10">bromine chloride, bromine, and chlorine species</span>, Inorg. Chem., 33, 5872–5878, <a href="https://doi.org/10.1021/IC00103A040" >doi:10.1021/IC00103A040</a> (1994). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2949',
    'Wang and Wong (2002)',
    'Wang, Y. H. &#x0026; Wong, P. K.: <span class="cmti-10">Mathematical</span> <span class="cmti-10">relationships between vapor pressure, water solubility, Henry&#x2019;s law</span> <span class="cmti-10">constant, </span><span class="cmmi-10">n</span><span class="cmti-10">-octanol/water partition coefficent and gas chromatographic</span> <span class="cmti-10">retention index of polychlorinated-dibenzo-dioxins</span>, Wat. Res., 36, 350–355, <a href="https://doi.org/10.1016/S0043-1354(01)00192-0" >doi:10.1016/S0043-1354(01)00192-0</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2536',
    'Wania and Dugani (2003)',
    'Wania, F. &#x0026; Dugani, C. B.: <span class="cmti-10">Assessing the</span> <span class="cmti-10">long-range transport potential of polybrominated diphenyl ethers: A</span> <span class="cmti-10">comparison of four multimedia models</span>, Environ. Toxicol. Chem., 22, 1252–1261, <a href="https://doi.org/10.1002/ETC.5620220610" >doi:10.1002/ETC.5620220610</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '67',
    'Warneck (1988)',
    'Warneck, P.: Chemistry of the Natural Atmosphere, Acad., San Diego, CA, ISBN 0127356304 (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1257',
    'Warneck (1999)',
    'Warneck, P.: <span class="cmti-10">The relative importance of various pathways</span> <span class="cmti-10">for the oxidation of sulfur dioxide and nitrogen dioxide in sunlit</span> <span class="cmti-10">continental fair weather clouds</span>, Phys. Chem. Chem. Phys., 1, 5471–5483, <a href="https://doi.org/10.1039/A906558J" >doi:10.1039/A906558J</a> (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2432',
    'Warneck (2003)',
    'Warneck, P.: <span class="cmti-10">The solubility of ozone in water</span>, in: Chemicals in the Atmosphere: Solubility, Sources and Reactivity, edited by Fogg, P. &#x0026; Sangster, J., pp. 225–228, John Wiley &#x0026; Sons, Inc., ISBN 978-0-471-98651-5 (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2212',
    'Warneck (2005)',
    'Warneck, P.: <span class="cmti-10">Multi-phase chemistry of</span> C<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">and</span> C<sub><span class="cmr-7">3</span></sub> <span class="cmti-10">organic</span> <span class="cmti-10">compounds in the marine atmosphere</span>, J. Atmos. Chem., 51, 119–159, <a href="https://doi.org/10.1007/S10874-005-5984-7" >doi:10.1007/S10874-005-5984-7</a> (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1831',
    'Warneck (2006)',
    'Warneck, P.: <span class="cmti-10">A note on the temperature dependence of</span> <span class="cmti-10">Henry&#x2019;s Law coefficients for methanol and ethanol</span>, Atmos. Environ., 40, 7146–7151, <a href="https://doi.org/10.1016/J.ATMOSENV.2006.06.024" >doi:10.1016/J.ATMOSENV.2006.06.024</a> (2006). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1855',
    'Warneck (2007)',
    'Warneck, P.: <span class="cmti-10">A review of Henry&#x2019;s law coefficients for</span> <span class="cmti-10">chlorine-containing</span> C<sub><span class="cmr-7">1</span></sub> <span class="cmti-10">and</span> C<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">hydrocarbons</span>, Chemosphere, 69, 347–361, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2007.04.088" >doi:10.1016/J.CHEMOSPHERE.2007.04.088</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2828',
    'Warneck and Williams (2012)',
    'Warneck, P. &#x0026; Williams, J.: The Atmospheric Chemist&#x2019;s Companion: Numerical Data for Use in the Atmospheric Sciences, Springer Verlag, <a href="https://doi.org/10.1007/978-94-007-2275-0" >doi:10.1007/978-94-007-2275-0</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '669',
    'Warneck et&nbsp;al. (1996)',
    'Warneck, P., Mirabel, P., Salmon, G. A., van Eldik, R., Vinckier, C., Wannowius, K. J., &#x0026; Zetzsch, C.: <span class="cmti-10">Chapter 2: Review of the</span> <span class="cmti-10">activities and achievements of the EUROTRAC subproject HALIPP</span>, in: Heterogeneous and Liquid-Phase Processes, edited by Warneck, P., pp. 7–74, Springer Verlag, Berlin, <a href="https://doi.org/10.1007/978-3-642-61445-3_2" >doi:10.1007/978-3-642-61445-3_2</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2900',
    'Warner et&nbsp;al. (1980)',
    'Warner, H. P., Cohen, J. M., &#x0026; Ireland, J. C.: <span class="cmti-10">Determination of Henry&#x2019;s</span> <span class="cmti-10">law constants of selected priority pollutants</span>, Tech. rep., U.S. EPA, Municipal Environmental Research Laboratory, Wastewater Research Division, Cincinnati, Ohio, 45268, USA (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2240',
    'Warner and Weiss (1985)',
    'Warner, M. J. &#x0026; Weiss, R. F.: <span class="cmti-10">Solubilities of</span> <span class="cmti-10">chlorofluorocarbons 11 and 12 in water and seawater</span>, Deep-Sea Res. A, 32, 1485–1497 (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3550',
    'Warr et&nbsp;al. (2015)',
    'Warr, O., Ballentine, C. J., Mu, J., &#x0026; Masters, A.: <span class="cmti-10">Optimizing noble gas-water interactions</span> <span class="cmti-10">via Monte Carlo simulations</span>, J. Phys. Chem. B, 119, 14 486–14 495, <a href="https://doi.org/10.1021/ACS.JPCB.5B06389" >doi:10.1021/ACS.JPCB.5B06389</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '791',
    'Wasik and Tsang (1970)',
    'Wasik, S. P. &#x0026; Tsang, W.: <span class="cmti-10">Gas chromatographic</span> <span class="cmti-10">determination of partition coefficients of some unsaturated hydrocarbons</span> <span class="cmti-10">and their deuterated isomers in aqueous silver nitrate solutions</span>, J. Phys. Chem., 74, 2970–2976, <a href="https://doi.org/10.1021/J100709A023" >doi:10.1021/J100709A023</a> (1970). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2538',
    'Watanabe (1993)',
    'Watanabe, T.: <span class="cmti-10">Relationship between volatilization rates</span> <span class="cmti-10">and physicochemical properties of some pesticides</span>, J. Pestic. Sci., 18, 201–209, <a href="https://doi.org/10.1584/JPESTICS.18.3_201" >doi:10.1584/JPESTICS.18.3_201</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '824',
    'Watts and Brimblecombe (1987)',
    'Watts, S. F. &#x0026; Brimblecombe, P.: <span class="cmti-10">The</span> <span class="cmti-10">Henry&#x2019;s law constant of dimethyl sulphoxide</span>, Environ. Technol. Lett., 8, 483–486, <a href="https://doi.org/10.1080/09593338709384509" >doi:10.1080/09593338709384509</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2375',
    'Wauchope and Haque (1972)',
    'Wauchope, R. D. &#x0026; Haque, R.: <span class="cmti-10">Aqueous</span> <span class="cmti-10">solutions of nonpolar compounds. Heat-capacity effects</span>, Can. J. Chem., 50, 133–138, <a href="https://doi.org/10.1139/V72-022" >doi:10.1139/V72-022</a> (1972). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2537',
    'Webster et&nbsp;al. (1985)',
    'Webster, G. R. B., Friesen, K. J., Sarna, L. P., &#x0026; Muir, D. C. G.: <span class="cmti-10">Environmental</span> <span class="cmti-10">fate modelling of chlorodioxins: Determination of physical constants</span>, Chemosphere, 14, 609–622, <a href="https://doi.org/10.1016/0045-6535(85)90169-9" >doi:10.1016/0045-6535(85)90169-9</a> (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '205',
    'Weinstein-Lloyd and Schwartz (1991)',
    'Weinstein-Lloyd, J. &#x0026; Schwartz, S. E.: <span class="cmti-10">Low-intensity radiolysis study of free-radical reactions in cloudwater:</span> H<sub><span class="cmr-7">2</span></sub>O<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">production and destruction</span>, Environ. Sci. Technol., 25, 791–800, <a href="https://doi.org/10.1021/ES00016A027" >doi:10.1021/ES00016A027</a> (1991). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3746',
    'Weisenberger and Schumpe (1996)',
    'Weisenberger, S. &#x0026; Schumpe, A.: <span class="cmti-10">Estimation of gas solubilities in salt</span> <span class="cmti-10">solutions at temperatures from 273K to 363K</span>, AIChE J., 42, 298–300, <a href="https://doi.org/10.1002/AIC.690420130" >doi:10.1002/AIC.690420130</a> (1996). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1779',
    'Weiss (1974)',
    'Weiss, R. F.: <span class="cmti-10">Carbon dioxide in water and seawater:</span> <span class="cmti-10">The solubility of a non-ideal gas</span>, Mar. Chem., 2, 203–215, <a href="https://doi.org/10.1016/0304-4203(74)90015-2" >doi:10.1016/0304-4203(74)90015-2</a> (1974). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2832',
    'Weiss and Price (1980)',
    'Weiss, R. F. &#x0026; Price, B. A.: <span class="cmti-10">Nitrous oxide solubility in water and seawater</span>, Mar. Chem., 8, 347–359, <a href="https://doi.org/10.1016/0304-4203(80)90024-9" >doi:10.1016/0304-4203(80)90024-9</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3646',
    'Welke et&nbsp;al. (1998)',
    'Welke, B., Ettlinger, K., &#x0026; Riederer, M.: <span class="cmti-10">Sorption of volatile organic chemicals in plant surfaces</span>, Environ. Sci. Technol., 32, 1099–1104, <a href="https://doi.org/10.1021/ES970763V" >doi:10.1021/ES970763V</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2484',
    'Wen and Muccitelli (1979)',
    'Wen, W.-Y. &#x0026; Muccitelli, J. A.: <span class="cmti-10">Thermodynamics of some perfluorocarbon gases in</span> <span class="cmti-10">water</span>, J. Solution Chem., 8, 225–246, <a href="https://doi.org/10.1007/BF00648882" >doi:10.1007/BF00648882</a> (1979). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '849',
    'Westcott et&nbsp;al. (1981)',
    'Westcott, J. W., Simon, C. G., &#x0026; Bidleman, T. F.: <span class="cmti-10">Determination of polychlorinated</span> <span class="cmti-10">biphenyl vapor pressures by a semimicro gas saturation method</span>, Environ. Sci. Technol., 15, 1375–1378, <a href="https://doi.org/10.1021/ES00093A012" >doi:10.1021/ES00093A012</a> (1981). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '776',
    'Westheimer and Ingraham (1956)',
    'Westheimer, F. H. &#x0026; Ingraham, L. L.: <span class="cmti-10">The entropy of chelation</span>, J. Phys. Chem., 60, 1668–1670, <a href="https://doi.org/10.1021/J150546A024" >doi:10.1021/J150546A024</a> (1956). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3799',
    'Wetlaufer et&nbsp;al. (1964)',
    'Wetlaufer, D. B., Malik, S. K., Stoller, L., &#x0026; Coffin, R. L.: <span class="cmti-10">Nonpolar group</span> <span class="cmti-10">participation in the denaturation of proteins by urea and guanidinium</span> <span class="cmti-10">salts. Model compound studies</span>, J. Am. Chem. Soc., 86, 508–514, <a href="https://doi.org/10.1021/JA01057A045" >doi:10.1021/JA01057A045</a> (1964). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3541',
    'Whitney and Vivian (1941a)',
    'Whitney, R. P. &#x0026; Vivian, J. E.: <span class="cmti-10">Solubility of chlorine in water</span>, Ind. Eng. Chem., 33, 741–744, <a href="https://doi.org/10.1021/IE50378A014" >doi:10.1021/IE50378A014</a> (1941a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3928',
    'Whitney and Vivian (1941b)',
    'Whitney, R. P. &#x0026; Vivian, J. E.: <span class="cmti-10">Solubility of</span> <span class="cmti-10">chlorine in water</span>, Pap. Trade J., 113, 31–32 (1941b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3070',
    'WHO (1990)',
    'WHO: <span class="cmti-10">Environmental Health Criteria 101 – methylmercury</span>, Tech. rep., World Health Organization, URL https://www.inchem.org/documents/ehc/ehc/ehc101.htm (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3215',
    'Wieland et&nbsp;al. (2015)',
    'Wieland, F., Neff, A., Gloess, A. N., Poisson, L., Atlan, S., Larrain, D., Prêtre, D., Blank, I., &#x0026; Yeretzian, C.: <span class="cmti-10">Temperature dependence of</span> <span class="cmti-10">Henry&#x2019;s law constants: An automated, high-throughput gas stripping cell</span> <span class="cmti-10">design coupled to PTR-ToF-MS</span>, Int. J. Mass Spectrom., 387, 69–77, <a href="https://doi.org/10.1016/J.IJMS.2015.07.015" >doi:10.1016/J.IJMS.2015.07.015</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3765',
    'Wieser et&nbsp;al. (2023)',
    'Wieser, F., Sander, R., &#x0026; Taraborrelli, D.: <span class="cmti-10">Development of a multiphase chemical mechanism to</span> <span class="cmti-10">improve secondary organic aerosol formation</span> <span class="cmti-10">in CAABA/MECCA (version 4.5.6-rc.1)</span>, Geosci. Model Dev. Discuss., <a href="https://doi.org/10.5194/GMD-2023-102" >doi:10.5194/GMD-2023-102</a>, in review (2023). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '190',
    'Wilhelm et&nbsp;al. (1977)',
    'Wilhelm, E., Battino, R., &#x0026; Wilcock, R. J.: <span class="cmti-10">Low-pressure solubility of gases in liquid water</span>, Chem. Rev., 77, 219–262, <a href="https://doi.org/10.1021/CR60306A003" >doi:10.1021/CR60306A003</a> (1977). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3271',
    'Willey et&nbsp;al. (2017)',
    'Willey, J. D., Powell, J. P., Avery, G. B., Kieber, R. J., &#x0026; Mead, R. N.: <span class="cmti-10">Use of experimentally determined Henry&#x2019;s Law and salting-out</span> <span class="cmti-10">constants for ethanol in seawater for determination of the saturation</span> <span class="cmti-10">state of ethanol in coastal waters</span>, Chemosphere, 182, 426–432, <a href="https://doi.org/10.1016/J.CHEMOSPHERE.2017.05.044" >doi:10.1016/J.CHEMOSPHERE.2017.05.044</a> (2017). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '160',
    'Winiwarter et&nbsp;al. (1988)',
    'Winiwarter, W., Puxbaum, H., Fuzzi, S., Facchini, M. C., Orsi, G., Beltz, N., Enderle, K.-H., &#x0026; Jaeschke, W.: <span class="cmti-10">Organic acid gas and liquid-phase</span> <span class="cmti-10">measurements in Po valley fall-winter conditions in the presence of fog</span>, Tellus, 40B, 348–357, <a href="https://doi.org/10.1111/J.1600-0889.1988.TB00109.X" >doi:10.1111/J.1600-0889.1988.TB00109.X</a> (1988). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3718',
    'Winkler (1907)',
    'Winkler, L.: <span class="cmti-10">A g</span><span class="cmti-10">ázok oldhat</span><span class="cmti-10">ós</span><span class="cmti-10">ága vzben (Solubility of gases</span> <span class="cmti-10">in water)</span>, Math. Termész. Értesitö, 25, 86–108 (1907). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3026',
    'Winkler (1891a)',
    'Winkler, L. W.: <span class="cmti-10">Die</span> <span class="cmti-10">L</span><span class="cmti-10">öslichkeit der Gase in Wasser (erste Abhandlung)</span>, Ber. Dtsch. Chem. Ges., 24, 89–101, <a href="https://doi.org/10.1002/CBER.18910240116" >doi:10.1002/CBER.18910240116</a> (1891a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3027',
    'Winkler (1891b)',
    'Winkler, L. W.: <span class="cmti-10">Die L</span><span class="cmti-10">öslichkeit der Gase in Wasser</span> <span class="cmti-10">(zweite Abhandlung)</span>, Ber. Dtsch. Chem. Ges., 24, 3602–3610, <a href="https://doi.org/10.1002/CBER.189102402237" >doi:10.1002/CBER.189102402237</a> (1891b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '871',
    'Winkler (1899)',
    'Winkler, L. W.: <span class="cmti-10">L</span><span class="cmti-10">öslichkeit des Broms in Wasser</span>, Chem. Ztg., 23, 687–689 (1899). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2834',
    'Winkler (1901)',
    'Winkler, L. W.: <span class="cmti-10">Die L</span><span class="cmti-10">öslichkeit der Gase in Wasser</span> <span class="cmti-10">(dritte Abhandlung)</span>, Ber. Dtsch. Chem. Ges., 34, 1408–1422, <a href="https://doi.org/10.1002/CBER.19010340210" >doi:10.1002/CBER.19010340210</a> (1901). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '862',
    'Winkler (1906)',
    'Winkler, L. W.: <span class="cmti-10">Gesetzm</span><span class="cmti-10">ässigkeit bei</span> <span class="cmti-10">der Absorption der Gase in Fl</span><span class="cmti-10">üssigkeiten</span>, Z. Phys. Chem., 55, 344–354, <a href="https://doi.org/10.1515/ZPCH-1906-5518" >doi:10.1515/ZPCH-1906-5518</a> (1906). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2830',
    'Wisegarver and Cline (1985)',
    'Wisegarver, D. P. &#x0026; Cline, J. D.: <span class="cmti-10">Solubility</span> <span class="cmti-10">of trichlorofluoromethane (F-11) and dichlorodifluoromethane (F-12) in</span> <span class="cmti-10">seawater and its relationship to surface concentrations in the North</span> <span class="cmti-10">Pacific</span>, Deep-Sea Res. A, 32, 97–106 (1985). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3636',
    'Wittig et&nbsp;al. (2001)',
    'Wittig, R., Lohmann, J., Joh, R., Horstmann, S., &#x0026; Gmehling, J.: <span class="cmti-10">Vapor-liquid equilibria and enthalpies of mixing in a temperature</span> <span class="cmti-10">range from 298.15 to 413.15K for the further development of</span> <span class="cmti-10">modified UNIFAC (Dortmund)</span>, Ind. Eng. Chem. Res., 40, 5831–5838, <a href="https://doi.org/10.1021/IE010444J" >doi:10.1021/IE010444J</a> (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2956',
    'Wolfe et&nbsp;al. (1980)',
    'Wolfe, N. L., Burns, L. A., &#x0026; Steen, W. C.: <span class="cmti-10">Use of linear free energy</span> <span class="cmti-10">relationships and an evaluative model to assess the fate and transport of</span> <span class="cmti-10">phthalate esters in the aquatic environment</span>, Chemosphere, 9, 393–402, <a href="https://doi.org/10.1016/0045-6535(80)90022-3" >doi:10.1016/0045-6535(80)90022-3</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2529',
    'Wolfe et&nbsp;al. (1982)',
    'Wolfe, N. L., Zepp, R. G., Schlotzhauer, P., &#x0026; Sink, M.: <span class="cmti-10">Transformation pathways of</span> <span class="cmti-10">hexachlorocyclopentadiene in the aquatic environment</span>, Chemosphere, 11, 91–101, <a href="https://doi.org/10.1016/0045-6535(82)90160-6" >doi:10.1016/0045-6535(82)90160-6</a> (1982). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2379',
    'Wolfenden (1976)',
    'Wolfenden, R.: <span class="cmti-10">Free energies of hydration</span> <span class="cmti-10">and hydrolysis of gaseous acetamide</span>, J. Am. Chem. Soc., 98, 1987–1988, <a href="https://doi.org/10.1021/JA00423A068" >doi:10.1021/JA00423A068</a> (1976). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3084',
    'Wolfenden and Williams (1983)',
    'Wolfenden, R. &#x0026; Williams, R.: <span class="cmti-10">Affinities of</span> <span class="cmti-10">phosphoric acids, esters, and amides for solvent water</span>, J. Am. Chem. Soc., 105, 1028–1031, <a href="https://doi.org/10.1021/JA00342A063" >doi:10.1021/JA00342A063</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '983',
    'Wong and Wang (1997)',
    'Wong, P. K. &#x0026; Wang, Y. H.: <span class="cmti-10">Determination of the</span> <span class="cmti-10">Henry&#x2019;s law constant for dimethyl sulfide in seawater</span>, Chemosphere, 35, 535–544, <a href="https://doi.org/10.1016/S0045-6535(97)00118-5" >doi:10.1016/S0045-6535(97)00118-5</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3581',
    'Woo and McNeill (2015)',
    'Woo, J. L. &#x0026; McNeill, V. F.: <span class="cmti-10">simpleGAMMA v1.0</span> <span class="cmti-10">– a reduced model of secondary organic aerosol formation in the</span> <span class="cmti-10">aqueous aerosol phase (aaSOA)</span>, Geosci. Model Dev., 8, 1821–1829, <a href="https://doi.org/10.5194/GMD-8-1821-2015" >doi:10.5194/GMD-8-1821-2015</a> (2015). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3090',
    'Woodrow et&nbsp;al. (1990)',
    'Woodrow, J. E., McChesney, M. M., &#x0026; Seiber, J. N.: <span class="cmti-10">Modeling the volatilization</span> <span class="cmti-10">of pesticides and their distribution in the atmosphere</span>, in: Long Range Transport of Pesticides, edited by Kurtz, D. A., pp. 61–81, CRC Press (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2804',
    'Worthington and Wade (2007)',
    'Worthington, E. K. &#x0026; Wade, E. A.: <span class="cmti-10">Henry&#x2019;s</span> <span class="cmti-10">Law coefficients of chloropicrin and methyl isothiocyanate</span>, Atmos. Environ., 41, 5510–5515, <a href="https://doi.org/10.1016/J.ATMOSENV.2007.02.019" >doi:10.1016/J.ATMOSENV.2007.02.019</a> (2007). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '855',
    'Wright et&nbsp;al. (1992)',
    'Wright, D. A., Sandler, S. I., &#x0026; DeVoll, D.: <span class="cmti-10">Infinite dilution activity coefficients and solubilities</span> <span class="cmti-10">of halogenated hydrocarbons in water at ambient temperatures</span>, Environ. Sci. Technol., 26, 1828–1831, <a href="https://doi.org/10.1021/ES00033A018" >doi:10.1021/ES00033A018</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3719',
    'Wu et&nbsp;al. (2022a)',
    'Wu, S., Hayati, S. K., Kim, E., de la Mata, A. P., Harynuk, J. J., Wang, C., &#x0026; Zhao, R.: <span class="cmti-10">Henry&#x2019;s law constants and indoor partitioning</span> <span class="cmti-10">of microbial volatile organic compounds</span>, Environ. Sci. Technol., 56, 7143–7152, <a href="https://doi.org/10.1021/ACS.EST.1C07882" >doi:10.1021/ACS.EST.1C07882</a> (2022a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3776',
    'Wu et&nbsp;al. (2022b)',
    'Wu, S., Kim, E., Vethanayagam, D., &#x0026; Zhao, R.: <span class="cmti-10">Indoor partitioning and potential</span> <span class="cmti-10">thirdhand exposure to carbonyl flavoring agents added in e-cigarettes</span> <span class="cmti-10">and hookah tobacco</span>, Environ. Sci. Processes Impacts, 24, 2294–2309, <a href="https://doi.org/10.1039/D2EM00365A" >doi:10.1039/D2EM00365A</a> (2022b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3013',
    'Wu and Chang (2011)',
    'Wu, Y. &#x0026; Chang, V. W.-C.: <span class="cmti-10">The effect of surface</span> <span class="cmti-10">adsorption and molecular geometry on the determination of Henry&#x2019;s law</span> <span class="cmti-10">constants for fluorotelomer alcohols</span>, J. Chem. Eng. Data, 56, 3442–3448, <a href="https://doi.org/10.1021/JE200466W" >doi:10.1021/JE200466W</a> (2011). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2978',
    'Xiao et&nbsp;al. (2004)',
    'Xiao, H., Li, N., &#x0026; Wania, F.: <span class="cmti-10">Compilation, evaluation, and selection of physical-chemical property data</span> <span class="cmti-10">for </span><span class="cmmi-10">α</span><span class="cmti-10">-, </span><span class="cmmi-10">β</span><span class="cmti-10">-, and </span><span class="cmmi-10">γ</span><span class="cmti-10">-hexachlorocyclohexane</span>, J. Chem. Eng. Data, 49, 173–185, <a href="https://doi.org/10.1021/JE034214I" >doi:10.1021/JE034214I</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2990',
    'Xiao et&nbsp;al. (2012)',
    'Xiao, H., Shen, L., Su, Y., Barresi, E., DeJong, M., Hung, H., Lei, Y.-D., Wania, F., Reiner, E. J., Sverko, E., &#x0026; Kang, S.-C.: <span class="cmti-10">Atmospheric</span> <span class="cmti-10">concentrations of halogenated flame retardants at two remote locations:</span> <span class="cmti-10">The Canadian High Arctic and the Tibetan Plateau</span>, Environ. Pollut., 161, 154–161, <a href="https://doi.org/10.1016/J.ENVPOL.2011.09.041" >doi:10.1016/J.ENVPOL.2011.09.041</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1942',
    'Xie et&nbsp;al. (2004)',
    'Xie, Z., Le Calvé, S., Feigenbrugel, V., Preuß, T. G., Vinken, R., Ebinghaus, R., &#x0026; Ruck, W.: <span class="cmti-10">Henry&#x2019;s law constants measurements</span> <span class="cmti-10">of the nonylphenol isomer 4(3&#x2019;,5&#x2019;-dimethyl-3&#x2019;-heptyl)-phenol, tertiary</span> <span class="cmti-10">octylphenol and </span><span class="cmmi-10">γ</span><span class="cmti-10">-hexachlorocyclohexane between 278 and 298 </span>K, Atmos. Environ., 38, 4859–4868, <a href="https://doi.org/10.1016/J.ATMOSENV.2004.05.013" >doi:10.1016/J.ATMOSENV.2004.05.013</a> (2004). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3096',
    'Xu and Kropscott (2012)',
    'Xu, S. &#x0026; Kropscott, B.: <span class="cmti-10">A method for simultaneous determination of partition coefficients for</span> <span class="cmti-10">cyclic volatile methylsiloxanes and dimethylsilanediol</span>, Anal. Chem., 84, 1948–1955, <a href="https://doi.org/10.1021/AC202953T" >doi:10.1021/AC202953T</a> (2012). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3094',
    'Xu and Kropscott (2014)',
    'Xu, S. &#x0026; Kropscott, B.: <span class="cmti-10">Evaluation of the</span> <span class="cmti-10">three-phase equilibrium method for measuring temperature dependence of</span> <span class="cmti-10">internally consistent partition coefficients (</span><span class="cmmi-10">K</span><sub><span class="cmr-7">OW</span></sub><span class="cmti-10">, </span><span class="cmmi-10">K</span><sub><span class="cmr-7">OA</span></sub><span class="cmti-10">, and </span><span class="cmmi-10">K</span><sub><span class="cmr-7">AW</span></sub><span class="cmti-10">) for</span> <span class="cmti-10">volatile methylsiloxanes and trimethylsilanol</span>, Environ. Toxicol. Chem., 33, 2702–2710, <a href="https://doi.org/10.1002/ETC.2754" >doi:10.1002/ETC.2754</a> (2014). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2535',
    'Yaffe et&nbsp;al. (2003)',
    'Yaffe, D., Cohen, Y., Espinosa, G., Arenas, A., &#x0026; Giralt, F.: <span class="cmti-10">A fuzzy</span> <span class="cmti-10">ARTMAP-based quantitative structure-property relationship (QSPR) for</span> <span class="cmti-10">the Henry&#x2019;s law constant of organic compounds</span>, J. Chem. Inf. Comput. Sci., 43, 85–112, <a href="https://doi.org/10.1021/CI025561J" >doi:10.1021/CI025561J</a> (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3937',
    'Yakovkin (1900)',
    'Yakovkin, A. A.: <span class="cmti-10">About the hydrolysis of chlorine</span>, J. Russ. Phys. Chem. Soc., 32, 673–721, (in Russian) (1900). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3731',
    'Yao et&nbsp;al. (2002)',
    'Yao, X., aand X. Zhang, M. L., Hu, Z., &#x0026; Fan, B.: <span class="cmti-10">Radial basis</span> <span class="cmti-10">function network-based quantitative structure-property relationship for</span> <span class="cmti-10">the prediction of Henry&#x2019;s law constant</span>, Anal. Chim. Acta, 462, 101–117, <a href="https://doi.org/10.1016/S0003-2670(02)00273-8" >doi:10.1016/S0003-2670(02)00273-8</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1330',
    'Yates and Gan (1998)',
    'Yates, S. R. &#x0026; Gan, J. Y.: <span class="cmti-10">Volatility, adsorption, and</span> <span class="cmti-10">degradation of propargyl bromide as a soil fumigant</span>, J. Agric. Food Chem., 46, 755–761, <a href="https://doi.org/10.1021/JF9707849" >doi:10.1021/JF9707849</a> (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2440',
    'Yaws (1999)',
    'Yaws, C. L.: Chemical Properties Handbook, McGraw-Hill, Inc., ISBN 0070734011 (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'yaws2003handbook',
    'Yaws (2003)',
    'Yaws, C. L.: Yaws&#x2019; Handbook of Thermodynamic and Physical Properties of Chemical Compounds, Knovel: Norwich, NY, USA, ISBN 1591244447 (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '789',
    'Yaws and Yang (1992)',
    'Yaws, C. L. &#x0026; Yang, H.-C.: <span class="cmti-10">Henry&#x2019;s law constant for</span> <span class="cmti-10">compound in water</span>, in: Thermodynamic and Physical Property Data, edited by Yaws, C. L., pp. 181–206, Gulf Publishing Company, Houston, TX, ISBN 0884150313 (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1115',
    'Yaws et&nbsp;al. (1997)',
    'Yaws, C. L., Hopper, J. R., Sheth, S. D., Han, M., &#x0026; Pike, R. W.: <span class="cmti-10">Solubility and</span> <span class="cmti-10">Henry&#x2019;s law constant for alcohols in water</span>, Waste Manage., 17, 541–547, <a href="https://doi.org/10.1016/S0956-053X(97)10057-5" >doi:10.1016/S0956-053X(97)10057-5</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3091',
    'Yaws et&nbsp;al. (1998)',
    'Yaws, C. L., Sheth, S. D., &#x0026; Han, M.: <span class="cmti-10">Using solubility and Henry&#x2019;s law constant data for ketones in water</span>, Pollut. Eng., 30, 44–46 (1998). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3811',
    'Yaws et&nbsp;al. (1999)',
    'Yaws, C. L., Hopper, J. R., Wang, X., Rathinsamy, A. K., &#x0026; Pike, R. W.: <span class="cmti-10">Calculating solubility &#x0026; Henry&#x2019;s law constants for gases in water</span>, Chem. Eng., pp. 102–105 (1999). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'yaws2001',
    'Yaws et&nbsp;al. (2001)',
    'Yaws, C. L., Hopper, J. R., Mishra, S. R., &#x0026; Pike, R. W.: <span class="cmti-10">Solubility and Henry&#x2019;s law</span> <span class="cmti-10">constants for amines in water</span>, Chem. Eng., 108, 84–88 (2001). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3812',
    'Yaws et&nbsp;al. (2003)',
    'Yaws, C. L., Bajaj, P., Singh, H., &#x0026; Pike, R. W.: <span class="cmti-10">Solubility &#x0026; Henry&#x2019;s law constants for sulfur</span> <span class="cmti-10">compounds in water</span>, Chem. Eng., pp. 60–64 (2003). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    'yaws2005',
    'Yaws et&nbsp;al. (2005)',
    'Yaws, C. L., Narasimhan, P. K., Lou, H. H., &#x0026; Pike, R. W.: <span class="cmti-10">Solubility &#x0026; Henry&#x2019;s law</span> <span class="cmti-10">constants for chlorinated compounds in water</span>, Chem. Eng., 112, 50–56 (2005). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '852',
    'Yin and Hassett (1986)',
    'Yin, C. &#x0026; Hassett, J. P.: <span class="cmti-10">Gas-partitioning approach</span> <span class="cmti-10">for laboratory and fiels studies of mirex fugacity in water</span>, Environ. Sci. Technol., 20, 1213–1217, <a href="https://doi.org/10.1021/ES00154A003" >doi:10.1021/ES00154A003</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3085',
    'Yoo et&nbsp;al. (1986)',
    'Yoo, K.-P., Lee, S. Y., &#x0026; Lee, W. H.: <span class="cmti-10">Ionization and Henry&#x2019;s law constants for volatile, weak electrolyte water</span> <span class="cmti-10">pollutants</span>, Korean J. Chem. Eng., 3, 67–72, <a href="https://doi.org/10.1007/BF02697525" >doi:10.1007/BF02697525</a> (1986). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2971',
    'Yoshida et&nbsp;al. (1983)',
    'Yoshida, K., Shigeoka, T., &#x0026; Yamauchi, F.: <span class="cmti-10">Non-steady state equilibrium model for</span> <span class="cmti-10">the preliminary prediction of the fate of chemicals in the environment</span>, Ecotoxicol. Environ. Saf., 7, 179–190, <a href="https://doi.org/10.1016/0147-6513(83)90064-7" >doi:10.1016/0147-6513(83)90064-7</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2292',
    'Yoshida et&nbsp;al. (1987)',
    'Yoshida, K., Shigeoka, T., &#x0026; Yamauchi, F.: <span class="cmti-10">Evaluation of aquatic environmental</span> <span class="cmti-10">fate of 2,4,6-trichlorophenol with a mathematical model</span>, Chemosphere, 16, 2531–2544, <a href="https://doi.org/10.1016/0045-6535(87)90311-0" >doi:10.1016/0045-6535(87)90311-0</a> (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '527',
    'Yoshizumi et&nbsp;al. (1984)',
    'Yoshizumi, K., Aoki, K., Nouchi, I., Okita, T., Kobayashi, T., Kamakura, S., &#x0026; Tajima, M.: <span class="cmti-10">Measurements of the concentration in rainwater and</span> <span class="cmti-10">of the Henry&#x2019;s law constant of hydrogen peroxide</span>, Atmos. Environ., 18, 395–401, <a href="https://doi.org/10.1016/0004-6981(84)90114-8" >doi:10.1016/0004-6981(84)90114-8</a> (1984). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3852',
    'Young (1981a)',
    'Young, C. L.: IUPAC Solubility Data Series, Volume 5/6, Hydrogen and Deuterium, Pergamon Press, Oxford, ISBN 0080239277 (1981a). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3854',
    'Young (1981b)',
    'Young, C. L.: IUPAC Solubility Data Series, Volume 8, Oxides of Nitrogen, Pergamon Press, Oxford, <a href="https://doi.org/10.1016/C2009-0-00222-0" >doi:10.1016/C2009-0-00222-0</a> (1981b). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3857',
    'Young (1983)',
    'Young, C. L.: IUPAC Solubility Data Series, Volume 12, Sulfur Dioxide, Chlorine, Fluorine and Chlorine Oxides, Pergamon Press, Oxford, <a href="https://doi.org/10.1016/C2013-0-03419-6" >doi:10.1016/C2013-0-03419-6</a> (1983). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3675',
    'Yu (1992)',
    'Yu, H.-Z.: The use of Henry&#x2019;s law constants in the determination of factors that influence VOC concentration in aqueous and gaseous phases in wastewater treatment plant, Master&#x2019;s thesis, New Jersey Institute of Technology, USA (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2479',
    'Yurteri et&nbsp;al. (1987)',
    'Yurteri, C., Ryan, D. F., Callow, J. J., &#x0026; Gurol, M. D.: <span class="cmti-10">The effect of chemical</span> <span class="cmti-10">composition of water on Henry&#x2019;s law constant</span>, J. Water Pollut. Control Fed., 59, 950–956 (1987). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '559',
    'Zafiriou and McFarland (1980)',
    'Zafiriou, O. C. &#x0026; McFarland, M.: <span class="cmti-10">Determination of trace levels of nitric oxide in aqueous solution</span>, Anal. Chem., 52, 1662–1667, <a href="https://doi.org/10.1021/AC50061A029" >doi:10.1021/AC50061A029</a> (1980). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2940',
    'Zhang et&nbsp;al. (2002)',
    'Zhang, S. B. L., Wang, S., &#x0026; Franzblau, A.: <span class="cmti-10">Partition coefficients for the trihalomethanes among</span> <span class="cmti-10">blood, urine, water, milk and air</span>, Sci. Total Environ., 284, 237–247, <a href="https://doi.org/10.1016/S0048-9697(01)00890-7" >doi:10.1016/S0048-9697(01)00890-7</a> (2002). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2921',
    'Zhang et&nbsp;al. (2013)',
    'Zhang, W., Huang, L., Yang, C., &#x0026; Ying, W.: <span class="cmti-10">Experimental method for estimating Henry&#x2019;s law</span> <span class="cmti-10">constant of volatile organic compound</span>, Asian J. Chem., 25, 2647–2650, <a href="https://doi.org/10.14233/AJCHEM.2013.13584" >doi:10.14233/AJCHEM.2013.13584</a> (2013). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2995',
    'Zhang et&nbsp;al. (2010)',
    'Zhang, X., Brown, T. N., Wania, F., Heimstad, E. S., &#x0026; Goss, K.-U.: <span class="cmti-10">Assessment of chemical screening outcomes based on different</span> <span class="cmti-10">partitioning property estimation methods</span>, Environ. Int., 36, 514–520, <a href="https://doi.org/10.1016/J.ENVINT.2010.03.010" >doi:10.1016/J.ENVINT.2010.03.010</a> (2010). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '2937',
    'Zhang and Pawliszyn (1993)',
    'Zhang, Z. &#x0026; Pawliszyn, J.: <span class="cmti-10">Headspace solid-phase microextraction</span>, Anal. Chem., 65, 1843–1852, <a href="https://doi.org/10.1021/AC00062A008" >doi:10.1021/AC00062A008</a> (1993). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '921',
    'Zheng et&nbsp;al. (1997)',
    'Zheng, D.-Q., Guo, T.-M., &#x0026; Knapp, H.: <span class="cmti-10">Experimental and modeling studies on the solubility of</span> CO<sub><span class="cmr-7">2</span></sub><span class="cmti-10">,</span> CHClF<sub><span class="cmr-7">2</span></sub><span class="cmti-10">,</span> CHF<sub><span class="cmr-7">3</span></sub><span class="cmti-10">,</span> C<sub><span class="cmr-7">2</span></sub>H<sub><span class="cmr-7">2</span></sub>F<sub><span class="cmr-7">4</span></sub> <span class="cmti-10">and</span> C<sub><span class="cmr-7">2</span></sub>H<sub><span class="cmr-7">4</span></sub>F<sub><span class="cmr-7">2</span></sub> <span class="cmti-10">in water and aqueous</span> NaCl <span class="cmti-10">solutions under low pressures</span>, Fluid Phase Equilib., 129, 197–209, <a href="https://doi.org/10.1016/S0378-3812(96)03177-9" >doi:10.1016/S0378-3812(96)03177-9</a> (1997). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '800',
    'Zhou and Lee (1992)',
    'Zhou, X. &#x0026; Lee, Y.-N.: <span class="cmti-10">Aqueous solubility and reaction</span> <span class="cmti-10">kinetics of hydroxymethyl hydroperoxide</span>, J. Phys. Chem., 96, 265–272, <a href="https://doi.org/10.1021/J100180A051" >doi:10.1021/J100180A051</a> (1992). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '630',
    'Zhou and Mopper (1990)',
    'Zhou, X. &#x0026; Mopper, K.: <span class="cmti-10">Apparent partition</span> <span class="cmti-10">coefficients of 15 carbonyl compounds between air and seawater and</span> <span class="cmti-10">between air and freshwater; Implications for air-sea exchange</span>, Environ. Sci. Technol., 24, 1864–1869, <a href="https://doi.org/10.1021/ES00082A013" >doi:10.1021/ES00082A013</a> (1990). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '1574',
    'Zhu et&nbsp;al. (2000)',
    'Zhu, J. Y., Liu, P. H., Chai, X. S., Bullock, K. R., &#x0026; Teja, A. S.: <span class="cmti-10">Henry&#x2019;s law constant of</span> <span class="cmti-10">methanol in pulping spent liquors</span>, Environ. Sci. Technol., 34, 1742–1746, <a href="https://doi.org/10.1021/ES990415O" >doi:10.1021/ES990415O</a> (2000). ');
INSERT INTO literature (bibtexkey, quotation, bibliography) VALUES (
    '3306',
    'Zin et&nbsp;al. (2016)',
    'Zin, R. M., Coquelet, C., Valtz, A., Abdul Mutalib, M. I., &#x0026; Sabil, K. M.: <span class="cmti-10">Measurement of Henry&#x2019;s Law constant and infinite</span> <span class="cmti-10">dilution activity coefficient of isopropyl mercaptan and isobutyl mercaptan</span> <span class="cmti-10">in (methyldiethanolamine (1) + water (2)) with </span><span class="cmmi-10">w</span><sub><span class="cmr-7">1</span></sub> <span class="cmti-10">= 0.25 and 0.50 at</span> <span class="cmti-10">temperature of (298 to 348) K using inert gas stripping method</span>, J. Chem. Thermodyn., 93, 193–199, <a href="https://doi.org/10.1016/J.JCT.2015.10.005" >doi:10.1016/J.JCT.2015.10.005</a> (2016). ');
-- -*- mode: sql; sql-product: postgres; -*-
-- created by henry_psql.f90, do not edit!

INSERT INTO categories (catname) VALUES (
    'Inorganic species');
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Oxygen (O)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'oxygen',
    'O<sub>2</sub>',
    '',
    '7782-44-7',
    'MYMOFIZGZYHOMD-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Oxygen (O)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'moreTdep',
    '', 'A detailed temperature dependence with more than one parameter is available in the original publication. Here, only the temperature dependence at 298.15&nbsp;K according to the van ''t Hoff equation is presented.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3410'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3410_O2',
    '3410,3412', 'Clever et&nbsp;al. (2014) recommend the data from Rettich et&nbsp;al. (2000).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3410_O2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1700',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='2828'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1400',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2105',
    '2555', 'The vapor pressure for water from Wagner and Pruss (1993) was used to calculate H<sub>s</sub>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='2301'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3853'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1400',
    'L',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3431'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3605'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3605-7782-44-7-ABC',
    '3605', 'The data from Millero et&nbsp;al. (2002a) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;130.91491 +6700.12242/T +17.04684&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3605-7782-44-7-ABC'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3605-sameas3606',
    '3606', 'Almost the same data were also published in Millero et&nbsp;al. (2002b).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3605-sameas3606'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3606'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3606-7782-44-7-ABC',
    '3606', 'The data from Millero et&nbsp;al. (2002b) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;118.73105 +6163.97787/T +15.22401&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3606-7782-44-7-ABC'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3606-sameas3605',
    '3605', 'Almost the same data were also published in Millero et&nbsp;al. (2002a).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3606-sameas3605'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3412'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3412-7782-44-7-ABC',
    '3412', 'The data from Rettich et&nbsp;al. (2000) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;179.13831 +8707.17767/T +24.33473&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3412-7782-44-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3603'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3603-7782-44-7-ABC',
    '3603', 'The data from Sherwood et&nbsp;al. (1991) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;197.67462 +9515.09306/T +27.11204&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3603-7782-44-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='491'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0491-7782-44-7-ABC',
    '491', 'The data from Rettich et&nbsp;al. (1981) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;178.21340 +8672.23354/T +24.19307&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0491-7782-44-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3601'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'highTextrapol',
    '', 'Measured at high temperature and extrapolated to T<sup>&ominus;</sup>&nbsp;= 298.15&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='highTextrapol'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3600'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at293K',
    '', 'Value at T = 293&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3628'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='2437'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3608'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at273K',
    '', 'Value at T = 273&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at273K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3787'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at310K',
    '', 'Value at T = 310&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3566'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3566-7782-44-7-ABC',
    '3566', 'The data from Murray and Riley (1969) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;180.22078 +8760.50130/T +24.49289&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3566-7782-44-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1200',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3785'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3785-7782-44-7-ABC',
    '3785', 'The data from Shoor et&nbsp;al. (1969) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;91.44799 +4548.67245/T +11.38821&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3785-7782-44-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='525'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0525-7782-44-7-ABC',
    '525', 'The data from Carpenter (1966) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;130.04464 +6687.45227/T +16.90114&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0525-7782-44-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3783'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3783-7782-44-7-ABC',
    '3783', 'The data from Morrison and Billett (1952) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;167.89288 +8254.02144/T +22.62741&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3783-7782-44-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3702'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3716'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1700',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3711'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3027'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3027-7782-44-7-ABC',
    '3027', 'The data from Winkler (1891b) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;155.30315 +7638.78869/T +20.77945&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3027-7782-44-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3713'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1800',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3712'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1200',
    'M',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3717'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1600',
    'V',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1600',
    'V',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'V',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3705'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3728',
    '', 'Calculated using machine learning matrix completion methods (MCMs).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    '?',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'whichref',
    '', 'Several references are given in the list of Henry''s law constants but not assigned to specific species.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1500',
    '?',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7782-44-7-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;161.84252 +7966.66767/T +21.73409&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7782-44-7-ABC'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '379',
    '3803,379', 'The partial pressure of water vapor (needed to convert some Henry''s law constants) was calculated using the formula given by Buck (1981). The quantities A and &#945; from Dean and Lange (1999) were assumed to be identical.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='oxygen'),
    (SELECT id from literature WHERE bibtexkey='3945'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'atroomT',
    '', 'Value at "room temperature".');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='atroomT'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'ozone',
    'O<sub>3</sub>',
    '',
    '10028-15-6',
    'CBENFWSGALASAD-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Oxygen (O)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;4</sup>',
    '2800',
    'L',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;4</sup>',
    '2800',
    'L',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3410'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3410_O3',
    '3410,3853', 'Clever et&nbsp;al. (2014) recommend the data from Battino (1981).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3410_O3'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;4</sup>',
    '2800',
    'L',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;4</sup>',
    '2800',
    'L',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.6&times;10<sup>&#8722;5</sup>',
    '3700',
    'L',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3589'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;4</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='2432'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3853'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3853_O3',
    '3853,2549', 'Battino (1981) concludes that ozone aqueous chemistry needs further clarification. Data from Roth and Sullivan (1981) are recommended, in spite of limitations and assumptions of the data.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3853_O3'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;4</sup>',
    '2000',
    'L',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3392'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;4</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='2512'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;5</sup>',
    '2600',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3414'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.2&times;10<sup>&#8722;5</sup>',
    '2600',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3590'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3005'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;4</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='480'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3591'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='2549'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2549',
    '2549', 'Roth and Sullivan (1981) found that H<sub>s</sub> depends on the concentration of OH<sup>&#8722;</sup>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2549'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '1900',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3592'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;5</sup>',
    '2600',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3593'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;4</sup>',
    '2000',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='528'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3594'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.0&times;10<sup>&#8722;5</sup>',
    '2900',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3595'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '4100',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3596'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3597'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at291K',
    '', 'Value at T = 291&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at291K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;4</sup>',
    '2600',
    'T',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='87'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='perry73'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'from492',
    '492', 'Value given here as quoted by Durham et&nbsp;al. (1981).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from492'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='477'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '477_O3',
    '477,3853', 'Lide and Frederikse (1995) present an unusually low value for the Henry solubility of ozone. They refer to Battino (1981) as the source, but the quoted value cannot be found there.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='477_O3'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.3&times;10<sup>&#8722;5</sup>',
    '2500',
    '?',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.3&times;10<sup>&#8722;5</sup>',
    '2500',
    '?',
    (SELECT id from species WHERE iupac='ozone'),
    (SELECT id from literature WHERE bibtexkey='103'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Hydrogen (H)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrogen atom',
    'H',
    '',
    '12385-13-6',
    'YZCKVEUIGOORGS-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Hydrogen (H)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>&#8722;6</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='hydrogen atom'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>&#8722;6</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='hydrogen atom'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>&#8722;6</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='hydrogen atom'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>&#8722;6</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='hydrogen atom'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.1&times;10<sup>&#8722;6</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen atom'),
    (SELECT id from literature WHERE bibtexkey='3562'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;6</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='hydrogen atom'),
    (SELECT id from literature WHERE bibtexkey='3544'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3544',
    '3544', 'Parker (1992) assumes that the free energy of solvation of atomic hydrogen is equal to that of He because of a similar van der Waals radius.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3544'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='hydrogen atom'),
    (SELECT id from literature WHERE bibtexkey='3530'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3530',
    '3530', 'Roduner and Bartels (1992) say that the free energy of solvation &#8710;G<sup><span class="roman">H</span></sup><sub><span class="roman">solv</span></sub> (and therefore Henry''s law constant) of atomic hydrogen is approximated well by that of molecular hydrogen. However, they apparently do not give a value for &#8710;G<sup><span class="roman">H</span></sup><sub><span class="roman">solv</span></sub>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3530'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrogen',
    'H<sub>2</sub>',
    '',
    '1333-74-0',
    'UFHFLCQGNIYNRP-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Hydrogen (H)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '530',
    'L',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '490',
    'L',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3852'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '490',
    'L',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '600',
    'L',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3431'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '620',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3777'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'HTdep_r2_lessthan_0.9',
    '', 'Fitting the temperature dependence <span class="roman">d</span>lnH/<span class="roman">d</span>(1/T) produced a low correlation coefficient (r<sup>2</sup> &lt; 0.9). The data should be treated with caution.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HTdep_r2_lessthan_0.9'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3777',
    '3777', 'Data digitized from Figs.&nbsp;2 and 3 in Schmidt (1979).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3777'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '480',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3782'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3782-1333-74-0-ABC',
    '3782', 'The data from Gordon et&nbsp;al. (1977) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;158.95051 +6959.76267/T +21.73478&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3782-1333-74-0-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '520',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3781'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3781-1333-74-0-ABC',
    '3781', 'The data from Crozier and Yamamoto (1974) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;129.44163 +5676.58091/T +17.31002&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3781-1333-74-0-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3786'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3787'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3785'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3785hydrogen',
    '3785', 'The data presented for hydrogen in Table&nbsp;II of Shoor et&nbsp;al. (1969) appear to be incorrect and are not reproduced here.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3785hydrogen'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.5&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3784'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at303K',
    '', 'Value at T = 303&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at303K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '510',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3783'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3783-1333-74-0-ABC',
    '3783', 'The data from Morrison and Billett (1952) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;94.36490 +4110.23880/T +12.07743&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3783-1333-74-0-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '540',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3711'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3795'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3795-1333-74-0-ABC',
    '3795', 'The data from Braun (1900) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 171.59451 &#8722;6856.02728/T &#8722;28.14739&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3795-1333-74-0-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '500',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3026'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3026-1333-74-0-ABC',
    '3026', 'The data from Winkler (1891a) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;103.47250 +4506.63123/T +13.44160&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3026-1333-74-0-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.5&times;10<sup>&#8722;6</sup>',
    '550',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3713'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '610',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3712'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.5&times;10<sup>&#8722;6</sup>',
    '20',
    'M',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3717'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'HTdep_r2_lessthan_0.5',
    '', 'Fitting the temperature dependence <span class="roman">d</span>lnH/<span class="roman">d</span>(1/T) produced a very low correlation coefficient (r<sup>2</sup> &lt; 0.5). The data should be treated with caution.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HTdep_r2_lessthan_0.5'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3717',
    '3717,3927,3930', 'The paper by Bunsen (1855a) was written in German. English versions with the same data were published by Bunsen (1855b) and Bunsen (1855c).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3717'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '500',
    'V',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '',
    'R',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='714'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '490',
    '?',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '500',
    '?',
    (SELECT id from species WHERE iupac='hydrogen'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-1333-74-0-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;98.78036 +4298.15060/T +12.74131&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-1333-74-0-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'deuterium',
    'D<sub>2</sub>',
    '',
    '7782-39-0',
    'UFHFLCQGNIYNRP-VVKOMZTBSA-N',
    (SELECT id from subcategories WHERE subcatname='Hydrogen (H)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='deuterium'),
    (SELECT id from literature WHERE bibtexkey='3852'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3852',
    '3852,3926', 'Young (1981a) recommend the data from Muccitelli and Wen (1978).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3852'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '720',
    'M',
    (SELECT id from species WHERE iupac='deuterium'),
    (SELECT id from literature WHERE bibtexkey='3926'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3926-7782-39-0-ABC',
    '3926', 'The data from Muccitelli and Wen (1978) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;103.11330 +4676.56978/T +13.28348&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3926-7782-39-0-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '720',
    'M',
    (SELECT id from species WHERE iupac='deuterium'),
    (SELECT id from literature WHERE bibtexkey='3926'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.2&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='deuterium'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydroxyl radical',
    'OH',
    '',
    '3352-57-6',
    'TUJKJAMUKRIRHC-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Hydrogen (H)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;1</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3500-E0',
    '', 'The free energy of solution was calculated based on electrochemical reduction potentials and related free energies.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;1</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;1</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;1</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;1</sup>',
    '4300',
    'T',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='515'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.2&times;10<sup>&#8722;1</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='189'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;1</sup>',
    '3100',
    'T',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='763'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '763',
    '', 'Calculated from correlation between the polarizabilities and solubilities of stable gases. The temperature dependence is an estimate of the upper limit.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='763'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;1</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='31'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0',
    '',
    'C',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='31'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.9&times;10<sup>1</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='31'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;1</sup>',
    '5300',
    'C',
    (SELECT id from species WHERE iupac='hydroxyl radical'),
    (SELECT id from literature WHERE bibtexkey='46'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '46OH',
    '46', 'Jacob (1986) assumed the temperature dependence to be the same as for water.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='46OH'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydroperoxy radical',
    'HO<sub>2</sub>',
    '',
    '3170-83-0',
    'MHAJPDPJQMAIIY-UHFFFAOYSA-M',
    (SELECT id from subcategories WHERE subcatname='Hydrogen (H)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.7&times;10<sup>1</sup>',
    '',
    'R',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='797'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.8',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.8',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.8',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.8',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>1</sup>',
    '5900',
    'T',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='515'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.9&times;10<sup>1</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='205'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.9&times;10<sup>1</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='87'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>1</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='209'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '209HO2',
    '209,31,34,46', 'In the abstract, Schwartz (1984) gives a range of 9.9&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>  &lt;  H<sub>s</sub><sup>cp</sup> &lt;  3.0&times;10<sup>1</sup>&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>. The mean value of this range (2.0&times;10<sup>1</sup>&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>) has been used by Lelieveld and Crutzen (1991), Pandis and Seinfeld (1989), and Jacob (1986).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='209HO2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.6&times;10<sup>1</sup>',
    '4800',
    'T',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='763'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='763'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '6600',
    'E',
    (SELECT id from species WHERE iupac='hydroperoxy radical'),
    (SELECT id from literature WHERE bibtexkey='46'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '46HO2',
    '209', 'The value of H<sub>s</sub><sup>&ominus;</sup> was taken from Schwartz (1984).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='46HO2'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrogen peroxide',
    'H<sub>2</sub>O<sub>2</sub>',
    '',
    '7722-84-1',
    'MHAJPDPJQMAIIY-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Hydrogen (H)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.6&times;10<sup>2</sup>',
    '7300',
    'L',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.6&times;10<sup>2</sup>',
    '7300',
    'L',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.5&times;10<sup>2</sup>',
    '7200',
    'L',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='3518'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>2</sup>',
    '6600',
    'L',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='2828'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>2</sup>',
    '7600',
    'L',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.6&times;10<sup>2</sup>',
    '7300',
    'L',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.8&times;10<sup>2</sup>',
    '6100',
    'L',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='1943'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '1943-7722-84-1-erratum',
    '1943', 'Erratum for page 264 of Fogg and Sangster (2003): the second value from their Ref.&nbsp;[10] refers to 291.15&nbsp;K, not 281.15&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1943-7722-84-1-erratum'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>3</sup>',
    '5900',
    'M',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='3578'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>3</sup>',
    '7000',
    'M',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='2827'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.2&times;10<sup>2</sup>',
    '7400',
    'M',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='516'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>2</sup>',
    '6300',
    'M',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='311'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '311',
    '258', 'This value is a correction of the solubility published by Lind and Kok (1986).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='311'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='799'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '799h2o2',
    '311', 'This value was measured at low pH. It is superseded by a later publication of the same group (Lind and Kok, 1994).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='799h2o2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.5&times;10<sup>2</sup>',
    '6500',
    'M',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='800'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.7&times;10<sup>2</sup>',
    '7900',
    'M',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='488'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>3</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='527'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.6&times;10<sup>2</sup>',
    '6600',
    'T',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='87'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>2</sup>',
    '7000',
    'T',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='43'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>1</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='2521'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>2</sup>',
    '7300',
    '?',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>2</sup>',
    '7300',
    '?',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='103'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hydrogen peroxide'),
    (SELECT id from literature WHERE bibtexkey='34'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '34H2O2',
    '34,258,311', 'Pandis and Seinfeld (1989) cite an incorrect value from Lind and Kok (1986); see erratum by Lind and Kok (1994).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='34H2O2'));
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Nitrogen (N)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitrogen',
    'N<sub>2</sub>',
    '',
    '7727-37-9',
    'IJGRMHOSHXDMSA-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1300',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1300',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='2828'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1300',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1300',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5&times;10<sup>&#8722;6</sup>',
    '1200',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5&times;10<sup>&#8722;6</sup>',
    '1200',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='2433'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5&times;10<sup>&#8722;6</sup>',
    '1200',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3856'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1300',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.4&times;10<sup>&#8722;6</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='2441'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5&times;10<sup>&#8722;6</sup>',
    '1400',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3696'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.3&times;10<sup>&#8722;6</sup>',
    '1300',
    'L',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3431'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1300',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='2434'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2434-7727-37-9-ABC',
    '2434', 'The data from Rettich et&nbsp;al. (1984) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;187.67954 +8903.42524/T +25.60079&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2434-7727-37-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.6&times;10<sup>&#8722;6</sup>',
    '1300',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3601'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='highTextrapol'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.5&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3787'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1300',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3565'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3565-7727-37-9-ABC',
    '3565', 'The data from Murray et&nbsp;al. (1969) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;174.95275 +8370.22025/T +23.67878&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3565-7727-37-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5&times;10<sup>&#8722;6</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3783'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3783-7727-37-9-ABC',
    '3783', 'The data from Morrison and Billett (1952) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;193.68175 +9249.63150/T +26.45117&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3783-7727-37-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.6&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3702'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5&times;10<sup>&#8722;6</sup>',
    '1100',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3707'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.6&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3715'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at311K',
    '', 'Value at T = 311&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at311K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1200',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3716'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.3&times;10<sup>&#8722;6</sup>',
    '2200',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3795'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3795-7727-37-9-ABC',
    '3795', 'The data from Braun (1900) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 291.66324 &#8722;11637.66767/T &#8722;46.44134&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3795-7727-37-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.3&times;10<sup>&#8722;6</sup>',
    '1300',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3027'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3027-7727-37-9-ABC',
    '3027', 'The data from Winkler (1891b) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;164.15156 +7906.86704/T +22.05399&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3027-7727-37-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.7&times;10<sup>&#8722;6</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3713'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.8&times;10<sup>&#8722;6</sup>',
    '1200',
    'M',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3717'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3717'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1300',
    'V',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5&times;10<sup>&#8722;6</sup>',
    '1300',
    'V',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5&times;10<sup>&#8722;6</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3705'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.6&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '1600',
    '?',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3407'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.3&times;10<sup>&#8722;6</sup>',
    '1200',
    '?',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.7&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.3&times;10<sup>&#8722;6</sup>',
    '1300',
    '?',
    (SELECT id from species WHERE iupac='nitrogen'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7727-37-9-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;163.64571 +7887.30480/T +21.97696&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7727-37-9-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'ammonia',
    'NH<sub>3</sub>',
    '',
    '7664-41-7',
    'QGZKDVFQNNGYKY-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.9&times;10<sup>&#8722;1</sup>',
    '4200',
    'L',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.9&times;10<sup>&#8722;1</sup>',
    '4200',
    'L',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.9&times;10<sup>&#8722;1</sup>',
    '4200',
    'L',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.9&times;10<sup>&#8722;1</sup>',
    '4200',
    'L',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.8&times;10<sup>&#8722;1</sup>',
    '4400',
    'L',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='3085'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.0&times;10<sup>&#8722;1</sup>',
    '4200',
    'L',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='540'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;1</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.8&times;10<sup>&#8722;1</sup>',
    '3200',
    'M',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='3003'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9',
    '6600',
    'M',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='3668'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3668',
    '3668', 'Tsuji et&nbsp;al. (1990) provide effective Henry''s law constants at several pH values. Here, only the value at pH = 5.8 is shown for the (acidic) S compounds and the value at pH = 8.6 for the alkaline N compounds.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3668'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.0&times;10<sup>&#8722;1</sup>',
    '4200',
    'M',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='620'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.5&times;10<sup>&#8722;1</sup>',
    '4100',
    'M',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='487'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;1</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='287'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.4&times;10<sup>&#8722;1</sup>',
    '3700',
    'M',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='482'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.6&times;10<sup>&#8722;1</sup>',
    '4200',
    'T',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='487'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.7&times;10<sup>&#8722;1</sup>',
    '4100',
    'T',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='87'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.9&times;10<sup>&#8722;1</sup>',
    '4100',
    'T',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='3540'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.1&times;10<sup>&#8722;1</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='vankrevelen49'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'from510',
    '510', 'Value given here as quoted by Betterton (1992).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from510'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;1</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.7&times;10<sup>&#8722;1</sup>',
    '2400',
    '?',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7664-41-7-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 206.08500 &#8722;7165.18642/T &#8722;32.18383&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7664-41-7-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.7&times;10<sup>&#8722;1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.1&times;10<sup>&#8722;1</sup>',
    '4100',
    '?',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.8&times;10<sup>&#8722;1</sup>',
    '4100',
    '?',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='103'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.2&times;10<sup>&#8722;1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='ammonia'),
    (SELECT id from literature WHERE bibtexkey='641'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '641amine',
    '641,793', 'Bone et&nbsp;al. (1983) give Carter et&nbsp;al. (1968) as the source. However, no data were found in that reference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='641amine'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrazoic acid',
    'HN<sub>3</sub>',
    '',
    '7782-79-8',
    'JUINSXZKUKVTMD-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;1</sup>',
    '3800',
    'L',
    (SELECT id from species WHERE iupac='hydrazoic acid'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2626typo',
    '2626', 'There is a typo in Sander et&nbsp;al. (2011): the value for A should be &#8722;10.19 not 10.19.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2626typo'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.8&times;10<sup>&#8722;2</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='hydrazoic acid'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;1</sup>',
    '3700',
    'M',
    (SELECT id from species WHERE iupac='hydrazoic acid'),
    (SELECT id from literature WHERE bibtexkey='2806'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;2</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='hydrazoic acid'),
    (SELECT id from literature WHERE bibtexkey='2807'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at303K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;1</sup>',
    '4200',
    'M',
    (SELECT id from species WHERE iupac='hydrazoic acid'),
    (SELECT id from literature WHERE bibtexkey='4017'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.6&times;10<sup>&#8722;1</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='hydrazoic acid'),
    (SELECT id from literature WHERE bibtexkey='2530'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2530',
    '2530', 'Modarresi et&nbsp;al. (2007) use different descriptors for their calculations. They conclude that a genetic algorithm/radial basis function network (GA/RBFN) is the best QSPR model. Only these results are shown here.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2530'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hydrazoic acid'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3500-7782-79-8',
    '3500', 'Incorrect data are given by Burkholder et&nbsp;al. (2019) for HN<sub>3</sub>. The correct parameter for the temperature dependence is A = &#8722;10.19 (Robert E.&nbsp;Huie, personal communication, 2021).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-7782-79-8'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hydrazoic acid'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3245-7782-79-8',
    '3245', 'Incorrect data are given by Burkholder et&nbsp;al. (2015) for HN<sub>3</sub>. The correct parameter for the temperature dependence is A = &#8722;10.19 (Robert E.&nbsp;Huie, personal communication, 2021).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3245-7782-79-8'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrazine',
    'H<sub>4</sub>N<sub>2</sub>',
    '',
    '302-01-2',
    'OAKJQQAXSVQMHS-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>1</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='hydrazine'),
    (SELECT id from literature WHERE bibtexkey='3088'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'dinitrogen monoxide',
    'N<sub>2</sub>O',
    'nitrous oxide; laughing gas',
    '10024-97-2',
    'GQPLMRYTRLFLPF-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>&#8722;4</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'seawater',
    '', 'Solubility in sea water.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='seawater'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>&#8722;4</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='seawater'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='2828'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2500',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3854'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;4</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='2441'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;4</sup>',
    '2500',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3696'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2500',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='2832'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2832-10024-97-2-ABC',
    '2832', 'The data from Weiss and Price (1980) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;180.63611 +9824.20147/T +24.46112&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2832-10024-97-2-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;4</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3701'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='2833'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3697'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2400',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3710'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3703'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.2&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3705'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at296K',
    '', 'Value at T = 296&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at296K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3702'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2500',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3709'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2400',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3704'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2700',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3711'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.3&times;10<sup>&#8722;4</sup>',
    '2900',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3944'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3944-10024-97-2-ABC',
    '3944', 'The data from Roth (1897) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;125.17909 +7706.80638/T +15.96486&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3944-10024-97-2-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;4</sup>',
    '2600',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3708'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>&#8722;4</sup>',
    '2500',
    'X',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='gordon1895'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'from3701',
    '3701', 'Value given here as quoted by Gabel and Schultz (1973).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3701'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;4</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='harris51'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'from3703',
    '3703', 'Value given here as quoted by Sy and Hasbrouck (1964).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3703'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.0&times;10<sup>&#8722;4</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='macintosh58'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3703'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;4</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='orcutt37'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at311K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3703'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.8&times;10<sup>&#8722;4</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3705'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.2&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '3600',
    'Q',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '2700',
    '?',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2500',
    '?',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '2800',
    '?',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='dinitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='563'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitrogen monoxide',
    'NO',
    'nitric oxide',
    '10102-43-9',
    'MWUXSHHQAYIFBG-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='2828'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2626-10102-43-9-consicheck',
    '2626', 'The H298 and A, B, C data listed in Table&nbsp;5.4 of Sander et&nbsp;al. (2011) are inconsistent, with 94&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2626-10102-43-9-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '1945-10102-43-9-consicheck',
    '1945', 'The H298 and A, B, C data listed in Table&nbsp;5.4 of Sander et&nbsp;al. (2006) are inconsistent, with 94&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1945-10102-43-9-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='449'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1400',
    'L',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3854'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3854-10102-43-9',
    '3854', 'A minus sign is missing in the fitting parameter presented by Young (1981b). It should be &#8722;62.8086, not 62.8086.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3854-10102-43-9'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='559'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at297K',
    '', 'Value at T = 297&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at297K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.3&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3120'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at288K',
    '', 'Value at T = 288&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='1138'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='2834'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2834-10102-43-9-ABC',
    '2834', 'The data from Winkler (1901) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;184.00012 +8924.34832/T +25.13228&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2834-10102-43-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;5</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1300',
    'C',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='732'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0732-10102-43-9-ABC',
    '732', 'The data from Loomis (1928) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;223.88313 +10620.37030/T +31.13453&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0732-10102-43-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '1500',
    'Q',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '1600',
    '?',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1400',
    '?',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '1500',
    '?',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-10102-43-9-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;160.19223 +7888.02642/T +21.56401&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-10102-43-9-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='1151'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3500-10102-43-9',
    '3500', 'Incorrect data are given by Burkholder et&nbsp;al. (2019) for NO. The correct parameters for the temperature dependence are A = &#8722;163.86, B = 8234, C = 22.816 (Robert E.&nbsp;Huie, personal communication, 2021).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-10102-43-9'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3245-10102-43-9',
    '3245', 'Incorrect data are given by Burkholder et&nbsp;al. (2015) for NO. The correct parameters for the temperature dependence are A = &#8722;163.86, B = 8234, C = 22.816 (Robert E.&nbsp;Huie, personal communication, 2021).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3245-10102-43-9'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='nitrogen monoxide'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '190',
    '190', 'The fitting parameters A, B, C, and D in Table&nbsp;I of Wilhelm et&nbsp;al. (1977) do not reproduce the data in their Table&nbsp;III.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='190'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitrogen dioxide',
    'NO<sub>2</sub>',
    '',
    '10102-44-0',
    'JCXJVPUVTGWSNB-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;5</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='2828'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;4</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='449'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '2400',
    'M',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='1502'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.9&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='59'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at295K',
    '', 'Value at T = 295&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at295K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.3&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='3120'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '2500',
    'T',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='87'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;4</sup>',
    '1800',
    'T',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='763'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='763'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='34'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '34NO2wrongref',
    '34,209', 'Pandis and Seinfeld (1989) refer to Schwartz (1984) as the source, but the quoted value cannot be found there.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='34NO2wrongref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.0&times;10<sup>&#8722;4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitrogen dioxide'),
    (SELECT id from literature WHERE bibtexkey='1151'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitrogen trioxide',
    'NO<sub>3</sub>',
    'nitrate radical',
    '12033-49-7',
    'YPJKMVATUPSWOH-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;2</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='895'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.9&times;10<sup>&#8722;3</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='585'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '585NO3',
    '', 'Value obtained by estimating the diffusion coefficient for NO<sub>3</sub> to be D&nbsp;= 1.0&times;10<sup>&#8722;5</sup>&nbsp;cm<sup>2</sup>/s.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='585NO3'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;4</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;4</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;4</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;4</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;1</sup>',
    '1900',
    'T',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='88'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;4</sup>',
    '2000',
    'T',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='763'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='763'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'E',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='46'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '46NO3',
    '46', 'Jacob (1986) assumes that NO<sub>3</sub> has the same Henry''s law constant as HNO<sub>3</sub>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='46NO3'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='935'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '935',
    '935,34', 'Seinfeld and Pandis (1997) probably refer to the incorrect value given by Pandis and Seinfeld (1989).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='935'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'dinitrogen trioxide',
    'N<sub>2</sub>O<sub>3</sub>',
    '',
    '10544-73-7',
    'LZDSILRDTDCIQT-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.9&times;10<sup>&#8722;3</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='449'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;1</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen trioxide'),
    (SELECT id from literature WHERE bibtexkey='1138'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'dinitrogen tetroxide',
    'N<sub>2</sub>O<sub>4</sub>',
    '',
    '10544-72-6',
    'WFPZPJSADLPSON-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;2</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen tetroxide'),
    (SELECT id from literature WHERE bibtexkey='449'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;2</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen tetroxide'),
    (SELECT id from literature WHERE bibtexkey='3120'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;2</sup>',
    '3500',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen tetroxide'),
    (SELECT id from literature WHERE bibtexkey='1138'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.1&times;10<sup>&#8722;2</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen tetroxide'),
    (SELECT id from literature WHERE bibtexkey='1151'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;2</sup>',
    '1100',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen tetroxide'),
    (SELECT id from literature WHERE bibtexkey='3123'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'dinitrogen pentoxide',
    'N<sub>2</sub>O<sub>5</sub>',
    'nitric anhydride',
    '10102-03-1',
    'ZWWCURLKEXEFQT-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.0&times;10<sup>&#8722;2</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='dinitrogen pentoxide'),
    (SELECT id from literature WHERE bibtexkey='3962'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3962',
    '', 'Calculated from the solvation free energy.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3962'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.9&times;10<sup>&#8722;3</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='dinitrogen pentoxide'),
    (SELECT id from literature WHERE bibtexkey='3963'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3963',
    '', 'Calculated from the solvation free energy.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3963'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.9&times;10<sup>&#8722;3</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='dinitrogen pentoxide'),
    (SELECT id from literature WHERE bibtexkey='3964'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3964',
    '', 'Calculated from the solvation free energy.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3964'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.7&times;10<sup>&#8722;4</sup>',
    '3600',
    'T',
    (SELECT id from species WHERE iupac='dinitrogen pentoxide'),
    (SELECT id from literature WHERE bibtexkey='514'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '514N2O5',
    '', 'This value was extrapolated from data at T = 230&nbsp;K and T = 273&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='514N2O5'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.9&times;10<sup>&#8722;2</sup>',
    '4300',
    'Q',
    (SELECT id from species WHERE iupac='dinitrogen pentoxide'),
    (SELECT id from literature WHERE bibtexkey='3961'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3961_N2O5',
    '3961,449', 'Robinson et&nbsp;al. (1997) applied an empirical correlation between Henry''s law solubilities and boiling points from Schwartz and White (1981).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3961_N2O5'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.9&times;10<sup>&#8722;2</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='dinitrogen pentoxide'),
    (SELECT id from literature WHERE bibtexkey='3436'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3436',
    '449', 'Estimate based on the relation between boiling points and Henry''s law constants for other nitrogen oxides from Schwartz and White (1981).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3436'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='dinitrogen pentoxide'),
    (SELECT id from literature WHERE bibtexkey='271'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'infty',
    '', 'Fast, irreversible hydrolysis is assumed, which is equivalent to an infinite effective Henry''s law constant.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='infty'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='dinitrogen pentoxide'),
    (SELECT id from literature WHERE bibtexkey='46'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='infty'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydroxylamine',
    'H<sub>3</sub>NO',
    '',
    '7803-49-8',
    'AVXURJPOCDRRFD-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>3</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='hydroxylamine'),
    (SELECT id from literature WHERE bibtexkey='3088'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3088-1145',
    '1145', 'Calculated based on the method by Meylan and Howard (1991).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3088-1145'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitrous acid',
    'HNO<sub>2</sub>',
    '',
    '7782-77-6',
    'IOVCWXUNBOPUCH-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.8&times;10<sup>&#8722;1</sup>',
    '4800',
    'L',
    (SELECT id from species WHERE iupac='nitrous acid'),
    (SELECT id from literature WHERE bibtexkey='449'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.7&times;10<sup>&#8722;1</sup>',
    '4900',
    'M',
    (SELECT id from species WHERE iupac='nitrous acid'),
    (SELECT id from literature WHERE bibtexkey='1056'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.7&times;10<sup>&#8722;1</sup>',
    '4900',
    'M',
    (SELECT id from species WHERE iupac='nitrous acid'),
    (SELECT id from literature WHERE bibtexkey='625'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.8&times;10<sup>&#8722;1</sup>',
    '4900',
    'M',
    (SELECT id from species WHERE iupac='nitrous acid'),
    (SELECT id from literature WHERE bibtexkey='592'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.7&times;10<sup>&#8722;1</sup>',
    '9000',
    'M',
    (SELECT id from species WHERE iupac='nitrous acid'),
    (SELECT id from literature WHERE bibtexkey='1138'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.7&times;10<sup>&#8722;1</sup>',
    '4700',
    'T',
    (SELECT id from species WHERE iupac='nitrous acid'),
    (SELECT id from literature WHERE bibtexkey='74'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.8&times;10<sup>&#8722;1</sup>',
    '4800',
    'T',
    (SELECT id from species WHERE iupac='nitrous acid'),
    (SELECT id from literature WHERE bibtexkey='87'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.8&times;10<sup>&#8722;1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitrous acid'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitric acid',
    'HNO<sub>3</sub>',
    '',
    '7697-37-2',
    'GRYLNZFGIOXLOG-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.8&times;10<sup>2</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='492'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>3</sup>',
    '8700',
    'R',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='31'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '31HNO3',
    '31,449', 'Lelieveld and Crutzen (1991) assume the temperature dependence to be the same as for a(H<sup>+</sup>)a(NO<sub>3</sub><sup>&#8722;</sup>)/p(HNO<sub>3</sub>) in Schwartz and White (1981).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='31HNO3'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='2839'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2839-7697-37-2',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 2.6&times;10<sup>7</sup>&times;exp(8700&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2839-7697-37-2'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='530'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '530-7697-37-2',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 2.4&times;10<sup>7</sup>&times;exp(8700&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='530-7697-37-2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>4</sup>',
    '8700',
    'T',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='87'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>3</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='449'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>3</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='34'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '34HNO3',
    '34,209,449', 'Pandis and Seinfeld (1989) refer to Schwartz (1984) as the source, but it is probably from Schwartz and White (1981).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='34HNO3'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>3</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>3</sup>',
    '8800',
    '?',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='103'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='nitric acid'),
    (SELECT id from literature WHERE bibtexkey='260'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '260',
    '530', 'The value is incorrect. See erratum by Brimblecombe and Clegg (1989).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='260'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'pernitric acid',
    'HNO<sub>4</sub>',
    '',
    '26404-66-0',
    'UUZZMWZGAZGXSF-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Nitrogen (N)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.9&times;10<sup>&#8722;1</sup>',
    '8400',
    'L',
    (SELECT id from species WHERE iupac='pernitric acid'),
    (SELECT id from literature WHERE bibtexkey='1182'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.9&times;10<sup>1</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='pernitric acid'),
    (SELECT id from literature WHERE bibtexkey='670'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>2</sup>',
    '6900',
    'T',
    (SELECT id from species WHERE iupac='pernitric acid'),
    (SELECT id from literature WHERE bibtexkey='797'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>2</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='pernitric acid'),
    (SELECT id from literature WHERE bibtexkey='1257'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>2</sup>',
    '0',
    'C',
    (SELECT id from species WHERE iupac='pernitric acid'),
    (SELECT id from literature WHERE bibtexkey='76'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'E',
    (SELECT id from species WHERE iupac='pernitric acid'),
    (SELECT id from literature WHERE bibtexkey='507'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '507HNO4',
    '507', 'Möller and Mauersberger (1992) assumed the solubility of HNO<sub>4</sub> to be comparable to that of HNO<sub>3</sub>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='507HNO4'));
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Fluorine (F)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'fluorine atom',
    'F',
    '',
    '14762-94-8',
    'YCKRFDGAMUMZLT-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Fluorine (F)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;4</sup>',
    '400',
    'T',
    (SELECT id from species WHERE iupac='fluorine atom'),
    (SELECT id from literature WHERE bibtexkey='763'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='763'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrogen fluoride',
    'HF',
    '',
    '7664-39-3',
    'KRHYYFGTRYWZRS-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Fluorine (F)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>2</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='hydrogen fluoride'),
    (SELECT id from literature WHERE bibtexkey='1914'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen fluoride'),
    (SELECT id from literature WHERE bibtexkey='530'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '530-7664-39-3',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 9.4&times;10<sup>1</sup>&times;exp(7400&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='530-7664-39-3'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.8&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='hydrogen fluoride'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hydrogen fluoride'),
    (SELECT id from literature WHERE bibtexkey='260'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='260'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'difluorine monoxide',
    'F<sub>2</sub>O',
    '',
    '7783-41-7',
    'UJMWVICAENGCRF-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Fluorine (F)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;5</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='difluorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='735'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitrogen trifluoride',
    'NF<sub>3</sub>',
    '',
    '7783-54-2',
    'GVGCUCJTUSOZKP-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Fluorine (F)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '1900',
    'L',
    (SELECT id from species WHERE iupac='nitrogen trifluoride'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '1900',
    'L',
    (SELECT id from species WHERE iupac='nitrogen trifluoride'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '1900',
    'L',
    (SELECT id from species WHERE iupac='nitrogen trifluoride'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '1900',
    'L',
    (SELECT id from species WHERE iupac='nitrogen trifluoride'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '1700',
    'M',
    (SELECT id from species WHERE iupac='nitrogen trifluoride'),
    (SELECT id from literature WHERE bibtexkey='3806'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3806-7783-54-2-ABC',
    '3806', 'The data from Dean et&nbsp;al. (1973) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;164.52717 +8214.77776/T +21.97482&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3806-7783-54-2-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '1900',
    'M',
    (SELECT id from species WHERE iupac='nitrogen trifluoride'),
    (SELECT id from literature WHERE bibtexkey='2831'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2831-7783-54-2-ABC',
    '2831', 'The data from Ashton et&nbsp;al. (1968) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;279.21972 +13536.60588/T +38.97386&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2831-7783-54-2-ABC'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'dinitrogen tetrafluoride',
    'N<sub>2</sub>F<sub>4</sub>',
    'tetrafluorohydrazine',
    '10036-47-2',
    'GFADZIUESKAXAK-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Fluorine (F)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '2500',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen tetrafluoride'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '2500',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen tetrafluoride'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '2500',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen tetrafluoride'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '2500',
    'L',
    (SELECT id from species WHERE iupac='dinitrogen tetrafluoride'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '2400',
    'M',
    (SELECT id from species WHERE iupac='dinitrogen tetrafluoride'),
    (SELECT id from literature WHERE bibtexkey='3806'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3806-10036-47-2-ABC',
    '3806', 'The data from Dean et&nbsp;al. (1973) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;318.29953 +15733.17858/T +44.55320&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3806-10036-47-2-ABC'));
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Chlorine (Cl)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'chlorine (molecular)',
    'Cl<sub>2</sub>',
    '',
    '7782-50-5',
    'KZBUYRJDOAKODT-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'HeffCl2',
    '', 'The value presented here appears to be the total solubility of chlorine (i.e., the sum of Cl<sub>2</sub> and HOCl) at a partial pressure of p(Cl<sub>2</sub>)&nbsp; = 101325&nbsp;Pa. This is different from Henry''s law constant, which is defined at extrapolation to infinite dilution.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3857'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3857_Cl2',
    '3857,3925', 'Young (1983) recommends values calculated from Table&nbsp;1 of Adams and Edmonds (1937). Thus, the data refer to effective values that take into account the hydrolysis in the aqueous phase: <br clear="all" /><table border="0" width="100%"><tr><td><table align="center" cellspacing="0"  cellpadding="2"><tr><td nowrap="nowrap" align="center">H<sub><span class="roman">s</span><span class="roman">,</span><span class="roman">eff</span></sub> = ([Cl<sub>2</sub>]+[HOCl])/p(Cl<sub>2</sub>).</td></tr></table></td></tr></table> In addition, the values apply to a partial pressure of p(Cl<sub>2</sub>)&nbsp; = 101325&nbsp;Pa, and not to infinite dilution.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3857_Cl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.1&times;10<sup>&#8722;4</sup>',
    '3200',
    'M',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3535'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.2&times;10<sup>&#8722;4</sup>',
    '3500',
    'M',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3541'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3541',
    '3928', 'The same experimental data were also published by Whitney and Vivian (1941b).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3541'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3934'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3718'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.0&times;10<sup>&#8722;4</sup>',
    '3000',
    'M',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3937'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3937-7782-50-5-ABC',
    '3937', 'The data from Yakovkin (1900) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;122.31264 +7690.40834/T +15.63947&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3937-7782-50-5-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3938'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3935'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3933'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.4&times;10<sup>&#8722;4</sup>',
    '2600',
    'R',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='1230'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.9&times;10<sup>&#8722;4</sup>',
    '',
    'R',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3929'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3929',
    '3929,3925', 'Leaist (1986) converted the total solubility of chlorine in pure water from Adams and Edmonds (1937) to an intrinsic Henry''s law constant.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3929'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '3200',
    'R',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='750'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'R',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3925'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3925',
    '3925,3937,3942', 'Adams and Edmonds (1937) re-analyzed the data from Yakovkin (1900) and Arkadiev (1918), considering deviations from the perfect gas law. They calculated the total solubility of chlorine (i.e., the sum of Cl<sub>2</sub> and HOCl) at several partial pressures of Cl<sub>2</sub>. This is different from Henry''s law constant, which is defined at extrapolation to infinite dilution.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3925'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'R',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3942'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3942',
    '3942,3937', 'Arkadiev (1918) re-analyzed the measurements of Yakovkin (1900). In addition to the data between 15&nbsp;<sup>&#176;</sup>C and 83.4&nbsp;<sup>&#176;</sup>C, he also analyzed the experimental results at 0&nbsp;<sup>&#176;</sup>C and obtained a dimensionless Henry solubility of H<sub>s</sub><sup>cc</sup> = 4.115 at that temperature.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3942'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.1&times;10<sup>&#8722;4</sup>',
    '2800',
    'T',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='489'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.7&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='1038'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '1038tdep',
    '1038', 'The value of &#8710;H<sup>&#176;</sup> listed in Table&nbsp;2 of Bartlett and Margerum (1999) is incorrect. The correct value can be found in the text on page 3411.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1038tdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='chlorine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '190_Cl2',
    '190,3927,3930,3541,3928', 'Wilhelm et&nbsp;al. (1977) present a fitting function for Cl<sub>2</sub> based on four papers which are cited in the footnotes of Table&nbsp;I. However, Bunsen (1855b) and Bunsen (1855c) do not contain any data for Cl<sub>2</sub>, and the data from Whitney and Vivian (1941a) and Whitney and Vivian (1941b) are inconsistent with the fitting function.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='190_Cl2'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HeffCl2'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'chlorine atom',
    'Cl',
    '',
    '22537-15-1',
    'ZAMOUSCENKQFHK-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.3&times;10<sup>&#8722;2</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='chlorine atom'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.3&times;10<sup>&#8722;2</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='chlorine atom'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.3&times;10<sup>&#8722;2</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='chlorine atom'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.3&times;10<sup>&#8722;2</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='chlorine atom'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;3</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='chlorine atom'),
    (SELECT id from literature WHERE bibtexkey='189'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '189ClBr',
    '761', 'Calculated from the free energy of solution by Schwarz and Dodson (1984).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='189ClBr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;4</sup>',
    '1500',
    'T',
    (SELECT id from species WHERE iupac='chlorine atom'),
    (SELECT id from literature WHERE bibtexkey='763'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='763'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrogen chloride',
    'HCl',
    '',
    '7647-01-0',
    'VEXZGXHMUGYJMC-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='1071'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '1071',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 2.0&times;10<sup>7</sup>&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1071'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>1</sup>',
    '',
    'R',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='1137'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='2840'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2840-7647-01-0',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 2.0&times;10<sup>7</sup>&times;exp(9000&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2840-7647-01-0'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='530'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '530-7647-01-0',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 2.0&times;10<sup>7</sup>&times;exp(9000&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='530-7647-01-0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;2</sup>',
    '2300',
    'T',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='167'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='489'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '489HCl',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 2.0&times;10<sup>7</sup>&times;exp(9000&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='489HCl'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;1</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='82'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;1</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='935'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='935'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;1</sup>',
    '620',
    '?',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7647-01-0-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 9.16427 +45.13997/T &#8722;1.92853&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7647-01-0-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2',
    '2000',
    '?',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='34'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '34HClwrongref',
    '34,167', 'Pandis and Seinfeld (1989) refer to Marsh and McElroy (1985) as the source, but the quoted value cannot be found there.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='34HClwrongref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hydrogen chloride'),
    (SELECT id from literature WHERE bibtexkey='260'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='260'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hypochlorous acid',
    'HOCl',
    '',
    '7790-92-3',
    'QWPPOHNGKGFGJK-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5',
    '5900',
    'L',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5',
    '5900',
    'L',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5',
    '5900',
    'L',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5',
    '5900',
    'L',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5',
    '5900',
    'L',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='315'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1',
    '',
    'M',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='288'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.7',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='183'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '183HOCl',
    '', 'This value was extrapolated from data at T = 215&nbsp;K and T = 263&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='183HOCl'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.0',
    '',
    'M',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='3232'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.0',
    '4900',
    'M',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='287'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '287HOCl',
    '', 'Value at pH = 6.5.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='287HOCl'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>1</sup>',
    '5200',
    'M',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='3957'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='highTextrapol'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4',
    '8900',
    'M',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='3809'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6',
    '5100',
    'T',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='489'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.4',
    '',
    'Q',
    (SELECT id from species WHERE iupac='hypochlorous acid'),
    (SELECT id from literature WHERE bibtexkey='2521'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'perchloric acid',
    'HClO<sub>4</sub>',
    '',
    '7601-90-3',
    'VLTRZXGMWDSKGL-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>3</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='perchloric acid'),
    (SELECT id from literature WHERE bibtexkey='1536'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at200K',
    '', 'Value at T = 200&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at200K'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'monochlorine monoxide',
    'ClO',
    '',
    '14989-30-1',
    'MLWGAEVSWJXOQJ-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;3</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='monochlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;3</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='monochlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;3</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='monochlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;3</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='monochlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-E0'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'dichlorine monoxide',
    'Cl<sub>2</sub>O',
    '',
    '7791-21-1',
    'RCJVRSBWZCNNQT-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='3807'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3807',
    '3807', 'Secoy and Cady (1941) measured the gas-aqueous equilibrium constant p(Cl<sub>2</sub>O) / c(HOCl)<sup>2</sup> but not the intrinsic Henry''s law constant of Cl<sub>2</sub>O.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3807'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='3809'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3809',
    '3809', 'Ourisson and Kastner (1939) measured the gas-aqueous equilibrium constant p(Cl<sub>2</sub>O) / c(HOCl)<sup>2</sup> but not the intrinsic Henry''s law constant of Cl<sub>2</sub>O.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3809'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;2</sup>',
    '5900',
    'R',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='this'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'refthis-7791-21-1-ABC',
    '', 'The data from this work were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 1680.49677 &#8722;69933.08019/T &#8722;254.37188&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='refthis-7791-21-1-ABC'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'this_Cl2O',
    '3810', 'The gas-aqueous equilibrium constant p(Cl<sub>2</sub>O) / c(HOCl)<sup>2</sup> was combined with the temperature-dependent aqueous-phase hydration constant c(HOCl)<sup>2</sup> / c(Cl<sub>2</sub>O) from Roth (1929) in order to calculate the intrinsic Henry''s law constant of Cl<sub>2</sub>O.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='this_Cl2O'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'this_Cl2O_ref3807',
    '3807', 'Data for the equilibrium between gaseous Cl<sub>2</sub>O and aqueous HOCl were taken from Secoy and Cady (1941).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='this_Cl2O_ref3807'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.6&times;10<sup>&#8722;2</sup>',
    '5600',
    'R',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='this'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='this_Cl2O'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'this_Cl2O_ref3809',
    '3809', 'Data for the equilibrium between gaseous Cl<sub>2</sub>O and aqueous HOCl were taken from Ourisson and Kastner (1939).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='this_Cl2O_ref3809'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;2</sup>',
    '',
    'R',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='3949'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at277K',
    '', 'Value at T = 277&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at277K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'JPL_Cl2O',
    '190,3807', 'The recommended value from Wilhelm et&nbsp;al. (1977) appears to be dubious as it refers to Secoy and Cady (1941), who do not provide a value for the intrinsic Henry''s law constant of Cl<sub>2</sub>O.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='JPL_Cl2O'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='JPL_Cl2O'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='JPL_Cl2O'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='JPL_Cl2O'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='3857'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3857_Cl2O',
    '3857,3807', 'Young (1983) cites data from Secoy and Cady (1941). However, that paper only describes the equilibrium between gas-phase Cl<sub>2</sub>O and aqueous-phase HOCl. A Henry''s law constant of Cl<sub>2</sub>O is not provided. In addition, the values given by Young (1983) are not extrapolated to infinite dilution but to 1&nbsp;atm partial pressure of Cl<sub>2</sub>O. It is not explained how the nonlinear pressure dependence was extrapolated to 1&nbsp;atm.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3857_Cl2O'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='dichlorine monoxide'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '190_Cl2O',
    '190,3807', 'Wilhelm et&nbsp;al. (1977) cite Secoy and Cady (1941) as the source for their value. However, that paper only describes the equilibrium between gas-phase Cl<sub>2</sub>O and aqueous-phase HOCl. A Henry''s law constant of Cl<sub>2</sub>O is not provided.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='190_Cl2O'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'chlorine dioxide',
    'ClO<sub>2</sub>',
    '',
    '10049-04-4',
    'OSVXSBDYLRYLIG-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;2</sup>',
    '3500',
    'L',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;2</sup>',
    '3500',
    'L',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;2</sup>',
    '3500',
    'L',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;2</sup>',
    '3500',
    'L',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;2</sup>',
    '3300',
    'L',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3857'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;2</sup>',
    '3300',
    'L',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;2</sup>',
    '3200',
    'M',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3956'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;2</sup>',
    '3100',
    'M',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3965'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;2</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3966'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.6&times;10<sup>&#8722;3</sup>',
    '1200',
    'M',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3967'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.7&times;10<sup>&#8722;3</sup>',
    '3600',
    'M',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3969'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'C',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3955'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3955',
    '3955,3969', 'Even though Haller and Northgraves (1955) have been cited several times as the source of the ClO<sub>2</sub> solubility data, they did not perform any measurements. They took the data from the 1952 edition of the Kirk-Othmer Encyclopedia of Chemical Technology which apparently reproduced data from Holst (1944).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3955'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;2</sup>',
    '3100',
    '?',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3951'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.7&times;10<sup>&#8722;3</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3950'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;2</sup>',
    '3300',
    '?',
    (SELECT id from species WHERE iupac='chlorine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'perchloryl fluoride',
    'ClO<sub>3</sub>F',
    '',
    '7616-94-6',
    'XHFXMNZYIKFCPN-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='perchloryl fluoride'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitrosyl chloride',
    'NOCl',
    '',
    '2696-92-6',
    'VPCDQGACGWYTMC-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&gt; 4.9&times;10<sup>&#8722;4</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='nitrosyl chloride'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&gt; 4.9&times;10<sup>&#8722;4</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='nitrosyl chloride'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&gt; 4.9&times;10<sup>&#8722;4</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='nitrosyl chloride'),
    (SELECT id from literature WHERE bibtexkey='1060'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitryl chloride',
    'ClNO<sub>2</sub>',
    '',
    '13444-90-1',
    'HSSFHZJIMRUXDM-UHFFFAOYSA-M',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;4</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='nitryl chloride'),
    (SELECT id from literature WHERE bibtexkey='937'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;4</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='nitryl chloride'),
    (SELECT id from literature WHERE bibtexkey='536'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '536ClNO2',
    '', 'Derived as a fitting parameter used in numerical modeling.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='536ClNO2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.9&times;10<sup>&#8722;4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='nitryl chloride'),
    (SELECT id from literature WHERE bibtexkey='2360'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'chlorine nitrate',
    'ClNO<sub>3</sub>',
    '',
    '14545-72-3',
    'XYLGPCWDPLOBGP-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;2</sup>',
    '4500',
    'Q',
    (SELECT id from species WHERE iupac='chlorine nitrate'),
    (SELECT id from literature WHERE bibtexkey='3961'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3961_ClNO2',
    '3961', 'Robinson et&nbsp;al. (1997) assumed that the entropy of vaporization is the same for HOCl and ClNO<sub>3</sub> according to Trouton''s rule. On their page 3592, they mention a value of 7&nbsp;M&nbsp;atm<sup>&#8722;1</sup> at 250&nbsp;K. However, checking their Fig.&nbsp;9 and applying the temperature-dependence equation from their Table&nbsp;3, it seems that the value of 7&nbsp;M&nbsp;atm<sup>&#8722;1</sup> refers to 298&nbsp;K, not 250&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3961_ClNO2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='chlorine nitrate'),
    (SELECT id from literature WHERE bibtexkey='271'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='infty'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'chloramine',
    'NH<sub>2</sub>Cl',
    'chloramide',
    '10599-90-3',
    'QDHHCQZDFGDHMP-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.6&times;10<sup>&#8722;1</sup>',
    '6000',
    'L',
    (SELECT id from species WHERE iupac='chloramine'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.6&times;10<sup>&#8722;1</sup>',
    '6000',
    'L',
    (SELECT id from species WHERE iupac='chloramine'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.6&times;10<sup>&#8722;1</sup>',
    '6000',
    'L',
    (SELECT id from species WHERE iupac='chloramine'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.6&times;10<sup>&#8722;1</sup>',
    '6000',
    'L',
    (SELECT id from species WHERE iupac='chloramine'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.2&times;10<sup>&#8722;1</sup>',
    '4800',
    'M',
    (SELECT id from species WHERE iupac='chloramine'),
    (SELECT id from literature WHERE bibtexkey='287'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='chloramine'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'dichloramine',
    'NHCl<sub>2</sub>',
    'chlorimide',
    '3400-09-7',
    'JSYGRUBHOCKMGQ-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;1</sup>',
    '4200',
    'L',
    (SELECT id from species WHERE iupac='dichloramine'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;1</sup>',
    '4200',
    'L',
    (SELECT id from species WHERE iupac='dichloramine'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;1</sup>',
    '4200',
    'L',
    (SELECT id from species WHERE iupac='dichloramine'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;1</sup>',
    '4200',
    'L',
    (SELECT id from species WHERE iupac='dichloramine'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.8&times;10<sup>&#8722;1</sup>',
    '4200',
    'M',
    (SELECT id from species WHERE iupac='dichloramine'),
    (SELECT id from literature WHERE bibtexkey='287'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitrogen trichloride',
    'NCl<sub>3</sub>',
    '',
    '10025-85-1',
    'QEHKBHWEUPXBCW-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Chlorine (Cl)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;4</sup>',
    '4100',
    'L',
    (SELECT id from species WHERE iupac='nitrogen trichloride'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;4</sup>',
    '4100',
    'L',
    (SELECT id from species WHERE iupac='nitrogen trichloride'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;4</sup>',
    '4100',
    'L',
    (SELECT id from species WHERE iupac='nitrogen trichloride'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;4</sup>',
    '4100',
    'L',
    (SELECT id from species WHERE iupac='nitrogen trichloride'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;4</sup>',
    '4100',
    'M',
    (SELECT id from species WHERE iupac='nitrogen trichloride'),
    (SELECT id from literature WHERE bibtexkey='287'));
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Bromine (Br)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'bromine (molecular)',
    'Br<sub>2</sub>',
    '',
    '7726-95-6',
    'GDTBXPJZTBHREO-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Bromine (Br)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;3</sup>',
    '4400',
    'L',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;3</sup>',
    '4400',
    'L',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;3</sup>',
    '4400',
    'L',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;3</sup>',
    '4400',
    'L',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;2</sup>',
    '3600',
    'M',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='759'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '759',
    '759', 'Dubik et&nbsp;al. (1987) measured the solubility in concentrated salt solutions (natural brines).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='759'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.8&times;10<sup>&#8722;3</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='627'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.6&times;10<sup>&#8722;3</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='753'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;3</sup>',
    '4100',
    'M',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;2</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='3934'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;3</sup>',
    '3800',
    'M',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='862'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;3</sup>',
    '3600',
    'M',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='871'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;3</sup>',
    '4100',
    'V',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='1943'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;3</sup>',
    '3900',
    'R',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='753'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;3</sup>',
    '4000',
    'T',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='489'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.2&times;10<sup>&#8722;3</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='giona69'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'from3232',
    '3232', 'Value given here as quoted by McCoy et&nbsp;al. (1990).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3232'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.6&times;10<sup>&#8722;3</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='1038'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1038tdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.5&times;10<sup>&#8722;3</sup>',
    '3900',
    '?',
    (SELECT id from species WHERE iupac='bromine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7726-95-6-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;148.75612 +9709.79389/T +19.53402&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7726-95-6-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'bromine atom',
    'Br',
    '',
    '10097-32-2',
    'WKBOTKDWSSQWDR-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Bromine (Br)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='bromine atom'),
    (SELECT id from literature WHERE bibtexkey='189'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='189ClBr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;4</sup>',
    '1800',
    'T',
    (SELECT id from species WHERE iupac='bromine atom'),
    (SELECT id from literature WHERE bibtexkey='763'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='763'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrogen bromide',
    'HBr',
    '',
    '10035-10-6',
    'CPELXLSAUQHCOX-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Bromine (Br)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen bromide'),
    (SELECT id from literature WHERE bibtexkey='2840'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2840-10035-10-6',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 8.2&times;10<sup>9</sup>&times;exp(10000&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2840-10035-10-6'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen bromide'),
    (SELECT id from literature WHERE bibtexkey='530'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '530-10035-10-6',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 1.3&times;10<sup>10</sup>&times;exp(10000&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='530-10035-10-6'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen bromide'),
    (SELECT id from literature WHERE bibtexkey='489'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '489HBr',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 7.0&times;10<sup>9</sup>&times;exp(10000&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='489HBr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.8&times;10<sup>&#8722;2</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='hydrogen bromide'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    '?',
    (SELECT id from species WHERE iupac='hydrogen bromide'),
    (SELECT id from literature WHERE bibtexkey='171'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '171HBr',
    '171,46,87', 'Chameides and Stelson (1992) give a value of H<sub>s</sub>&#8242;&nbsp;= 7.1&times;10<sup>9</sup>&times;exp(6100&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa). They refer to Jacob (1986) and Chameides (1984) as the source, but this value cannot be found there.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='171HBr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;1</sup>',
    '250',
    '?',
    (SELECT id from species WHERE iupac='hydrogen bromide'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-10035-10-6-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;60.28318 +2830.41867/T +8.66642&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-10035-10-6-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hydrogen bromide'),
    (SELECT id from literature WHERE bibtexkey='260'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='260'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hypobromous acid',
    'HOBr',
    '',
    '13517-11-8',
    'CUILPNURFADTPE-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Bromine (Br)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&gt; 1.3&times;10<sup>1</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&gt; 1.3&times;10<sup>1</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&gt; 1.9&times;10<sup>1</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='288'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>1</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='3232'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;2</sup>',
    '4000',
    'T',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='276'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '276',
    '', 'The value is from Table&nbsp;1 of the paper. However, <i>J.&nbsp;Geophys.&nbsp;Res.</i> forgot to print the tables, and I received them directly from the author.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='276'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.0&times;10<sup>1</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='937'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;1</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='446'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'JPL_HOBr',
    '3500', 'The value presented for HOBr is incorrect. A corrected version was later published by Burkholder et&nbsp;al. (2019).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='JPL_HOBr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='JPL_HOBr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hypobromous acid'),
    (SELECT id from literature WHERE bibtexkey='910'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '910wrong',
    '910', 'Fickert (1998) extracted a value from wetted-wall flow tube experiments. However, it was later discovered that under the experimental conditions no evaluation of H<sub>s</sub> is possible (John Crowley, personal communication, 1999).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='910wrong'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'nitryl bromide',
    'BrNO<sub>2</sub>',
    '',
    '13536-70-4',
    'SEYAFCXQVVHRPY-UHFFFAOYSA-M',
    (SELECT id from subcategories WHERE subcatname='Bromine (Br)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.0&times;10<sup>&#8722;3</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='nitryl bromide'),
    (SELECT id from literature WHERE bibtexkey='937'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'bromine nitrate',
    'BrNO<sub>3</sub>',
    '',
    '40423-14-1',
    'RRTWEEAEXPZMPY-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Bromine (Br)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='bromine nitrate'),
    (SELECT id from literature WHERE bibtexkey='271'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='infty'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'bromine chloride',
    'BrCl',
    '',
    '13863-41-7',
    'CODNYICXDISAEA-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Bromine (Br)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.7&times;10<sup>&#8722;3</sup>',
    '5600',
    'L',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.7&times;10<sup>&#8722;3</sup>',
    '5600',
    'L',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.7&times;10<sup>&#8722;3</sup>',
    '5600',
    'L',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.7&times;10<sup>&#8722;3</sup>',
    '5600',
    'L',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&lt; 6.2&times;10<sup>&#8722;2</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='1323'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at275K',
    '', 'Value at T = 275&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at275K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;2</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='1018'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at290K',
    '', 'Value at T = 290&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at290K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.3&times;10<sup>&#8722;3</sup>',
    '5600',
    'M',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='1038'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.2&times;10<sup>&#8722;2</sup>',
    '4000',
    'M',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='759'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='759'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;2</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='this'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'BrClRS',
    '489,318', 'Calculated using data from Wagman et&nbsp;al. (1982) and the aqueous-phase equilibrium Cl<sub>2</sub> + Br<sub>2</sub> &harr;&nbsp;2 BrCl from Wang et&nbsp;al. (1994).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='BrClRS'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.9&times;10<sup>&#8722;4</sup>',
    '4000',
    'E',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='3770'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.8&times;10<sup>&#8722;3</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='bromine chloride'),
    (SELECT id from literature WHERE bibtexkey='937'));
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Iodine (I)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'iodine (molecular)',
    'I<sub>2</sub>',
    '',
    '7553-56-2',
    'PNDPGZBMCMUPRI-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.8&times;10<sup>&#8722;2</sup>',
    '4300',
    'M',
    (SELECT id from species WHERE iupac='iodine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='1149'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.8&times;10<sup>&#8722;2</sup>',
    '3900',
    'V',
    (SELECT id from species WHERE iupac='iodine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='1943'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.0&times;10<sup>&#8722;2</sup>',
    '4400',
    'R',
    (SELECT id from species WHERE iupac='iodine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='582'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.1&times;10<sup>&#8722;2</sup>',
    '4600',
    'R',
    (SELECT id from species WHERE iupac='iodine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='763'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.2&times;10<sup>&#8722;2</sup>',
    '4800',
    'T',
    (SELECT id from species WHERE iupac='iodine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='489'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;2</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='iodine (molecular)'),
    (SELECT id from literature WHERE bibtexkey='203'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '203I2',
    '203', 'Thompson and Zafiriou (1983) quote a paper as the source that gives only the solubility but not the Henry''s law constant.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='203I2'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'iodine atom',
    'I',
    '',
    '14362-44-8',
    'ZCYVEMRRCGMTRW-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;4</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='iodine atom'),
    (SELECT id from literature WHERE bibtexkey='189'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '189I',
    '762', 'Calculated from the free energy of solution by Schwarz and Bielski (1986).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='189I'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.2&times;10<sup>&#8722;5</sup>',
    '2300',
    'T',
    (SELECT id from species WHERE iupac='iodine atom'),
    (SELECT id from literature WHERE bibtexkey='763'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='763'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrogen iodide',
    'HI',
    '',
    '10034-85-2',
    'XMBWDFGMSWQBCA-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen iodide'),
    (SELECT id from literature WHERE bibtexkey='530'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '530-10034-85-2',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 2.5&times;10<sup>10</sup>&times;exp(9800&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='530-10034-85-2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'T',
    (SELECT id from species WHERE iupac='hydrogen iodide'),
    (SELECT id from literature WHERE bibtexkey='489'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '489HI',
    '', 'H<sub>s</sub>&#8242;&nbsp;= 2.1&times;10<sup>10</sup>&times;exp(9800&nbsp;K (1/T&#8722;1/T<sup>&ominus;</sup>))&nbsp;mol<sup>2</sup>/(m<sup>6</sup>&nbsp;Pa)');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='489HI'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;1</sup>',
    '9800',
    'E',
    (SELECT id from species WHERE iupac='hydrogen iodide'),
    (SELECT id from literature WHERE bibtexkey='3770'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='hydrogen iodide'),
    (SELECT id from literature WHERE bibtexkey='897'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='infty'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hydrogen iodide'),
    (SELECT id from literature WHERE bibtexkey='260'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='260'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'iodine monoxide',
    'IO',
    '',
    '14696-98-1',
    'AFSVSXMRDKPOEW-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4',
    '',
    '?',
    (SELECT id from species WHERE iupac='iodine monoxide'),
    (SELECT id from literature WHERE bibtexkey='3760'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3760',
    '3760,2312', 'Saiz-Lopez et&nbsp;al. (2014) refer to Saiz-Lopez et&nbsp;al. (2008) as the source, but the quoted value cannot be found there.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3760'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'iodine dioxide',
    'OIO',
    '',
    '13494-92-3',
    'WXDJHDMIIZKXSK-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='iodine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3760'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3760'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'diiodine dioxide',
    'I<sub>2</sub>O<sub>2</sub>',
    '',
    '215239-62-6',
    'IELAHHPSAVYAOC-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='diiodine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3771'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'whichisomer',
    '', 'It is unclear to which isomer the value of the Henry''s law constant refers to.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichisomer'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3771-IxOy',
    '', 'Assumed to be infinity by analogy with INO<sub>3</sub>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3771-IxOy'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='diiodine dioxide'),
    (SELECT id from literature WHERE bibtexkey='897'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichisomer'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='infty'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='diiodine dioxide'),
    (SELECT id from literature WHERE bibtexkey='3760'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichisomer'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3760'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'diiodine trioxide',
    'I<sub>2</sub>O<sub>3</sub>',
    '',
    '11085-17-9',
    'NMNCVPKLBXOKQA-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='diiodine trioxide'),
    (SELECT id from literature WHERE bibtexkey='3771'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichisomer'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3771-IxOy'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='diiodine trioxide'),
    (SELECT id from literature WHERE bibtexkey='3760'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichisomer'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3760'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'diiodine tetroxide',
    'I<sub>2</sub>O<sub>4</sub>',
    '',
    '1024652-24-1',
    'XHTWXUOEQMOFEJ-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='diiodine tetroxide'),
    (SELECT id from literature WHERE bibtexkey='3771'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichisomer'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3771-IxOy'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='diiodine tetroxide'),
    (SELECT id from literature WHERE bibtexkey='3760'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichisomer'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3760'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hypoiodous acid',
    'HOI',
    '',
    '14332-21-9',
    'GEOVEUCEIQCBKH-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&gt; 4.1',
    '',
    'C',
    (SELECT id from species WHERE iupac='hypoiodous acid'),
    (SELECT id from literature WHERE bibtexkey='582'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'E',
    (SELECT id from species WHERE iupac='hypoiodous acid'),
    (SELECT id from literature WHERE bibtexkey='203'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '203HOI',
    '203', 'Thompson and Zafiriou (1983) assume that H<sub>s</sub><sup>cp</sup>(HOI) is between 4.4&times;10<sup>&#8722;1</sup>&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup> and 4.4&times;10<sup>2</sup>&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='203HOI'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'iodine nitrite',
    'INO<sub>2</sub>',
    '',
    '15465-40-4',
    'PSZTWRLUACEPOW-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.0&times;10<sup>&#8722;3</sup>',
    '',
    'E',
    (SELECT id from species WHERE iupac='iodine nitrite'),
    (SELECT id from literature WHERE bibtexkey='3771'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3771-INO2',
    '3771', 'Badia et&nbsp;al. (2019) assume that INO<sub>2</sub> has the same Henry''s law constant as BrNO<sub>2</sub>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3771-INO2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='iodine nitrite'),
    (SELECT id from literature WHERE bibtexkey='897'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='infty'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'iodine nitrate',
    'INO<sub>3</sub>',
    '',
    '14696-81-2',
    'CCJHDZZUWZIVJF-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='iodine nitrate'),
    (SELECT id from literature WHERE bibtexkey='897'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='infty'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'iodine chloride',
    'ICl',
    '',
    '7790-99-0',
    'QZRGKCOWNLSUDK-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1',
    '',
    'T',
    (SELECT id from species WHERE iupac='iodine chloride'),
    (SELECT id from literature WHERE bibtexkey='489'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'iodine bromide',
    'IBr',
    '',
    '7789-33-5',
    'CBEQRNSPHCCXSH-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Iodine (I)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;1</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='iodine bromide'),
    (SELECT id from literature WHERE bibtexkey='489'));
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Sulfur (S)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'sulfur',
    'S',
    '',
    '7704-34-9',
    'NINIDFKCEFEMDL-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Sulfur (S)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='sulfur'),
    (SELECT id from literature WHERE bibtexkey='3549'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3549',
    '', 'Data taken from the AGRITOX database file agritox-20210608.zip.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3549'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'hydrogen sulfide',
    'H<sub>2</sub>S',
    '',
    '7783-06-4',
    'RWSOTUBLDIXVET-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Sulfur (S)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2100',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2100',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2100',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2100',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2000',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2200',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='615'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2000',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3863'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3863_H2S',
    '3863', 'Fogg and Young (1988) provide two different fitting functions: one for T &lt; 283.2&nbsp;K and one for T &gt; 283.2&nbsp;K. At T=283.2&nbsp;K, the functions have different values and different slopes. Here, only the function that is valid at T<sup>&ominus;</sup> is used.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3863_H2S'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2000',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3085'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2100',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='540'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2100',
    'L',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;4</sup>',
    '1700',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='2929'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.2&times;10<sup>&#8722;4</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3534'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.6&times;10<sup>&#8722;4</sup>',
    '2100',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='586'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;3</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='997'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;3</sup>',
    '1700',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3668'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3668'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.4&times;10<sup>&#8722;4</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='2928'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2100',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3917'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3917-7783-06-4-ABC',
    '3917', 'The data from Clarke and Glew (1971) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;133.37135 +7422.07576/T +17.82903&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3917-7783-06-4-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3718'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2100',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='862'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;3</sup>',
    '2000',
    'M',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3933'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3933-7783-06-4-ABC',
    '3933', 'The data from Schoenfeld (1855) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 98.96644 &#8722;3021.28876/T &#8722;16.78233&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3933-7783-06-4-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.6&times;10<sup>&#8722;4</sup>',
    '2000',
    'R',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='2630'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '',
    'R',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='714'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2300',
    'T',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3540'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2000',
    '?',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;3</sup>',
    '2100',
    '?',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7783-06-4-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;122.57010 +6962.28299/T +16.20245&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7783-06-4-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'W',
    (SELECT id from species WHERE iupac='hydrogen sulfide'),
    (SELECT id from literature WHERE bibtexkey='2629'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2629',
    '2630', 'The parameter fit for the temperature dependence is incorrect. A corrected version was later presented by Iliuta and Larachi (2007).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2629'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'deuterium sulfide',
    'D<sub>2</sub>S',
    '',
    '13536-94-2',
    'RWSOTUBLDIXVET-ZSJDYOACSA-N',
    (SELECT id from subcategories WHERE subcatname='Sulfur (S)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;4</sup>',
    '2100',
    'M',
    (SELECT id from species WHERE iupac='deuterium sulfide'),
    (SELECT id from literature WHERE bibtexkey='3917'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3917-13536-94-2-ABC',
    '3917', 'The data from Clarke and Glew (1971) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;152.96053 +8324.82999/T +20.73129&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3917-13536-94-2-ABC'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3917',
    '', 'Obtained with D<sub>2</sub>O as solvent.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3917'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'sulfur dioxide',
    'SO<sub>2</sub>',
    '',
    '7446-09-5',
    'RAHZWNYVWXNFOC-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Sulfur (S)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;2</sup>',
    '2900',
    'L',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;2</sup>',
    '2900',
    'L',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;2</sup>',
    '2900',
    'L',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;2</sup>',
    '2900',
    'L',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3085'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;2</sup>',
    '2900',
    'L',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3857'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3200',
    'L',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='834'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3000',
    'L',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='540'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;2</sup>',
    '2800',
    'L',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.0&times;10<sup>&#8722;1</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3064'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at353K',
    '', 'Value at T = 353&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at353K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3100',
    'M',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='1072'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;2</sup>',
    '3400',
    'M',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3933'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3933-7446-09-5-ABC',
    '3933', 'The data from Schoenfeld (1855) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 265.79241 &#8722;9131.99684/T &#8722;42.01987&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3933-7446-09-5-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;2</sup>',
    '1200',
    'V',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3122'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3100',
    'T',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='87'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3100',
    'T',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3540'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='rodriguez2001'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'from3536',
    '3536', 'Value given here as quoted by Rodríguez-Sevilla et&nbsp;al. (2002).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3536'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3100',
    'C',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='34'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3300',
    'C',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='821'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;2</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;2</sup>',
    '2900',
    '?',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;2</sup>',
    '3100',
    '?',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7446-09-5-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 153.05871 &#8722;4328.05304/T &#8722;25.05397&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7446-09-5-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3100',
    '?',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='523'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;2</sup>',
    '3100',
    '?',
    (SELECT id from species WHERE iupac='sulfur dioxide'),
    (SELECT id from literature WHERE bibtexkey='103'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'sulfur trioxide',
    'SO<sub>3</sub>',
    '',
    '7446-11-9',
    'AKEJUJNQAAGONA-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Sulfur (S)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.1&times;10<sup>&#8722;8</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='sulfur trioxide'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '&#8734;',
    '',
    'E',
    (SELECT id from species WHERE iupac='sulfur trioxide'),
    (SELECT id from literature WHERE bibtexkey='271'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='infty'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'sulfuric acid',
    'H<sub>2</sub>SO<sub>4</sub>',
    '',
    '7664-93-9',
    'QAOWNCQODCNURD-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Sulfur (S)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='sulfuric acid'),
    (SELECT id from literature WHERE bibtexkey='769'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '769',
    '769', 'Marti et&nbsp;al. (1997) give partial pressures of H<sub>2</sub>SO<sub>4</sub> over a concentrated solution (e.g., 2.6&times;10<sup>&#8722;9</sup>&nbsp;Pa for 54.1&nbsp;wt&nbsp;% at 298&nbsp;K). Extrapolating this to dilute solutions can only be considered an order-of-magnitude approximation for H<sub>s</sub>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='769'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='sulfuric acid'),
    (SELECT id from literature WHERE bibtexkey='201'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '201',
    '201', 'Ayers et&nbsp;al. (1980) give partial pressures of H<sub>2</sub>SO<sub>4</sub> over concentrated solutions at high temperatures. Extrapolating this to dilute solutions can only be considered an order-of-magnitude approximation for H<sub>s</sub>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='201'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='sulfuric acid'),
    (SELECT id from literature WHERE bibtexkey='752'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '752',
    '752', 'Gmitro and Vermeulen (1964) give partial pressures of H<sub>2</sub>SO<sub>4</sub> over a concentrated solution (e.g., 10<sup>&#8722;7</sup>&nbsp;mmHg for 70&nbsp;wt&nbsp;% at 298&nbsp;K). Extrapolating this to dilute solutions can only be considered an order-of-magnitude approximation for H<sub>s</sub>.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='752'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'V',
    (SELECT id from species WHERE iupac='sulfuric acid'),
    (SELECT id from literature WHERE bibtexkey='2838'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2838',
    '2838', 'Clegg et&nbsp;al. (1998) estimate a Henry''s law constant of 5&times;10<sup>11</sup>&nbsp;atm<sup>&#8722;1</sup> at 303.15&nbsp;K for the reaction H<sub>2</sub>SO<sub>4</sub>(g) &harr; 2 H<sup>+</sup>(aq) + SO<sub>4</sub><sup>2&#8722;</sup>(aq) but do not give a definition for it. Probably it is defined as x<sup>2</sup>(H<sup>+</sup>)&times;x(SO<sub>4</sub><sup>2&#8722;</sup>)/p(H<sub>2</sub>SO<sub>4</sub>), where x is the aqueous-phase mixing ratio.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2838'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>13</sup>',
    '20000',
    'T',
    (SELECT id from species WHERE iupac='sulfuric acid'),
    (SELECT id from literature WHERE bibtexkey='140'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>7</sup>',
    '10000',
    'T',
    (SELECT id from species WHERE iupac='sulfuric acid'),
    (SELECT id from literature WHERE bibtexkey='1076'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'sulfur hexafluoride',
    'SF<sub>6</sub>',
    '',
    '2551-62-4',
    'SFZCNBIFKDRMGX-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Sulfur (S)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;6</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='2828'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;6</sup>',
    '2100',
    'L',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;6</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.3&times;10<sup>&#8722;6</sup>',
    '2700',
    'M',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='1776'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref1776-2551-62-4-ABC',
    '1776', 'The data from Bullister et&nbsp;al. (2002) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;281.50843 +14256.43847/T +38.73689&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref1776-2551-62-4-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='2924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='2486'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;6</sup>',
    '260',
    'M',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3601'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HTdep_r2_lessthan_0.5'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='highTextrapol'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3786'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3787'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3785'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3785SF6',
    '3785', 'The data presented for SF6 in Table&nbsp;II of Shoor et&nbsp;al. (1969) appear to be incorrect and are not reproduced here.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3785SF6'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;6</sup>',
    '2400',
    'M',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='2831'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2831-2551-62-4-ABC',
    '2831', 'The data from Ashton et&nbsp;al. (1968) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;431.90650 +20715.81650/T +61.33841&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2831-2551-62-4-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.2&times;10<sup>&#8722;6</sup>',
    '3500',
    'M',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3801'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.2&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3078'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3730'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3586_val',
    '', 'Value from the validation set for checking whether the model is satisfactory for compounds that are absent from the training set.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_val'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '3200',
    'Q',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.2&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3586_exp',
    '', 'Experimental value, extracted from HENRYWIN.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_exp'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '2800',
    '?',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;6</sup>',
    '2400',
    '?',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='sulfur hexafluoride'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'sulfuryl fluoride',
    'SO<sub>2</sub>F<sub>2</sub>',
    '',
    '2699-79-8',
    'OBTWBSRJZRCYQV-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Sulfur (S)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.9&times;10<sup>&#8722;5</sup>',
    '3100',
    'M',
    (SELECT id from species WHERE iupac='sulfuryl fluoride'),
    (SELECT id from literature WHERE bibtexkey='2203'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.3&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='sulfuryl fluoride'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3586_V',
    '', 'Estimation based on the quotient between vapor pressure and water solubility, extracted from HENRYWIN.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_V'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.2',
    '',
    'Q',
    (SELECT id from species WHERE iupac='sulfuryl fluoride'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.3&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='sulfuryl fluoride'),
    (SELECT id from literature WHERE bibtexkey='3549'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3549'));
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Rare gases (He, Ne, Ar, Kr, Xe, Rn)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'helium',
    'He',
    '',
    '7440-59-7',
    'SWQJXJOGLNCZEY-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Rare gases (He, Ne, Ar, Kr, Xe, Rn)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.9&times;10<sup>&#8722;6</sup>',
    '15',
    'L',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;6</sup>',
    '83',
    'L',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;6</sup>',
    '83',
    'L',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3849'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;6</sup>',
    '92',
    'L',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.7&times;10<sup>&#8722;6</sup>',
    '320',
    'L',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3431'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.9&times;10<sup>&#8722;6</sup>',
    '69',
    'M',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='2436'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'M',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3785'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3785helium',
    '3785', 'The data presented for helium in Table&nbsp;II of Shoor et&nbsp;al. (1969) appear to be incorrect and are not reproduced here.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3785helium'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.7&times;10<sup>&#8722;6</sup>',
    '120',
    'M',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='764'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0764-7440-59-7-ABC',
    '764', 'The data from Morrison and Johnstone (1954) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;267.15298 +11440.04263/T +37.95994&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0764-7440-59-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3801'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;6</sup>',
    '210',
    'M',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3789'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3789-7440-59-7-ABC',
    '3789', 'The data from Lannung (1930) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 84.35043 &#8722;4135.59197/T &#8722;14.55881&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3789-7440-59-7-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.7&times;10<sup>&#8722;6</sup>',
    '380',
    'M',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3548'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.3&times;10<sup>&#8722;6</sup>',
    '-700',
    'M',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3714'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HTdep_r2_lessthan_0.5'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.7&times;10<sup>&#8722;6</sup>',
    '220',
    'V',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.3&times;10<sup>&#8722;6</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.3&times;10<sup>&#8722;6</sup>',
    '4',
    'Q',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HTdep_r2_lessthan_0.5'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.3&times;10<sup>&#8722;6</sup>',
    '71',
    'Q',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HTdep_r2_lessthan_0.5'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3517-Warr',
    '3550', 'Calculated employing molecular force field models for the solutes from Warr et&nbsp;al. (2015).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3517-Warr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.9&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3550'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;6</sup>',
    '83',
    '?',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.9&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.7&times;10<sup>&#8722;6</sup>',
    '200',
    '?',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7440-59-7-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;153.15219 +6434.36008/T +20.89911&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7440-59-7-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='helium'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'neon',
    'Ne',
    '',
    '7440-01-9',
    'GKAOGPIIYCISHV-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Rare gases (He, Ne, Ar, Kr, Xe, Rn)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;6</sup>',
    '430',
    'L',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;6</sup>',
    '470',
    'L',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;6</sup>',
    '470',
    'L',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3849'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;6</sup>',
    '450',
    'L',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;6</sup>',
    '440',
    'M',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='2436'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;6</sup>',
    '510',
    'M',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='2557'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.3&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3787'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;6</sup>',
    '460',
    'M',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='764'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0764-7440-01-9-ABC',
    '764', 'The data from Morrison and Johnstone (1954) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;171.84866 +7492.61303/T +23.58966&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0764-7440-01-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.6&times;10<sup>&#8722;6</sup>',
    '37',
    'M',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3789'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3789-7440-01-9-ABC',
    '3789', 'The data from Lannung (1930) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;40.04033 +1266.80589/T +4.12574&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3789-7440-01-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.6&times;10<sup>&#8722;6</sup>',
    '-990',
    'M',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3714'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;6</sup>',
    '510',
    'V',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.8&times;10<sup>&#8722;6</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;6</sup>',
    '250',
    'Q',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HTdep_r2_lessthan_0.9'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.7&times;10<sup>&#8722;6</sup>',
    '470',
    'Q',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='HTdep_r2_lessthan_0.9'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3517-Warr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.6&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3550'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;6</sup>',
    '470',
    '?',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;6</sup>',
    '550',
    '?',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7440-01-9-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;150.94728 +6639.96438/T +20.42365&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7440-01-9-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='neon'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'argon',
    'Ar',
    '',
    '7440-37-1',
    'XKRFYHLGVUSROY-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Rare gases (He, Ne, Ar, Kr, Xe, Rn)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1700',
    'L',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='2828'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1400',
    'L',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3851'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='2435'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2435-7440-37-1-ABC',
    '2435', 'The data from Rettich et&nbsp;al. (1992) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;178.55165 +8674.63293/T +24.26764&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2435-7440-37-1-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='2436'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='2486'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='2557'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1200',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3601'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='highTextrapol'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1300',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3547'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3567'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3567-7440-37-1-ABC',
    '3567', 'The data from Murray and Riley (1970) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;151.84230 +7548.13106/T +20.24085&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3567-7440-37-1-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3785'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3785-7440-37-1-ABC',
    '3785', 'The data from Shoor et&nbsp;al. (1969) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;177.19900 +8740.49327/T +23.99118&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3785-7440-37-1-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='2831'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2831-7440-37-1-ABC',
    '2831', 'The data from Ashton et&nbsp;al. (1968) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;160.52023 +7898.05096/T +21.56102&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2831-7440-37-1-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='764'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0764-7440-37-1-ABC',
    '764', 'The data from Morrison and Johnstone (1954) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;159.49603 +7859.86242/T +21.39868&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0764-7440-37-1-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1800',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3801'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3789'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3789-7440-37-1-ABC',
    '3789', 'The data from Lannung (1930) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;183.19260 +8856.79081/T +24.97248&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3789-7440-37-1-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '1300',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3714'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='862'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1400',
    'V',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1400',
    'V',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.5&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '1200',
    'Q',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3517-Warr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '1100',
    'Q',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3517-Vrabec',
    '3551', 'Calculated employing molecular force field models for the solutes from Vrabec et&nbsp;al. (2001).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3517-Vrabec'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3550'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1500',
    '?',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1500',
    '?',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7440-37-1-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;143.77232 +7158.59719/T +19.05403&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7440-37-1-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='argon'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'krypton',
    'Kr',
    '',
    '7439-90-9',
    'DNNSSWSSYDEUBZ-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Rare gases (He, Ne, Ar, Kr, Xe, Rn)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '1700',
    'L',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '1900',
    'L',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '1900',
    'L',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3850'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '1900',
    'L',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='2441'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '2000',
    'L',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3696'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '1800',
    'M',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='2436'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '1900',
    'M',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='2557'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>&#8722;5</sup>',
    '1800',
    'M',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3601'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='highTextrapol'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;5</sup>',
    '1700',
    'M',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='764'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0764-7439-90-9-ABC',
    '764', 'The data from Morrison and Johnstone (1954) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;153.87925 +7855.39037/T +20.51280&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0764-7439-90-9-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3714'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3714_krypton',
    '3714', 'Two series of measurements with considerably different results are presented by von Antropoff (1910) for krypton.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3714_krypton'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.8&times;10<sup>&#8722;5</sup>',
    '1900',
    'M',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3714'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3714_krypton'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;5</sup>',
    '1800',
    'V',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;5</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '1600',
    'Q',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3517-Warr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '1400',
    'Q',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3517-Vrabec'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3550'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '1900',
    '?',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '1800',
    '?',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7439-90-9-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;220.92114 +10903.79433/T +30.49407&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7439-90-9-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='krypton'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'xenon',
    'Xe',
    '',
    '7440-63-3',
    'FHNFHKCVQCLJFQ-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Rare gases (He, Ne, Ar, Kr, Xe, Rn)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;5</sup>',
    '2200',
    'L',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.3&times;10<sup>&#8722;5</sup>',
    '2300',
    'L',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.3&times;10<sup>&#8722;5</sup>',
    '2300',
    'L',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3850'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.2&times;10<sup>&#8722;5</sup>',
    '2200',
    'L',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.3&times;10<sup>&#8722;5</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='2441'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.5&times;10<sup>&#8722;5</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3696'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.0&times;10<sup>&#8722;5</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3431'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3431',
    '3431', 'The value b for Xe given by Himmelblau (1960) in their Table&nbsp;III is incorrect. Most likely, only a minus sign is missing.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3431'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.3&times;10<sup>&#8722;5</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='2436'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.2&times;10<sup>&#8722;5</sup>',
    '2400',
    'M',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='2557'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.2&times;10<sup>&#8722;5</sup>',
    '2200',
    'M',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='764'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0764-7440-63-3-ABC',
    '764', 'The data from Morrison and Johnstone (1954) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;165.83721 +8808.62019/T +22.15186&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0764-7440-63-3-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;5</sup>',
    '2500',
    'M',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3714'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.2&times;10<sup>&#8722;5</sup>',
    '2200',
    'V',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.5&times;10<sup>&#8722;5</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;5</sup>',
    '2300',
    'Q',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3517-Warr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.9&times;10<sup>&#8722;5</sup>',
    '1800',
    'Q',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3517-Vrabec'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3550'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.3&times;10<sup>&#8722;5</sup>',
    '2300',
    '?',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.9&times;10<sup>&#8722;5</sup>',
    '2200',
    '?',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-7440-63-3-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;199.40126 +10306.10786/T +27.18844&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-7440-63-3-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.3&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='xenon'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'xenon-133',
    '<sup>133</sup>Xe',
    '',
    '14932-42-4',
    'FHNFHKCVQCLJFQ-NJFSPNSNSA-N',
    (SELECT id from subcategories WHERE subcatname='Rare gases (He, Ne, Ar, Kr, Xe, Rn)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.2&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='xenon-133'),
    (SELECT id from literature WHERE bibtexkey='3700'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'radon',
    'Rn',
    '',
    '10043-92-2',
    'SYUHGPGVQRZVTB-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Rare gases (He, Ne, Ar, Kr, Xe, Rn)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;5</sup>',
    '2900',
    'L',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;5</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='3850'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.2&times;10<sup>&#8722;5</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.4&times;10<sup>&#8722;5</sup>',
    '2600',
    'M',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='3545'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3545-10043-92-2-ABC',
    '3545', 'The data from Lewis et&nbsp;al. (1987) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( 5.03587 +1555.37916/T &#8722;3.42648&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3545-10043-92-2-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;4</sup>',
    '2700',
    'M',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='3546'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;4</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.4&times;10<sup>&#8722;5</sup>',
    '2400',
    'Q',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='3517'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3517-Mick',
    '3552', 'Calculated employing molecular force field models for the solutes from Mick et&nbsp;al. (2016).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3517-Mick'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;5</sup>',
    '2600',
    '?',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;5</sup>',
    '2800',
    '?',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-10043-92-2-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;240.66156 +12686.97685/T +33.12171&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-10043-92-2-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='radon'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Other elements (B, Se, P, As, Hg)',
    (SELECT id from categories WHERE catname='Inorganic species'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'boric acid',
    'H<sub>3</sub>BO<sub>3</sub>',
    '',
    '10043-35-3',
    'KGBXLFKZBHKPEV-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='boric acid'),
    (SELECT id from literature WHERE bibtexkey='3088'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'selenium hydride',
    'H<sub>2</sub>Se',
    '',
    '7783-07-5',
    'BUGBHKTXTAQXES-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.1&times;10<sup>&#8722;4</sup>',
    '1700',
    'L',
    (SELECT id from species WHERE iupac='selenium hydride'),
    (SELECT id from literature WHERE bibtexkey='3863'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;4</sup>',
    '1900',
    'L',
    (SELECT id from species WHERE iupac='selenium hydride'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;4</sup>',
    '1900',
    'M',
    (SELECT id from species WHERE iupac='selenium hydride'),
    (SELECT id from literature WHERE bibtexkey='3808'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3808-7783-07-5-ABC',
    '3808', 'The data from Sisi et&nbsp;al. (1971) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;81.82525 +4954.57763/T +10.19950&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3808-7783-07-5-ABC'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'phosphorus trihydride',
    'PH<sub>3</sub>',
    'phosphine',
    '7803-51-2',
    'XYFCBTPGUUZFHI-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.1&times;10<sup>&#8722;5</sup>',
    '2000',
    'L',
    (SELECT id from species WHERE iupac='phosphorus trihydride'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.9&times;10<sup>&#8722;5</sup>',
    '3000',
    'M',
    (SELECT id from species WHERE iupac='phosphorus trihydride'),
    (SELECT id from literature WHERE bibtexkey='3033'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3033seawater',
    '', 'Solubility in natural sea water. Measurements at different salinities were also performed but only at a fixed temperature of 296.15&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3033seawater'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.1&times;10<sup>&#8722;5</sup>',
    '2000',
    '?',
    (SELECT id from species WHERE iupac='phosphorus trihydride'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'arsenic hydride',
    'AsH<sub>3</sub>',
    'arsine',
    '7784-42-1',
    'RQNWIZPPADIBDY-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.8&times;10<sup>&#8722;5</sup>',
    '2100',
    'L',
    (SELECT id from species WHERE iupac='arsenic hydride'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'mercury',
    'Hg',
    '',
    '7439-97-6',
    'QSHDDOUJBYECFT-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;3</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;3</sup>',
    '2600',
    'L',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;3</sup>',
    '5400',
    'L',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='3862'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;3</sup>',
    '4800',
    'L',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2279'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;3</sup>',
    '2600',
    'M',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2820'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;3</sup>',
    '2500',
    'M',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2280'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.7&times;10<sup>&#8722;4</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2471'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;3</sup>',
    '5700',
    'V',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2578'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;3</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2285'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;3</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='3070'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '2300',
    'Q',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2577'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2577-LFER',
    '', 'Temperature dependence calculated using linear free energy relationships (LFERs).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2577-LFER'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;3</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2171'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;3</sup>',
    '2700',
    '?',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2171'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;3</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='2286'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2286',
    '2286', 'Petersen et&nbsp;al. (1998) give the invalid unit "mol&nbsp;L<sup>&#8722;1</sup>&nbsp;ppm<sup>&#8722;1</sup>". Here, it is assumed that "ppm" is used as a synonym for "10<sup>&#8722;6</sup>&nbsp;atm".');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2286'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.2&times;10<sup>&#8722;4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury'),
    (SELECT id from literature WHERE bibtexkey='695'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'mercury(II) oxide',
    'HgO',
    '',
    '21908-53-2',
    'UKWHYYKOEPRTIC-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.2&times;10<sup>4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury(II) oxide'),
    (SELECT id from literature WHERE bibtexkey='2285'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2285wrongrefHgO',
    '2285,2286', 'Shon et&nbsp;al. (2005) refer to Petersen et&nbsp;al. (1998) as the source, but a different value is listed there.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2285wrongrefHgO'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.7&times;10<sup>10</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury(II) oxide'),
    (SELECT id from literature WHERE bibtexkey='2171'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury(II) oxide'),
    (SELECT id from literature WHERE bibtexkey='2286'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2286'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'mercury dihydroxide',
    'Hg(OH)<sub>2</sub>',
    '',
    '12135-13-6',
    'VLKKXDVIWIBHHS-UHFFFAOYSA-L',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>2</sup>',
    '4200',
    'C',
    (SELECT id from species WHERE iupac='mercury dihydroxide'),
    (SELECT id from literature WHERE bibtexkey='3070'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>2</sup>',
    '4200',
    'C',
    (SELECT id from species WHERE iupac='mercury dihydroxide'),
    (SELECT id from literature WHERE bibtexkey='2290'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'mercury dichloride',
    'HgCl<sub>2</sub>',
    '',
    '7487-94-7',
    'LWJROJCJINYWOX-UHFFFAOYSA-L',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>3</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2572'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'at333K',
    '', 'Value at T = 333&nbsp;K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at333K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>4</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2577'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.2&times;10<sup>4</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2577'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>4</sup>',
    '7400',
    'R',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2571'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>5</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2285'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>4</sup>',
    '5300',
    'C',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='3070'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>4</sup>',
    '5300',
    'C',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2290'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.2&times;10<sup>4</sup>',
    '7400',
    'Q',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2577'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2577',
    '', 'Calculated using linear free energy relationships (LFERs).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2577'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.7&times;10<sup>4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2171'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at293K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>4</sup>',
    '9500',
    '?',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2570'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='highTextrapol'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.3&times;10<sup>2</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury dichloride'),
    (SELECT id from literature WHERE bibtexkey='2576'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'morethanoneref',
    '', 'More than one reference is given as the source of this value.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='morethanoneref'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'mercury dibromide',
    'HgBr<sub>2</sub>',
    '',
    '7789-47-1',
    'NGYIMTKLQULBOO-UHFFFAOYSA-L',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>3</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='mercury dibromide'),
    (SELECT id from literature WHERE bibtexkey='2577'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.6&times;10<sup>2</sup>',
    '7400',
    'C',
    (SELECT id from species WHERE iupac='mercury dibromide'),
    (SELECT id from literature WHERE bibtexkey='2571'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>3</sup>',
    '7100',
    'Q',
    (SELECT id from species WHERE iupac='mercury dibromide'),
    (SELECT id from literature WHERE bibtexkey='2577'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2577'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.7&times;10<sup>4</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury dibromide'),
    (SELECT id from literature WHERE bibtexkey='2169'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2169wrongrefHgBr2',
    '2169,2255', 'Hedgecock et&nbsp;al. (2005) refer to Hedgecock and Pirrone (2004) as the source, but this value cannot be found there.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2169wrongrefHgBr2'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.2&times;10<sup>1</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury dibromide'),
    (SELECT id from literature WHERE bibtexkey='2576'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='morethanoneref'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'mercury diiodide',
    'HgI<sub>2</sub>',
    '',
    '7774-29-0',
    'YFDLHELOZYVNJE-UHFFFAOYSA-L',
    (SELECT id from subcategories WHERE subcatname='Other elements (B, Se, P, As, Hg)'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.7&times;10<sup>1</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='mercury diiodide'),
    (SELECT id from literature WHERE bibtexkey='2577'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>2</sup>',
    '6700',
    'Q',
    (SELECT id from species WHERE iupac='mercury diiodide'),
    (SELECT id from literature WHERE bibtexkey='2577'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2577'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9',
    '',
    '?',
    (SELECT id from species WHERE iupac='mercury diiodide'),
    (SELECT id from literature WHERE bibtexkey='2576'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='morethanoneref'));
INSERT INTO categories (catname) VALUES (
    'Hydrocarbons (C, H)');
INSERT INTO subcategories (subcatname, cat_id) VALUES (
    'Alkanes',
    (SELECT id from categories WHERE catname='Hydrocarbons (C, H)'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'methane',
    'CH<sub>4</sub>',
    '',
    '74-82-8',
    'VNWKTOKETHGBQD-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Alkanes'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1100',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='seawater'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '1100',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='seawater'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1900',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2828'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3673'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3861'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3861',
    '3861,491', 'Clever and Young (1987) recommend the data from Rettich et&nbsp;al. (1981).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3861'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='479'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1700',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1500',
    'L',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3431'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3695'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1800',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3690'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '2400',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3126'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1400',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2013'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2013-74-82-8-ABC',
    '2013', 'The data from Reichl (1995) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;133.87728 +6629.97157/T +17.62624&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2013-74-82-8-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3569'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3569-74-82-8-ABC',
    '3569', 'The data from Scharlin and Battino (1995) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;206.41168 +10058.77208/T +28.34417&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3569-74-82-8-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1800',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3734'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2557'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='491'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3601'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='highTextrapol'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1700',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3785'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3785-74-82-8-ABC',
    '3785', 'The data from Shoor et&nbsp;al. (1969) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;201.05778 +9920.37989/T +27.48020&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3785-74-82-8-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3924'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3924_also_3923',
    '3923', 'The same value was also published in McAuliffe (1963).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3924_also_3923'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3799'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3923'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3923_also_3924',
    '3924', 'The same value was also published in McAuliffe (1966).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3923_also_3924'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1600',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3783'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3783-74-82-8-ABC',
    '3783', 'The data from Morrison and Billett (1952) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;195.92072 +9624.37184/T +26.74976&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3783-74-82-8-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1700',
    'M',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2834'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2834-74-82-8-ABC',
    '2834', 'The data from Winkler (1901) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;203.15902 +9951.75251/T +27.82679&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2834-74-82-8-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_V'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3088'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='1145'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='642'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1600',
    'V',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.2&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='713'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'R',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='714'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.6&times;10<sup>&#8722;6</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='563'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='529'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.4&times;10<sup>&#8722;3</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;7</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3406'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2521'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '2300',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3731'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3731_RBFN',
    '3731,2440', 'Yao et&nbsp;al. (2002) compared two QSPR methods and found that radial basis function networks (RBFNs) are better than multiple linear regression. In their paper, they provide neither a definition nor the unit of their Henry''s law constants. Comparing the values with those that they cite from Yaws (1999), it is assumed that they use the variant H<sub>v</sub><sup>px</sup> and the unit atm.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3731_RBFN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2543'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2543',
    '2543', 'English and Carroll (2001) provide several calculations. Here, the preferred value with explicit inclusion of hydrogen bonding parameters from a neural network is shown.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2543-T',
    '', 'Value from the training dataset.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543-T'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.6&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3764'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='984'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='1162'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '1162PCR',
    '1162', 'Calculated with a principal component analysis (PCA); see Suzuki et&nbsp;al. (1992) for details.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1162PCR'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='1145'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '1700',
    '?',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='2440'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '1600',
    '?',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '1700',
    '?',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-74-82-8-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;185.72813 +9197.97387/T +25.21142&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-74-82-8-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='789'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='methane'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'ethane',
    'C<sub>2</sub>H<sub>6</sub>',
    '',
    '74-84-0',
    'OTMSDBZUPAUEDD-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Alkanes'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2105'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2105'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2300',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3673'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2300',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2300',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3855'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='479'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '2400',
    'L',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2013'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2013-74-84-0-ABC',
    '2013', 'The data from Reichl (1995) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;109.51433 +6313.03876/T +13.60483&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2013-74-84-0-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '2700',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3734'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='491'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '2700',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3601'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='highTextrapol'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3924_also_3923'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '2400',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3799'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3923'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3923_also_3924'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;5</sup>',
    '2100',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3783'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3783-74-84-0-ABC',
    '3783', 'The data from Morrison and Billett (1952) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;215.51394 +10861.98666/T +29.50128&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3783-74-84-0-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '2400',
    'M',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2834'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2834-74-84-0-ABC',
    '2834', 'The data from Winkler (1901) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;277.60377 +13887.90452/T +38.63046&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2834-74-84-0-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_V'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3088'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='642'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;5</sup>',
    '2000',
    'V',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;4</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='713'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.0&times;10<sup>&#8722;5</sup>',
    '',
    'T',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='766'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='yaws2003handbook'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'from3579',
    '3579', 'Value given here as quoted by Gharagheizi et&nbsp;al. (2010).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3579'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='529'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;3</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3409-LFER',
    '', 'Calculated using linear free energy relationships (LFERs).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-LFER'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3409-SPARC',
    '', 'Calculated using SPARC Performs Automated Reasoning in Chemistry (SPARC).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3409-COSMO',
    '', 'Calculated using COSMOtherm.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-COSMO'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3689'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'unknownT',
    '', 'Temperature is not specified.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='unknownT'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.3&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3406'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2546-T',
    '', 'Value from the training dataset.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-T'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2546-GROMHE',
    '', 'Calculated using the GROMHE model.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-GROMHE'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2546-SPARC',
    '', 'Calculated using the SPARC approach.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2546-HENRYWIN',
    '', 'Calculated using the HENRYWIN method.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-HENRYWIN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3579'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3579-NN-GC',
    '', 'Calculated using a combination of a group contribution method and neural networks.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3579-NN-GC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2521'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2530'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2530'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '2600',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.6&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3512'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3512',
    '3512', 'Modarresi et&nbsp;al. (2005) use different descriptors for the QSPR models. They conclude that their "COSA" method and the artificial neural network (ANN) are best. However, as COSA is not ideal for hydrocarbons with low solubility, only results obtained with ANN are shown here.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3512'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2535'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2535',
    '2535', 'Yaffe et&nbsp;al. (2003) present QSPR results calculated with the fuzzy ARTMAP (FAM) and with the back-propagation (BK-Pr) method. They conclude that FAM is better. Only the FAM results are shown here.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2535-Tr',
    '', 'Value from the training set.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535-Tr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3731'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3731_RBFN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543-T'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3764'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='1162'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1162PCR'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2482'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2550'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '2500',
    '?',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='2440'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '2300',
    '?',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '2400',
    '?',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='379'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref0379-74-84-0-ABC',
    '379', 'The data from Dean and Lange (1999) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;249.13770 +12672.58357/T +34.34947&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref0379-74-84-0-ABC'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='379'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='789'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='ethane'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'propane',
    'C<sub>3</sub>H<sub>8</sub>',
    '',
    '74-98-6',
    'ATUOYWHBWRKTHZ-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Alkanes'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2800',
    'L',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3673'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2800',
    'L',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3860'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='479'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '2700',
    'M',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3129'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2700',
    'M',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2013'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2013-74-98-6-ABC',
    '2013', 'The data from Reichl (1995) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;275.67877 +14048.75446/T +38.16041&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2013-74-98-6-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.7&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '3000',
    'M',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3734'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3924_also_3923'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '2700',
    'M',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3799'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3923'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3923_also_3924'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2600',
    'M',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3783'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3783-74-98-6-ABC',
    '3783', 'The data from Morrison and Billett (1952) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;257.69118 +13189.22089/T +35.51019&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3783-74-98-6-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_V'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3088'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='642'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2600',
    'V',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '2700',
    'V',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2550'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='yaws2003handbook'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3579'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='529'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.7&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-LFER'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.3&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.9&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-COSMO'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.1&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3406'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-T'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-GROMHE'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-HENRYWIN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.4&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3579'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3579-NN-GC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2521'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2530'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2530'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '2900',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.7&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3512'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3512'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535-Tr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3731'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3731_RBFN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543-T'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3764'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='1162'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1162PCR'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2482'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2550'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '2800',
    '?',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='2440'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '4500',
    '?',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='789'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='propane'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'butane',
    'C<sub>4</sub>H<sub>10</sub>',
    '',
    '106-97-8',
    'IJDNQMDRQITEOD-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Alkanes'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3500-106-97-8-consicheck',
    '3500', 'The H298 and A, B, C data listed in Table&nbsp;5-4 of Burkholder et&nbsp;al. (2019) are inconsistent, with 14&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-106-97-8-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3245-106-97-8-consicheck',
    '3245', 'The H298 and A, B, C data listed in Table&nbsp;5-4 of Burkholder et&nbsp;al. (2015) are inconsistent, with 14&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3245-106-97-8-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2626-106-97-8-consicheck',
    '2626', 'The H298 and A, B, C data listed in Table&nbsp;5.4 of Sander et&nbsp;al. (2011) are inconsistent, with 14&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2626-106-97-8-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '1945-106-97-8-consicheck',
    '1945', 'The H298 and A, B, C data listed in Table&nbsp;5.4 of Sander et&nbsp;al. (2006) are inconsistent, with 14&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1945-106-97-8-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3673'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3000',
    'L',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3860'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;5</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='479'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3100',
    'L',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '2300',
    'M',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3124'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.0&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at310K'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3200',
    'M',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3734'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3924_also_3923'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '3200',
    'M',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3799'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3923'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3923_also_3924'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '2900',
    'M',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3783'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref3783-106-97-8-ABC',
    '3783', 'The data from Morrison and Billett (1952) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;257.40529 +13425.82235/T +35.27658&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref3783-106-97-8-ABC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_V'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3088'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2611'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='635'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.6&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2439'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='642'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '2900',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3100',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2375'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2550'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.8&times;10<sup>&#8722;5</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='713'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='yaws2003handbook'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'from3682',
    '3682', 'Value given here as quoted by Dupeux et&nbsp;al. (2022).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3682'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='yaws2003handbook'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3579'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'C',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='529'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3682'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3682-COSMO',
    '', 'Calculated using the COSMO-RS method.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3682-COSMO'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-LFER'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.8&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-COSMO'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3689'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='unknownT'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.6&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3406'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-T'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-GROMHE'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-HENRYWIN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.4&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3579'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3579-NN-GC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2521'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '3300',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.5&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3512'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3512'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535-Tr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.6&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3731'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3731_RBFN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2543-V',
    '', 'Value from the validation dataset.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543-V'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3764'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='1162'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1162PCR'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2482'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2550'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '3300',
    '?',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='2440'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '3000',
    '?',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='789'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='butane'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    '2-methylpropane',
    'HC(CH<sub>3</sub>)<sub>3</sub>',
    'isobutane',
    '75-28-5',
    'NNPPMTNAJDCUHE-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Alkanes'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;6</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3500'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3500-75-28-5-consicheck',
    '3500', 'The H298 and A, B, C data listed in Table&nbsp;5-4 of Burkholder et&nbsp;al. (2019) are inconsistent, with 6&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3500-75-28-5-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;6</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3245'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3245-75-28-5-consicheck',
    '3245', 'The H298 and A, B, C data listed in Table&nbsp;5-4 of Burkholder et&nbsp;al. (2015) are inconsistent, with 6&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3245-75-28-5-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;6</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2626'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2626-75-28-5-consicheck',
    '2626', 'The H298 and A, B, C data listed in Table&nbsp;5.4 of Sander et&nbsp;al. (2011) are inconsistent, with 6&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2626-75-28-5-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;6</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='1945'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '1945-75-28-5-consicheck',
    '1945', 'The H298 and A, B, C data listed in Table&nbsp;5.4 of Sander et&nbsp;al. (2006) are inconsistent, with 6&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1945-75-28-5-consicheck'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '',
    'L',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='1943'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '1943-75-28-5',
    '1943,3860', 'Fogg and Sangster (2003) cite an incorrect fitting function from Hayduk (1986).');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1943-75-28-5'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.2&times;10<sup>&#8722;6</sup>',
    '2900',
    'L',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3673'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;6</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3860'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3860-75-28-5',
    '3860,3799', 'The fitting function and the data in the table on page 34 of Hayduk (1986) are inconsistent by a factor of about 3. A comparison with the original measurements by Wetlaufer et&nbsp;al. (1964) shows that the data in the table are correct. Refitting the data suggests that the third fitting parameter should be 52.4651, not 53.4651.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3860-75-28-5'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='479'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.0&times;10<sup>&#8722;6</sup>',
    '2700',
    'L',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='190'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;4</sup>',
    '5100',
    'M',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3117'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.5&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3924_also_3923'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;6</sup>',
    '2700',
    'M',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3799'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.5&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3923'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3923_also_3924'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_V'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3088'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2611'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='635'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='642'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.7&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2550'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.5&times;10<sup>&#8722;6</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='yaws2003handbook'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3579'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3728'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3728'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-LFER'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.3&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-COSMO'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.1&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3406'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-T'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-GROMHE'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.2&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-HENRYWIN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3579'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3579-NN-GC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.6&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2521'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2530'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2530'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '3300',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.7&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3512'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3512'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535-Tr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535-Tr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3731'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3731_RBFN'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3731_test',
    '', 'Value from the test set.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3731_test'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543-T'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3764'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='1162'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1162PCR'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2482'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.1&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2550'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '2900',
    '?',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.6&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2440'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.1&times;10<sup>&#8722;6</sup>',
    '2700',
    '?',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='3811'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.5&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='789'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.0&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='2-methylpropane'),
    (SELECT id from literature WHERE bibtexkey='2226'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    'pentane',
    'C<sub>5</sub>H<sub>12</sub>',
    '',
    '109-66-0',
    'OFBQJSOFQDEBGM-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Alkanes'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '3900',
    'L',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3518'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='moreTdep'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '3518-109-66-0-consicheck',
    '3518', 'Values at 298&nbsp;K in Tables&nbsp;C2 and C5 of Brockbank (2013) are inconsistent, with 15&nbsp;% difference.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3518-109-66-0-consicheck'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.7&times;10<sup>&#8722;6</sup>',
    '3500',
    'L',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3673'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.0&times;10<sup>&#8722;6</sup>',
    '3400',
    'L',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='1498'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.0&times;10<sup>&#8722;6</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='479'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.7&times;10<sup>&#8722;6</sup>',
    '3400',
    'M',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2910'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    'ref2910-109-66-0-ABC',
    '2910', 'The data from Jou and Mather (2000) were fitted to the three-parameter equation: H<sub>s</sub><sup>cp</sup>= exp( &#8722;400.38105 +20169.61328/T +56.35286&nbsp;ln(T))&nbsp;mol&nbsp;m<sup>&#8722;3</sup>&nbsp;Pa<sup>&#8722;1</sup>, with T in K.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='ref2910-109-66-0-ABC'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2910',
    '2910', 'The paper by Jou and Mather (2000) also contains high-temperature data. However, only data up to 373.2&nbsp;K were used here to calculate the temperature dependence.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2910'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.2&times;10<sup>&#8722;6</sup>',
    '3600',
    'M',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2909'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'M',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2218'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2611'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='635'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2478'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2388'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='642'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3924_also_3923'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3923'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3923_also_3924'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '3000',
    'T',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2224'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='yaws2003handbook'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3682'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='yaws2003handbook'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3579'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.3&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3682'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3682-COSMO'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.9&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3730'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.4&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_val'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-LFER'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.3&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-COSMO'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '4.7&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3406'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-T'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-GROMHE'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.2&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-HENRYWIN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.3&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3579'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3579-NN-GC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2521'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.6&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2530'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2530'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '3600',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.1&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3512'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3512'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535-Tr'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3731'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3731_RBFN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.0&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543-T'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3764'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='1162'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='1162PCR'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '9.9&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2482'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_exp'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '',
    '4200',
    '?',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='1917'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='2440'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.1&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='3614'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='789'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.0&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='pentane'),
    (SELECT id from literature WHERE bibtexkey='1500'));
INSERT INTO species (iupac, formula, trivial, casrn, inchikey, subcat_id) VALUES (
    '2-methylbutane',
    'C<sub>5</sub>H<sub>12</sub>',
    'isopentane',
    '78-78-4',
    'QWTDNUCVQCZILF-UHFFFAOYSA-N',
    (SELECT id from subcategories WHERE subcatname='Alkanes'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.7&times;10<sup>&#8722;6</sup>',
    '3000',
    'L',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3673'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'L',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='479'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3586_V'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.0&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3088'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2611'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.1&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='635'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2478'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='1910'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.6&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3924'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3924_also_3923'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.6&times;10<sup>&#8722;6</sup>',
    '',
    'V',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3923'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3923_also_3924'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'X',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='yaws2003handbook'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='from3579'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.7&times;10<sup>&#8722;4</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3586'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.3&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-LFER'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.0&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '2.0&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3409'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='at288K'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3409-COSMO'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3406'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2546-V',
    '', 'Value from the validation dataset.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-V'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-GROMHE'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-SPARC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2546'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2546-HENRYWIN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.5&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3579'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3579-NN-GC'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '6.4&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2521'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2530'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2530'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '3.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3512'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3512'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '5.8&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2535'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535'));
INSERT INTO notes (notelabel, noterefs, notetext) VALUES (
    '2535-Ts',
    '', 'Value from the test set.');
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2535-Ts'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3731'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='3731_RBFN'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.9&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='2543-T'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '1.5&times;10<sup>&#8722;5</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='3764'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '8.4&times;10<sup>&#8722;6</sup>',
    '',
    'Q',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='984'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='2440'));
INSERT INTO henry_notes (henry_id, notes_id) VALUES (
    (SELECT CURRVAL('henry_id_seq')),
    (SELECT id from notes WHERE notelabel='whichref'));
INSERT INTO henry(Hominus, mindHR, htype, species_id, literature_id) VALUES (
    '7.2&times;10<sup>&#8722;6</sup>',
    '',
    '?',
    (SELECT id from species WHERE iupac='2-methylbutane'),
    (SELECT id from literature WHERE bibtexkey='789'));
INSERT INTO henry_notes (henry